/*
 *   Copyright (c) 1999-2004 eVelopers Corporation. All rights reserved.
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Lesser General Public License for more details.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA.
 */
package com.evelopers.unimod.core.stateworks;



/**
 * <p>
 * Class <code>ControlledObjectHandler</code> is a container for Controlled Object
 * implementation. During design phase it contain only reference to real
 * implementation. During runtime getObject method instantiates implementation
 * class and return it. Such approach let you do not bother about controlled
 * object implementation during design.
 * </p>
 * <p>
 * Can have only incoming associations from state machine. On state chart
 * diagram of state machine <i>A</i> controlled object is referenced by
 * supplierRole of association between <i>A</i> and controlled object.
 * </p>
 * <p>
 * Underlining controlled object implementation must implement all input
 * and output actions according with state machine structure. If some
 * transition has label [o10.x4]/o10.z1, so controlled object implementation
 * with supplier role o10 must implement input action x4 and output action z1
 * with a help of methods
 * <pre>
 * public boolean x4(StateMachineContext context) throws SomeException {...}
 * public void z1(StateMachineContext context) throws SomeException {...}
 * </pre>
 * </p>
 * @author Vadim Gurov
 * @version $Revision: 1$
 */
public class ControlledObjectHandler extends ClassElement {

    public static final String CONTROLLED_OBJECT_STEREOTYPE = "controlledobject";

    /** @link dependency 
     * @label creates in runtime*/
    /*# ControlledObject lnkControlledObject; */

    /**
	 * Creates controlled object handler with given name and implementation class.
     * Sets stereotype to {@link ControlledObjectHandler#CONTROLLED_OBJECT_STEREOTYPE}.
	 *
	 * @param objectName object name (e.g. o1 or o10)
	 * @param implementationClassName
	 * 		  implementation object class (e.g. "com.evelopers.co.Connection")
	 */
	protected ControlledObjectHandler(String objectName, String implementationClassName) {
	    super(objectName, CONTROLLED_OBJECT_STEREOTYPE);
        setImplName(implementationClassName);
	}

    /**
     * Prohibits to add any outgoing associations
     * @throws UnsupportedOperationException if called
     */
    protected void addOutgoingAssociation(Association association) {
        throw new UnsupportedOperationException("ControlledObjectHandler can not has outgoinf associations");
    }

    /**
     * Prohibits to add incoming association from sources other then state machine.
     * @throws IllegalArgumentException if association source is not a state machine
     */
    protected void addIncomingAssociation(Association association) {
        if (association == null || !(association.getSource() instanceof StateMachine) || association.getTarget() != this) {
            throw new IllegalArgumentException("Incorrect incoming association");
        }
        
        super.addIncomingAssociation(association);
    }

    /**
     * Just publishes protected method of base class.
     */
    public void setImplName(String implName) {
        super.setImplName(implName);
    }
    
	public boolean equals(Object o) {
		if (!(o instanceof ControlledObjectHandler)) {
			return false;
		}

		return super.equals(o);
	}
}