/*
 *   Copyright (c) 1999-2004 eVelopers Corporation. All rights reserved.
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Lesser General Public License for more details.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA.
 */
package com.evelopers.unimod.core.stateworks;

import com.evelopers.unimod.core.ModelElement;
import com.evelopers.unimod.resources.Messages;

import java.util.ArrayList;
import java.util.List;

/**
 * Abstract base implementaion for all classes in model such
 * as ControlledObjectHandler, EventProviderHandler and StateMachine.
 * For utility purposes implements delegate to ImplementationHandler.
 * ControlledObjectHandler needs ImplementationHandler to handle ControlledObject implementation,
 * EventProviderHandler handles EventProvider, StateMachine handles ConfigStore.
 * 
 * @author vgurov
 * @version $Revision: 1$
 */
public abstract class ClassElement extends ModelElement {

	public static final String IMPL_NAME_PROPERTY = "IMPL_NAME_PROPERTY";
    public static final String OUTGOING_ASSOCIATIONS_PROPERTY = "OUTGOING_ASSOCIATIONS_PROPERTY";
    public static final String INCOMING_ASSOCIATIONS_PROPERTY = "INCOMING_ASSOCIATIONS_PROPERTY";

	/**
	 * List of incoming associations
	 * @associates <{Association}>
	 * @link aggregation
     * @undirected
     * @clientCardinality source
     * @supplierCardinality outgoing
     * @supplierRole 0..*
     * @clientRole 1
	 */
	protected List outgoingAssociations = new ArrayList();
	
	/**
     * List of outgoing associations that have this this class element 
	 * @associates <{Association}>
	 * @link aggregation
     * @undirected
     * @clientCardinality target
     * @supplierCardinality incoming
     * @clientRole 1
     * @supplierRole 0..*
	 */
	protected List incomingAssociations = new ArrayList();

    protected String implName;

    ClassElement(String name, String stereotype) {
        super(name, stereotype);
    }

    public String getImplName() {
        return implName;
    }

    protected void setImplName(String implName) {
        String oldName = getImplName();

        if ((oldName != null && !oldName.equals(implName)) || implName != null) {
            this.implName = implName;
            firePropertyChange(IMPL_NAME_PROPERTY, oldName, implName);
        }
    }

    protected void addOutgoingAssociation(Association association) {
        if (association == null) {
            throw new IllegalArgumentException(Messages.getMessages().getMessage(Messages.ASSOCIATION_NULL));
        }

        if (outgoingAssociations.add(association)) {
            firePropertyChange(OUTGOING_ASSOCIATIONS_PROPERTY, null, association);
        }
            
    }

    protected void removeOutgoingAssociation(Association association) {
        if (outgoingAssociations.contains(association)) {
            outgoingAssociations.remove(association);

            firePropertyChange(OUTGOING_ASSOCIATIONS_PROPERTY, association, null);
        }
    }

    protected void addIncomingAssociation(Association association) {
        if (association == null) {
            throw new IllegalArgumentException(Messages.getMessages().getMessage(Messages.ASSOCIATION_NULL));
        }

        if (incomingAssociations.add(association)) {
            firePropertyChange(INCOMING_ASSOCIATIONS_PROPERTY, null, association);
        }
    }

    protected void removeIncomingAssociation(Association association) {
        if (incomingAssociations.contains(association)) {
            incomingAssociations.remove(association);

            firePropertyChange(INCOMING_ASSOCIATIONS_PROPERTY, association, null);
        }
    }
    
    public List getOutgoingAssociations() {
        return outgoingAssociations;
    }
    
    public List getIncomingAssociations() {
        return incomingAssociations;
    }
    
	public int hashCode() {
		int h = 0;
		h = name == null ? h : name.hashCode();
		h = getImplName() == null ? h : h ^ getImplName().hashCode();
		
		return h;
	}

	public boolean equals(Object o) {
		if (!(o instanceof ClassElement)) {
			return false;
		}

		return name != null && name.equals(((ClassElement)o).getName()) &&
			getImplName() != null && name.equals(((ClassElement)o).getImplName());
	}
	
	/**
	 * Returns class identifier
	 * 
	 * @return class identifier
	 */
	public String toString() {
		return getName() + ": " + getImplName();
	}
}
