/*
 *   Copyright (c) 1999-2004 eVelopers Corporation. All rights reserved.
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Lesser General Public License for more details.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA.
 */
package com.evelopers.unimod.core.stateworks;

import com.evelopers.unimod.core.ModelElement;

/**
 * An association between {@link ClassElement}s 
 *
 * @author Maxim Mazin
 * @author Vadim Gurov
 * @version $Revision: 1$
 */
public class Association extends ModelElement {

    public static final String SUPPLIER_ROLE_PROPERTY = "SUPPLIER_ROLE_PROPERTY";
    public static final String CLIENT_ROLE_PROPERTY = "CLIENT_ROLE_PROPERTY";
    
    /**
     * Source class element
     */
    private ClassElement source;
    
    /**
     * Supplier role is the name that is by source to identify target
     */
    private String supplierRole;
    
    /**
     * Target class element
     */
    private ClassElement target;

    /**
     * Client role is the name that is by target to identify source
     */
    private String clientRole;
    
    /**
     * <p>
     * Creates new association using controlled object name as
     * supplier role. Adds this association to list of associations
     * of given state machine and to list of associations of given
     * controlled object.
     * </p>
     * <p>
     * <strong>Note:</strong> <code>sm</code>, <code>co</code> and
     * name of the <code>co</code>
     * shouldn't be <code>null</code>. If implementation
     * needs to pass <code>null</code> to this constructor then
     * methods {@link #hashCode} and {@link #equals} <strong>must</strong>
     * be overriden.
     * </p>
     * @param sm client of the association
     * @param co supplier of the association
     */
    protected Association(ClassElement source, ClassElement target) {
        this(source, null, target, null);
    }

    /**
     * <p>
     * Creates new association. Adds this association to list of associations
     * of given state machine and to list of associations of given
     * controlled object.
     * </p>
     * <p>
     * <strong>Note:</strong> <code>sm</code>, <code>co</code> and
     * <code>supplierRole</code>
     * shouldn't be <code>null</code>. If implementation
     * needs to pass <code>null</code> to this constructor then
     * methods {@link #hashCode} and {@link #equals} <strong>must</strong>
     * be overriden.
     * </p>
     * @param sm client of the association
     * @param co supplier of the association
     * @param supplierRole name of controlled object that is used
     * by state machine to identify controlled object.
     */
    protected Association(ClassElement source, String supplierRole, ClassElement target, String clientRole) {
        this.clientRole = clientRole == null ? (source == null ? null : source.getName()) : clientRole;
        this.supplierRole = supplierRole == null ? (target == null ? null : target.getName()) : supplierRole;
        reconnect(source, target);
    }

    /**
     * Returns state machine that is a client in this association.
     * @return client of this association
     */
    public ClassElement getSource() {
        return source;
    }

    /**
     * Returns controlled object that is a supplier in this association.
     * @return supplier of this association
     */
    public ClassElement getTarget() {
        return target;
    }

    /**
     * Supplier role is the name that is used be state machine
     * to identify the controlled object.
     * @return supplier's role of controlled object
     */
    public String getSupplierRole() {
        return supplierRole;
    }

    public String getClientRole() {
        return clientRole;
    }
    
	/**
	 * Reconnects transition to new source and target states. 
    	 * <li>Detachs from source and sets new source state
    	 * <li>Detachs from target and sets new target state
    	 * <li>Attaches to new source state
    	 * <li>Attaches to new target state
	 *  
	 * @param newSourceState new source state
	 * @param newTargetState
	 */
	public void reconnect(ClassElement newSource, ClassElement newTarget) {

		setSource(newSource);
		setTarget(newTarget);
		
		attachSource();
		attachTarget();
	}
	
	/**
	 * Sets new source class. If old source class was not null - removes this association
	 * from outgoing associations of source class.
	 *  
	 * @param newSource new source class. If null, then association will only removes itself
	 * from old source class outgoing associations list.
	 */
	public void setSource(ClassElement newSource) {
		detachSource();		
		source = newSource;
	}

	public void attachSource() {
		if (source == null) {
			return;
		}

		source.addOutgoingAssociation(this);
	}

	public void detachSource() {
		if (source == null) {
			return;
		}

		source.removeOutgoingAssociation(this);
	}

	/**
	 * Sets new target state. If old target state was not null - removes this transition
	 * from incoming transitions of target state. 
     *
	 * @param newTargetState new target state. If null, then transition will only removes itlsef
	 * from target state incoming transitions set.
	 */
	public void setTarget(ClassElement newTarget) {
		detachTarget();		
		target = newTarget;
	}

	public void attachTarget() {
		if (target== null) {
			return;
		}

		target.addIncomingAssociation(this);
	}

	public void detachTarget() {
		if (target == null) {
			return;
		}

		target.removeIncomingAssociation(this);
	}

    /**
     * <p>
     * Sets <code>newSupplierRole</code> as a supplier role for this association.
     * </p>
     * <p>
     * <strong>Note:</strong> if this method is used by extending
     * classes then methods {@link #hashCode} and {@link #equals}
     * <strong>must</strong> be overriden.
     * </p>
     * @param newSupplierRole new supplier role for this association.
     */
    protected void setSupplierRole(String newSupplierRole) {
        String oldSupplierRole = this.supplierRole;
        if (oldSupplierRole != newSupplierRole) {
            this.supplierRole = newSupplierRole;
            firePropertyChange(SUPPLIER_ROLE_PROPERTY, oldSupplierRole, newSupplierRole);
        }
    }

    protected void setClientRole(String newClientRole) {
        String oldClientRole = this.clientRole;
        if (oldClientRole != newClientRole) {
            this.clientRole = newClientRole;
            firePropertyChange(CLIENT_ROLE_PROPERTY, oldClientRole, newClientRole);
        }
    }
    
    /**
     * This implementation considers ends of association and value
     * of supplier role to be immutable.
     * @return hash code based on hash codes of
     * ends this association and hash code of supplier role.
     */
    public int hashCode() {
        return source.hashCode() ^
                target.hashCode() ^
                (supplierRole == null ? 0 : supplierRole.hashCode()) ^
                (clientRole == null ? 0 : clientRole.hashCode());
    }

    /**
     * This implementation considers ends of association and value
     * of supplier's role to be immutable.
     * @param o an object to compare this association to
     * @return whether this association is equal to <code>o</code>
     */
    public boolean equals(Object o) {
        if (o instanceof Association) {
            Association that = (Association) o;
            return this.source.equals(that.source) &&
                    this.target.equals(that.target) &&
                    ((this.supplierRole == null && that.supplierRole == null) || this.supplierRole.equals(that.supplierRole)) &&
                    ((this.clientRole == null && that.clientRole == null) || this.clientRole.equals(that.clientRole));
        } else {
            return false;
        }
    }
}