/*
 *   Copyright (c) 1999-2004 eVelopers Corporation. All rights reserved.
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Lesser General Public License for more details.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA.
 */
package com.evelopers.unimod.core;

import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeSupport;
import java.io.*;

/**
 * Base abstract class for all model elements.
 * 
 * @author Vadim Gurov
 * @version $Revision: 1$
 */
abstract public class ModelElement implements Serializable {

	protected String name;
	public static final String NAME_PROPERTY = "NAME_PROPERTY";
	
	protected String stereotype;
	public static final String STEREOTYPE_PROPERTY = "STEREOTYPE_PROPERTY";
	
	transient protected PropertyChangeSupport listeners;

	/**
	 * Creates empty model element.
	 */
	protected ModelElement() {
		this("", null);
	}

	/**
	 * Creates model element with given name.
	 *
	 * @param name model element name
	 *
	 * @throws IllegalArgumentException is name is null
	 */
	protected  ModelElement(String name) {
		this(name, null);
	}

	/**
	 * Creates model element witg given name and stereotype.
	 *
	 * @param name model element name
	 * @param stereotype model element stereotype
	 */
	protected  ModelElement(String name, String stereotype) {
		this.name = name;
		
		this.stereotype = stereotype;
	}

	/**
	 * Returns name
	 *
	 * @return name
	 */
	public String getName() {
		return name;
	}

    /**
     * Sets name
     * @param name - new name
     */
    public void setName(String name) {
    	String oldName = this.name;
        this.name = name;
        
        firePropertyChange(NAME_PROPERTY, oldName, name);
    }

	/**
	 * Returns stereotype
	 *
	 * @return stereotype
	 */
	public String getStereotype() {
		return stereotype;
	}

    /**
     * Sets stereotype
     * @param stereotype new stereotype.
     */
    protected void setStereotype(String stereotype) {
    	String oldStereotype = this.stereotype; 
        this.stereotype = stereotype;

		firePropertyChange(STEREOTYPE_PROPERTY, oldStereotype, stereotype);
    }

	/**
	 * Returns model element hash code. 
	 *
	 * @return hash code
	 */
	public abstract int hashCode();

	/**
	 * Compare with another model element. 
	 *
	 * @param o another model element
	 *
	 * @return boolean
	 */
	public abstract boolean equals(Object o);

	/**
	 * Returns name and stereotype.
	 *
	 * @return string
	 */
	public String toString() {
		return (stereotype == null) ? name : (name + "[" + stereotype + "]");
	}
	
	/**
	 * Accepts given visitor.
	 * 
	 * @param v visitor to visit model element
	 * @throws VisitorException thrown by visitor methods
	 */
	public void accept(ModelVisitor v) throws VisitorException {
		v.visit(this);
	}
	
	/**
	 * Adds new property change listener 
	 * 
	 * @param l property change listener
	 */
	public void addPropertyChangeListener(PropertyChangeListener l){
	  initListeners();
		
	  listeners.addPropertyChangeListener(l);
	}
  
	/**
	 * Removes property change listener 
	 * 
	 * @param l property change listener
	 */
	public void removePropertyChangeListener(PropertyChangeListener l){
	  initListeners();
		  
	  listeners.removePropertyChangeListener(l);
	}
  
  	/**
  	 * Fires property change event 
  	 * 
  	 * @param prop property name
  	 * @param oldValue old property value
  	 * @param newValue new property value
  	 */
	protected void firePropertyChange(String prop, Object oldValue, Object newValue){
  	  initListeners();

	  listeners.firePropertyChange(prop, oldValue, newValue); 
	}
	
	protected void initListeners() {
		if (listeners == null) {
			listeners = new PropertyChangeSupport(this);	
		}

	}
}
