/*
 *   Copyright (c) 1999-2004 eVelopers Corporation. All rights reserved.
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Lesser General Public License for more details.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA.
 */
package com.evelopers.unimod.contract;

import java.util.Iterator;
import java.util.StringTokenizer;

import com.evelopers.unimod.core.stateworks.Action;
import com.evelopers.unimod.core.stateworks.Event;
import com.evelopers.unimod.core.stateworks.Model;
import com.evelopers.unimod.core.stateworks.State;
import com.evelopers.unimod.core.stateworks.StateMachine;
import com.evelopers.unimod.core.stateworks.Transition;

/**
 * Converts model elements into String and vice versa
 */
public class CoreContract {

    private CoreContract() {}
    
    public static CoreContract create() {
        return new CoreContract();
    }
    
    private static final String TRANSITON_DELIMETER = "#";
    
    public static Event decodeEvent(String e) {
        return new Event(e);
    }
    
    public static String encode(Event e) {
        if (e == null) {
            throw new NullPointerException("CoreContract.encode(Event e)");
        }
        
        return e.getName();
    }
    
    public static String encode(Transition t) {
        if (t == null) {
            throw new NullPointerException("CoreContract.encode(Transition t)");
        }
        
        return 
        	t.getSourceState().getName() + TRANSITON_DELIMETER +
        	t.getTargetState().getName() + TRANSITON_DELIMETER +
        	t.getEvent().getName() + TRANSITON_DELIMETER + 
        	t.getGuard().getExpr();
    }
    
    public static Transition decodeTransition(StateMachine sm, String s) {
        StringTokenizer st = new StringTokenizer(s, TRANSITON_DELIMETER);
        String sourceStateName = st.nextToken();
        String targetStateName = st.nextToken();
        String eventName = st.nextToken();
        String guardExpr = st.nextToken(); 
        
        for (Iterator i = sm.getAllTransition().iterator(); i.hasNext();) {
            Transition t = (Transition)i.next();
            
            if (sourceStateName.equals(t.getSourceState().getName()) &&
                targetStateName.equals(t.getTargetState().getName()) &&
                eventName.equals(t.getEvent().getName()) &&
                guardExpr.equals(t.getGuard().getExpr())) {
                return t;
            }
                
        }
        
        return null;
    }    
    
    public static String encode(State s) {
        if (s == null) {
            throw new NullPointerException("CoreContract.encode(State s)");
        }
        
        return s.getName();
    }
    
    public static State decodeState(StateMachine sm, String s) {
        return sm.findState(s);
    }
    
    public static String encode(StateMachine sm) {
        if (sm == null) {
            throw new NullPointerException("CoreContract.encode(StateMachine sm)");
        }
        
        return sm.getName();
    }
    
    public static StateMachine decodeStateMachine(Model model, String stateMachine) {
        return model.getStateMachine(stateMachine);
    }
    
    public static String encode(Action a) {
        if (a == null) {
            throw new NullPointerException("CoreContract.encode(Action a)");
        }
        
        return a.getIdentifier();
    }
    
    public static Action decodeAction(State s, String action) {
        return decodeAction(s.getOnEnterActions().iterator(), action);
    }
    
    public static Action decodeInputAction(Transition t, String action) {
        return decodeAction(t.getGuard().getInputActions().iterator(), action);
    }

    public static Action decodeOutputAction(Transition t, String action) {
        return decodeAction(t.getOutputActions().iterator(), action);
    }
    
    protected static Action decodeAction(Iterator i, String action) {
        while (i.hasNext()) {
            Action a = (Action) i.next();
            
            if (a.getIdentifier().equals(action)) {
                return a;
            }
        }
        
        return null;
    }
}
