/*
 *   Copyright (c) 1999-2004 eVelopers Corporation. All rights reserved.
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Lesser General Public License for more details.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA.
 */
package com.evelopers.unimod.compilation;

import com.evelopers.unimod.core.ModelElement;
import com.evelopers.unimod.core.stateworks.*;
import com.evelopers.unimod.parser.ActionCollector;
import com.evelopers.unimod.parser.ExprParser;
import com.evelopers.unimod.parser.InterpreterException;
import com.evelopers.unimod.parser.ParserException;
import com.evelopers.unimod.transform.TransformException;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.apache.commons.lang.builder.ToStringBuilder;

/**
 * <br>
 * Compile given state machine. This process includes:
 * </br>
 * <li>Check that every model element has name;</li>
 * <li>Check that ControlledObjectHandlers and EventProviderHandlers have implClass;</li>
 * <li>Check that every State has StateType</li>
 * <li>Translate to AST all guard conditions;</li>
 * <li>Translate all actions' identifiers to ControlledObjectHandler and actionName;</li>
 * 
 * @author vgurov
 * @version Revision: 1
 */
public class StateMachineCompiler {
    private OperationResolver operationResolver;
	private List compilationListeners = new ArrayList();

    public StateMachineCompiler(OperationResolver operationResolver) {
        this.operationResolver = operationResolver;
    }

	public boolean addCompilationListener(CompilationListener listener) {
		return compilationListeners.add(listener);
	}

	public boolean removeCompilationListener(CompilationListener listener) {
		return compilationListeners.remove(listener);
	}
	
	/**
	 * Compiles whole state machine WITHOUT included state machines
	 * 
	 * @param sm
	 */
	public void compile(StateMachine sm) {
		if (sm.getName() == null || sm.getName().equals("")) {
			fireNullName(sm, sm);
		}

		for (Iterator i = sm.getEventProviderHandlers().iterator(); i.hasNext(); ) {
			compile(sm, (EventProviderHandler)i.next());
		}
		
		for (Iterator i = sm.getControlledObjectHandlers().iterator(); i.hasNext(); ) {
			compile(sm, (ControlledObjectHandler)i.next());
		}
		
		for (Iterator i = sm.getAllStates().iterator(); i.hasNext(); ) {
			compile(sm, (State)i.next());
		}

		for (Iterator i = sm.getAllTransition().iterator(); i.hasNext(); ) {
			compile(sm, (Transition)i.next());
		}
	}

	/**
	 * Compiles whole state machine WITH included state machines
	 * 
	 * @param sm
	 */
	public void compileWithIncluded(StateMachine sm) {
	    compile(sm);
	    
	    for (Iterator i = sm.getSubmachines().iterator(); i.hasNext(); ) {
	        compile((StateMachine)i.next());
	    }
	}
	
	/**
	 * Compiles given controlled object
	 * 
	 * @param sm
	 * @param co
	 */
	public void compile(StateMachine sm, ControlledObjectHandler co) {
		if (co.getName() == null || co.getName().equals("")) {
			fireNullName(sm, co);
		}

		if (co.getImplName() == null  || co.getImplName().equals("")) {
			fireNullImplClass(sm, co);
		}
	}

	/**
	 * Compiles given event provider object
	 * 
	 * @param sm
	 * @param ep
	 */
	public void compile(StateMachine sm, EventProviderHandler ep) {
		if (ep.getName() == null || ep.getName().equals("")) {
			fireNullName(sm, ep);
		}

		if (ep.getImplName() == null  || ep.getImplName().equals("")) {
			fireNullImplClass(sm, ep);
		}
	}
	
	/**
	 * Compiles given state with it's output actions
	 * 
	 * @param sm
	 * @param s
	 */
	public void compile(StateMachine sm, State s) {
		if (s.getName() == null || s.getName().equals("")) {
			fireNullName(sm, s);
		}

		if (s.getType() == null) {
			fireNullStateType(sm, s);
		}

        List l = s.getOnEnterActions();
		if (l != null) {
			for (Iterator i = l.iterator(); i.hasNext(); ) {
				Action e = (Action)i.next();
				compile(sm, s, e);				
			}
		}
	}

	/**
	 * Compiles given transition with it's guard, event and output actions
	 * 
	 * @param sm
	 * @param t
	 */
	public void compile(StateMachine sm, Transition t) {
        compile(sm, t, t.getGuard());
		
		if (t.getEvent() != null) {
			compile(sm, t.getEvent());	
		}
		
		List l = t.getOutputActions();
		
		if (l != null) {
			for (Iterator i = l.iterator(); i.hasNext(); ) {
				Action e = (Action)i.next();
				compile(sm, t, e);				
			}
		}
	}
	
	/**
	 * Compiles given guard
	 * 
	 * @param sm
	 * @param c
	 */
	public void compile(StateMachine sm, Transition t, Guard c) {
		
	    // do not try to compile "else" guard condition
	    if (c.equals(Guard.ELSE)) {
		    return;
		}
	    
	    try {
			c.setAST(ExprParser.parse(c.getExpr(), sm, operationResolver));
			c.setInputActions(ActionCollector.collect(c.getAST()));
		} catch (ParserException e) {
			fireIncorrectExpression(sm, t, c, e);
		} catch (InterpreterException e) {
			fireIncorrectExpression(sm, t, c, new ParserException(e));
		}
	}

	public void compile(StateMachine sm, ModelElement owner, Action e) {
        if (!checkIdentifier(e.getIdentifier())) {
            fireIncorrectActionIdentifier(sm, owner, e);
        }

		String name = getObjectName(e.getIdentifier());
		ClassElement ce = sm.getClassElement(name);

		if (ce == null) {
			fireUnresolvedClassElement(sm, owner, e, name);
		} else {
			e.setObject(ce);
		}

		String actionName = getActionName(e.getIdentifier());
        e.setActionName(actionName);

		if (operationResolver != null && ce instanceof ControlledObjectHandler &&
            operationResolver.getOperationType((ControlledObjectHandler)ce, actionName) == null) {
			fireUnresolvedAction(sm, owner, e, (ControlledObjectHandler)ce, actionName);
		} 
		
		if (ce instanceof StateMachine) {
		    compile((StateMachine)ce);
		}
	}
	
    /**
	 * Compiles given event
	 * 
	 * @param sm
	 * @param e
	 */
	public void compile(StateMachine sm, Event e) {
		if (e.getName() == null) {
			fireNullName(sm, e);
		}
	}
	
	private void fireNullName(StateMachine sm, ModelElement me) {
		for (Iterator i = compilationListeners.iterator(); i.hasNext();) {
			CompilationListener cl = (CompilationListener)i.next();
			cl.nullName(sm, me);
		}
	}

	private void fireNullImplClass(StateMachine sm, ControlledObjectHandler co) {
		for (Iterator i = compilationListeners.iterator(); i.hasNext();) {
			CompilationListener cl = (CompilationListener)i.next();
			cl.nullImplClass(sm, co);
		}
	}

	private void fireNullImplClass(StateMachine sm, EventProviderHandler ep) {
		for (Iterator i = compilationListeners.iterator(); i.hasNext();) {
			CompilationListener cl = (CompilationListener)i.next();
			cl.nullImplClass(sm, ep);
		}
	}
	
	private void fireNullStateType(StateMachine sm, State s) {
		for (Iterator i = compilationListeners.iterator(); i.hasNext();) {
			CompilationListener cl = (CompilationListener)i.next();
			cl.nullStateType(sm, s);
		}
	}

	private void fireIncorrectActionIdentifier(StateMachine sm, ModelElement me, Action e) {
		for (Iterator i = compilationListeners.iterator(); i.hasNext();) {
			CompilationListener cl = (CompilationListener)i.next();
			cl.incorrectActionIdentifier(sm, me, e);
		}
	}

    private void fireUnresolvedClassElement(StateMachine sm, ModelElement owner, Action e, String coName) {
        for (Iterator i = compilationListeners.iterator(); i.hasNext();) {
            CompilationListener cl = (CompilationListener)i.next();
            cl.unresolvedClassElement(sm, owner, e, coName);
        }
    }

    private void fireUnresolvedAction(StateMachine sm, ModelElement owner, Action e, ControlledObjectHandler co, String actionName) {
        for (Iterator i = compilationListeners.iterator(); i.hasNext();) {
            CompilationListener cl = (CompilationListener)i.next();
            cl.unresolvedAction(sm, owner, e, co, actionName);
        }
    }

	private void fireIncorrectExpression(StateMachine sm, Transition t, Guard c, ParserException e) {
		for (Iterator i = compilationListeners.iterator(); i.hasNext();) {
			CompilationListener cl = (CompilationListener)i.next();
			cl.incorrectExpression(sm, t, c, e);
		}
	}
	
	/**
	 * Parses object name from identifier
	 *
	 * @param identifier action identifier line o1.z2
	 * @return object name (for o1.z2 will returns o1)
	 */
	public static String getObjectName(String identifier) {
		return identifier.substring(0, identifier.indexOf('.'));
	}

	/**
	 * Parses action name from identifier
	 *
	 * @param identifier action identifier line o1.z2
	 * @return action name (for o1.z2 will return z2)
	 */
	public static String getActionName(String identifier) {
		return identifier.substring(identifier.indexOf('.') + 1);
	}

	protected boolean checkIdentifier(String identifier) {
        return identifier != null && identifier.indexOf('.') != -1;
	}

	public static void compileModel(Model m) throws TransformException {
        StateMachineCompiler c = new StateMachineCompiler(null);
        DefaultCompilationListener cl = new DefaultCompilationListener();
        c.addCompilationListener(cl);
        
        for (Iterator i = m.getStateMachines().iterator(); i.hasNext(); ) {
            c.compile((StateMachine)i.next());
        }

        /* check compilation errors */
        String[] errors = cl.getErrors();

        if (errors.length > 0) {
            throw new TransformException("Model compilation error:\n" + ToStringBuilder.reflectionToString(errors));
        }
	}

}
