/*
 *   Copyright (c) 1999-2004 eVelopers Corporation. All rights reserved.
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Lesser General Public License for more details.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA.
 */
package com.evelopers.unimod.compilation;

import com.evelopers.unimod.core.ModelElement;
import com.evelopers.unimod.core.stateworks.*;
import com.evelopers.unimod.parser.ParserException;

import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.List;

/**
 * Simple implementation of {@link CompilationListener}.
 * Collects all errors into internal array which may be got with a help of 
 * method {@link #getErrors()} after compilation.
 * 
 * @author vgurov
 * @version $Revision: 1$
 */
public class DefaultCompilationListener implements CompilationListener {

    private static final String NULL_NAME =       			   "{0}: Model element of class [{1}] has empty name";
    private static final String NULL_IMPL_NAME =   			   "{0}: Controlled object handler [{1}] has empty implementation class name";
    private static final String NULL_STATE_TYPE =  			   "{0}: State [{1}] has null state type";
    private static final String INCORRECT_ACTION_IDENTIFIER =  "{0}: Action [{1} has incorrect identifier";
    private static final String INCORRECT_EXPRESSION =         "{0}: Expression [{2}] on transition [{1}] is incorrect: {3}";
    private static final String UNRESOLVED_CLASS_ELEMENT =     "{0}: Model element [{1}] has action [{2}] with reference to unresolved class element [{3}]";
    private static final String UNRESOLVED_ACTION =            "{0}: Model element [{1}] has action [{2}] with reference to unresolved method [{3}] of controlled object";
        
    private List errors = new ArrayList();
    
    public String[] getErrors() {
        return (String[])errors.toArray(new String[errors.size()]);
    }
    
    protected String format(String pattern, Object s1, Object s2, Object s3, Object s4) {
        return MessageFormat.format(pattern, new Object[]{s1, s2, s3, s4});
    }

    protected String format(String pattern, Object s1, Object s2, Object s3) {
        return MessageFormat.format(pattern, new Object[]{s1, s2, s3});
    }

    protected String format(String pattern, Object s1, Object s2) {
        return MessageFormat.format(pattern, new Object[]{s1, s2});
    }
    
    public void nullName(StateMachine sm, ModelElement me) {
        errors.add(format(NULL_NAME, sm, me.getClass().getName()));
    }

    public void nullImplClass(StateMachine sm, ControlledObjectHandler co) {
        errors.add(format(NULL_IMPL_NAME, sm, co));
    }
    
    public void nullImplClass(StateMachine sm, EventProviderHandler ep) {
        errors.add(format(NULL_IMPL_NAME, sm, ep));
    }

    public void nullStateType(StateMachine sm, State s) {
        errors.add(format(NULL_STATE_TYPE, sm, s));
    }

    public void incorrectActionIdentifier(StateMachine sm, ModelElement me,
            Action e) {
        errors.add(format(INCORRECT_ACTION_IDENTIFIER, sm, me, e));
    }

    public void incorrectExpression(StateMachine sm, Transition t, Guard c,
            ParserException e) {
        errors.add(format(INCORRECT_EXPRESSION, sm, t, c, e.getMessage()));
    }

    public void unresolvedClassElement(StateMachine sm, ModelElement owner,
            Action e, String name) {
        errors.add(format(UNRESOLVED_CLASS_ELEMENT, sm, owner, e.getIdentifier(), name));
    }
    
    public void unresolvedAction(StateMachine sm, ModelElement owner, Action e,
            ControlledObjectHandler co, String actionName) {
        errors.add(format(UNRESOLVED_ACTION, sm, owner, e.getIdentifier(), actionName));
    }
}
