/*
 * Copyright (c) 1999-2005 eVelopers Corporation. All rights reserved.
 *
 * This is open source software; you can use, redistribute and/or modify 
 * it under the terms of the Open Software Licence v 2.1 as published by the Open 
 * Source Initiative.
 *
 * You should have received a copy of the Open Software Licence along with this
 * application; if not, contact the Open Source Initiative (http://opensource.org).
 */
package com.evelopers.unimod.analysis.executors;

import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import com.evelopers.unimod.analysis.TypeInfo;
import com.evelopers.unimod.analysis.TypeInfoProvider;

/**
 * @uml.dependency   supplier="com.evelopers.unimod.analysis.executors.Substitution" stereotypes="Basic::Create"
 * @uml.dependency   supplier="com.evelopers.unimod.analysis.executors.PredicateExecutorBuilder" stereotypes="Basic::Responsibility"
 * @uml.dependency   supplier="com.evelopers.unimod.analysis.TypeInfoProvider"
 */
public class SubstitutionBuilder {
    private TypeInfoProvider typeInfoProvider;

    private List booleanVars = new LinkedList();
    private List predicateVars = new LinkedList();
    private Map builders = new HashMap(); 
    
    public SubstitutionBuilder(TypeInfoProvider typeInfoProvider) {
        this.typeInfoProvider = typeInfoProvider;
    }
    
    public void addBooleanVar(String variableName) {
        if (typeInfoProvider.getTypeInfo(variableName).getType() == TypeInfo.BOOLEAN.getType()) {
            if (!booleanVars.contains(variableName)) {
                booleanVars.add(variableName);
            }
        } else {
            throw new IllegalArgumentException("Variable [" + variableName + "] is not boolean");
        }
    }
    
    public void addPredicate(int predicateType, String variableName, String constantValue) {
        if (! builders.containsKey(variableName)) {
            predicateVars.add(variableName);
        }
        PredicateExecutorBuilder builder = getBuilder(variableName);            
        builder.addPredicate(predicateType, 
                typeInfoProvider.parseConstant(variableName, constantValue));
    }

    public Substitution buildSubstitution() {
        int bits = 0;
        Map positions = new HashMap();
        List predicatePositions = new LinkedList();

        for (Iterator i = booleanVars.iterator(); i.hasNext();) {
            String variableName = (String) i.next();
            
            Position position = new Position();
            position.position = (int) bits;

            positions.put(variableName, position);
            bits ++;
        }
        
        Map predicateExecutors = getPredicateExecutors();
        for (Iterator i = predicateVars.iterator(); i.hasNext();) {
            String variableName = (String) i.next();
            PredicateExecutor executor = (PredicateExecutor) predicateExecutors.get(variableName);            
            
            PredicatePosition position = new PredicatePosition();
            position.position = (int) bits;
            position.predicateExecutor = executor;
            position.cardinality = executor.getCardinality();
            position.bits = ceilLog2(position.cardinality);                

            positions.put(variableName, position);
            predicatePositions.add(position);
            bits += position.bits;
        }   
        
        return new Substitution(bits, positions, predicatePositions);
    }
    
    private Map getPredicateExecutors() {
        Map predicateExecutors = new HashMap();
        for (Iterator i = builders.keySet().iterator(); i.hasNext();) {
            String variableName = (String) i.next();
            PredicateExecutorBuilder builder = (PredicateExecutorBuilder) builders.get(variableName);
            predicateExecutors.put(variableName, builder.createPredicateExecutor());
        }
        return predicateExecutors;
    }
    
    /**
     * Returns builder lazyly
     */
    private PredicateExecutorBuilder getBuilder(String variableName) {
        PredicateExecutorBuilder builder = (PredicateExecutorBuilder) builders.get(variableName);
        if (builder == null) {
            TypeInfo typeInfo = typeInfoProvider.getTypeInfo(variableName);
            if (typeInfo.getType() == TypeInfo.BYTE.getType() ||
                typeInfo.getType() == TypeInfo.SHORT.getType() ||
                typeInfo.getType() == TypeInfo.INT.getType() ||
                typeInfo.getType() == TypeInfo.LONG.getType() ||
                typeInfo.getType() == TypeInfo.CHAR.getType()) {
                builder = new DiscreteRangePredicateExecutorBuilder(typeInfo);
            } else if (
                typeInfo.getType() == TypeInfo.FLOAT.getType() ||
                typeInfo.getType() == TypeInfo.DOUBLE.getType()) {
                builder = new RangePredicateExecutorBuilder(typeInfo);
            } else if (
                typeInfo.getType() == TypeInfo.STRING.getType() ||
                typeInfo.getType() == TypeInfo.DISCRETE.getType()) {
                builder = new DiscreteSetPredicateExecutorBuilder();
            } else if (typeInfo.getType() == TypeInfo.FINIT_SET.getType()) {
                builder = new FinitSetPredicateExecutorBuilder(typeInfo.getValuesSet());
            } else {
                throw new IllegalArgumentException(
                        "Can't get TypeInfo about variable [" + variableName + "]");
            }
            builders.put(variableName, builder);
        }
        return builder;
    }
    
    private int ceilLog2(int argument) {
        int ceilLog2 = (int) Math.ceil(Math.log(argument) / Math.log(2));
        return ceilLog2;
    }
}
