/*
 * Copyright (c) 1999-2006 eVelopers Corporation. All rights reserved.
 *
 * This is open source software; you can use, redistribute and/or modify 
 * it under the terms of the Open Software Licence v 2.1 as published by the Open 
 * Source Initiative.
 *
 * You should have received a copy of the Open Software Licence along with this
 * application; if not, contact the Open Source Initiative (http://opensource.org).
 */
package com.evelopers.unimod.analysis.executors;

import java.util.Arrays;
import java.util.Map;

public class Simplifier {
    private LongSet substitutions;
    private Substitution substitution;
    private long[] mask;

    public Simplifier(Substitution substitution, LongSet substitutions) {
        this.substitution = substitution;
        this.substitutions = substitutions;
        this.mask = new long[substitutions.size()];
        for (int i = 0; i < substitutions.size(); i++) {
            mask[i] = (1 << substitution.getBits()) - 1;
        }
    }

    public void simplify() {        
        findRedundantLetters();
        removeRedundantSubstitutions();
    }

    public void findRedundantLetters() {
        Map positionMap = substitution.getPositions();
        Position[] positions = (Position[]) positionMap.values().toArray(
                new Position[positionMap.values().size()]);
        Arrays.sort(positions);
        
        /* For each variable in the formula */
        for (int i = 0; i < positions.length; i++) {
            if (positions[i] instanceof PredicatePosition) {
                /* If variable has range */
                PredicatePosition predicatePosition = (PredicatePosition) positions[i];
                buildPredicateMask(predicatePosition);                
            } else {
                /* If variable is boolean */
                buildBooleanVarMask(positions[i]);
            }
        }
    }
    
    public void removeRedundantSubstitutions() {
        for (int i = 0; i < mask.length; i++) {
            if (mask[i] == 0L) {
                substitutions = new LongSet();
                mask = new long[0];
                return;
            }
        }
        
        boolean[] redundant = new boolean[substitutions.size()];
        Arrays.fill(redundant, false);      
        for (int i = 0; i < substitutions.size(); i++) {
            for (int j = 0; j < substitutions.size(); j++) {
                if (i != j && ! redundant[j] && 
                    ((mask[i] & mask[j]) == mask[j]) && 
                    (substitutions.get(i) & mask[j]) == (substitutions.get(j) & mask[j])) {
                    redundant[i] = true;
                    break;
                }
            }
        }
        
        LongSet newSubstitutions = new LongSet();
        for (int i = 0; i < substitutions.size(); i++) {
            if (! redundant[i]) {
                newSubstitutions.add(substitutions.get(i));
            }
        }

        long[] newMask = new long[newSubstitutions.size()];
        for (int i = 0, j = 0; i < substitutions.size(); i++) {
            if (! redundant[i]) {
                newMask[j] = mask[i];
                j ++;
            }
        }
        
        substitutions = newSubstitutions;
        mask = newMask;
    }

    public long[] getMask() {
        return mask;
    }
    
    public LongSet getSubstitutions() {
        return substitutions;
    }
    
    private void buildBooleanVarMask(Position position) {
        long two_k = 1L << position.position;
        long not_Two_k = ~two_k;
        for (int t1 = 0; t1 < substitutions.size(); t1++) {
            long l1 = substitutions.get(t1);
            long l2 = l1 ^ two_k;
            int t2 = substitutions.indexOf(l2);
            if (t2 >= 0 && (mask[t1] & mask[t2]) == mask[t2]) {
                mask[t1] = mask[t1] & not_Two_k;
            }
        }
    }

    private void buildPredicateMask(PredicatePosition position) {
        long varBits = (1L << (position.position + position.bits)) - (1L << position.position);
        long notVarBits = ~varBits;
        
        subst: 
        for (int t1 = 0; t1 < substitutions.size(); t1++) {
            long l1 = substitutions.get(t1);
            values:
            for (long v = 0; v < position.cardinality; v++) {
                long l2 = (l1 & notVarBits) + (v << position.position);
                if (l1 == l2) {
                    continue values;
                }
                int t2 = substitutions.indexOf(l2);
                if (t2 < 0 || (mask[t1] & mask[t2]) != mask[t2]) {
                    continue subst;
                }
            }
            mask[t1] = mask[t1] & notVarBits;
        }
    }
}
