/*
 * Copyright (c) 1999-2006 eVelopers Corporation. All rights reserved.
 *
 * This is open source software; you can use, redistribute and/or modify 
 * it under the terms of the Open Software Licence v 2.1 as published by the Open 
 * Source Initiative.
 *
 * You should have received a copy of the Open Software Licence along with this
 * application; if not, contact the Open Source Initiative (http://opensource.org).
 */
package com.evelopers.unimod.analysis.executors;

import java.util.Arrays;
import java.util.Comparator;

import com.evelopers.unimod.analysis.TypeInfo;
import com.evelopers.unimod.analysis.TypeInfoProvider;

public class FormulaGenerator {
    private TypeInfoProvider provider;
    private Substitution substitution;
    private LongSet substitutions;
    private long[] mask;    

    public FormulaGenerator(TypeInfoProvider provider, Substitution substitution, LongSet substitutions, long[] mask) {
        this.provider = provider;
        this.substitution = substitution;
        this.substitutions = substitutions;
        this.mask = mask;
    }
    
    public String generateExpression() {
        String[] variableNames = (String[]) substitution.getPositions().keySet().toArray(
                new String[substitution.getPositions().keySet().size()]);
        Arrays.sort(variableNames, new Comparator() {
            public int compare(Object o1, Object o2) {
                Position p1 = (Position) substitution.getPositions().get(o1);
                Position p2 = (Position) substitution.getPositions().get(o2);
                return p1.compareTo(p2);
            }
        });

        StringBuffer buffer = new StringBuffer();
        boolean firstTerm = true, firstLetter;
        for (int i = 0; i < substitutions.size(); i++) {
            if (firstTerm) {
                firstTerm = false;                
            } else {
                buffer.append(" || ");
            }
            firstLetter = true;
            for (int j = 0; j < variableNames.length; j++) {
                Position position = (Position) substitution.getPositions().get(variableNames[j]);
                if ((mask[i] & (1L << position.position)) != 0) {
                    if (firstLetter) {
                        firstLetter = false;
                    } else {
                        buffer.append(" && ");
                    }
                    if (position instanceof PredicatePosition) {
                        addPredicateLetter(variableNames[j], (PredicatePosition) position, substitutions.get(i), buffer);
                    } else {
                        addBooleanVarLetter(variableNames[j], position, substitutions.get(i), buffer);
                    }
                }
            }
        }

        return buffer.toString();
    }

    private void addBooleanVarLetter(String variableName, Position position, long substitution, StringBuffer buffer) {        
        if ((substitution & (1L << position.position)) == 0) {
            buffer.append('!');
        }
        buffer.append(variableName);
    }

    private void addPredicateLetter(String variableName, PredicatePosition position, long substitution, StringBuffer buffer) {
        PredicateExecutor predicateExecutor = position.predicateExecutor;
        substitution = (substitution >> position.position) & ((1L << position.bits) - 1);
        if (predicateExecutor instanceof DiscreteSetPredicateExecutor) {
            DiscreteSetPredicateExecutor executor = (DiscreteSetPredicateExecutor) predicateExecutor;
            Object value = executor.getValue((int) substitution);
            String sValue = provider.constantToString(variableName, value);
            buffer.append(variableName).append(" == ").append(sValue);
        } else {
            RangePredicateExecutor executor = (RangePredicateExecutor) predicateExecutor;
            Interval interval = executor.getInterval((int) substitution);
            TypeInfo typeInfo = provider.getTypeInfo(variableName);

            boolean leftPredicate = false;
            
            if (interval.getMax().equals(interval.getMin()) && interval.isLeftIncluded() && interval.isRightIncluded()) {
                String value = provider.constantToString(variableName, interval.getMin());
                buffer.append(variableName).append(" == ").append(value);
            } else {            
                if (! interval.getMin().equals(typeInfo.getMin())) {
                    leftPredicate = true;
                    String min = provider.constantToString(variableName, interval.getMin());
                    String predicate = interval.isLeftIncluded() ? ">=" : ">";
                    buffer.append(variableName).append(" ").append(predicate).append(" ").append(min);
                }
    
                if (! interval.getMax().equals(typeInfo.getMax())) {
                    if (leftPredicate) {
                        buffer.append(" && ");
                    }
                    String max = provider.constantToString(variableName, interval.getMax());
                    String predicate = interval.isRightIncluded() ? "<=" : "<";
                    buffer.append(variableName).append(" ").append(predicate).append(" ").append(max);            
                }
            }
        }
    }

}
