/*
 * Copyright (c) 1999-2005 eVelopers Corporation. All rights reserved.
 *
 * This is open source software; you can use, redistribute and/or modify 
 * it under the terms of the Open Software Licence v 2.1 as published by the Open 
 * Source Initiative.
 *
 * You should have received a copy of the Open Software Licence along with this
 * application; if not, contact the Open Source Initiative (http://opensource.org).
 */
package com.evelopers.unimod.analysis;

import java.util.Set;

import org.apache.commons.lang.BooleanUtils;

/**
 * @uml.dependency   supplier="com.evelopers.unimod.analysis.TypeInfo" stereotypes="Basic::Create"
 */
public abstract class TypeInfoProvider {
    
    public TypeInfo getTypeInfo(String variableName) {
        if (getType(variableName) == TypeInfo.BYTE.getType()) {
            return TypeInfo.BYTE;
        } else if (getType(variableName) == TypeInfo.SHORT.getType()) {
            return TypeInfo.SHORT;
        } else if (getType(variableName) == TypeInfo.INT.getType()) {
            return TypeInfo.INT;
        } else if (getType(variableName) == TypeInfo.LONG.getType()) {
            return TypeInfo.LONG;
        } else if (getType(variableName) == TypeInfo.FLOAT.getType()) {
            return TypeInfo.FLOAT;
        } else if (getType(variableName) == TypeInfo.DOUBLE.getType()) {
            return TypeInfo.DOUBLE;
        } else if (getType(variableName) == TypeInfo.BOOLEAN.getType()) {
            return TypeInfo.BOOLEAN;
        } else if (getType(variableName) == TypeInfo.CHAR.getType()) {
            return TypeInfo.CHAR;
        } else if (getType(variableName) == TypeInfo.STRING.getType()) {
            return TypeInfo.STRING;
        } else if (getType(variableName) == TypeInfo.DISCRETE.getType()) {
            return TypeInfo.createDiscreteTypeInfo(getValueClass(variableName));
        } else if (getType(variableName) == TypeInfo.FINIT_SET.getType()) {
            return TypeInfo.createFinitSetTypeInfo(getValuesSet(variableName), getValueClass(variableName));
        }
        throw new IllegalArgumentException(
                "Can't get TypeInfo about variable [" + variableName + "]");
    }

    /**
     * <p>
     * Converts string value into appropriate object that represents
     * the constant value. 
     * </p>
     * <p>
     * <b>Note:</b>
     * This method should be overriden for {@link TypeInfo#DISCRETE} and
     * {@link TypeInfo#FINIT_SET}.
     * </p>
     * @param variableName full name of the variable compared with the constant.
     * @param constantValue string presentation of the constant.
     * @return constant value converted to the appropriate type. 
     */
    public Object parseConstant(String variableName, String constantValue) {
        if (getType(variableName) == TypeInfo.BYTE.getType()) {
            return Byte.decode(constantValue);
        } else if (getType(variableName) == TypeInfo.SHORT.getType()) {
            return Short.decode(constantValue);
        } else if (getType(variableName) == TypeInfo.INT.getType()) {
            return Integer.decode(constantValue);
        } else if (getType(variableName) == TypeInfo.LONG.getType()) {
            return Long.decode(constantValue);
        } else if (getType(variableName) == TypeInfo.FLOAT.getType()) {
            return Float.valueOf(constantValue);
        } else if (getType(variableName) == TypeInfo.DOUBLE.getType()) {
            return Double.valueOf(constantValue);
        } else if (getType(variableName) == TypeInfo.BOOLEAN.getType()) {
            return BooleanUtils.toBooleanObject(constantValue);
        } else if (getType(variableName) == TypeInfo.CHAR.getType()) {
            if (constantValue.startsWith("'") && constantValue.endsWith("'") && constantValue.length() == 3) {
                return new Character(constantValue.charAt(1));
            }
        } else if (getType(variableName) == TypeInfo.STRING.getType()) {
            if (constantValue.startsWith("\"") && constantValue.endsWith("\"") && constantValue.length() >= 2) {
                return constantValue.substring(1, constantValue.length() - 1);
            }
        }
        throw new IllegalArgumentException(
                "Can't cast constant [" + constantValue + "]" +
                " to type of variable [" + variableName + "]");
    }

    /**
     * <p>
     * Converts object into appropriate string value that represents
     * the constant value. 
     * </p>
     * <p>
     * <b>Note:</b>
     * This method should be overriden for {@link TypeInfo#DISCRETE} and
     * {@link TypeInfo#FINIT_SET}.
     * </p>
     * @param variableName full name of the variable compared with the constant.
     * @param constantValue object presentation of the constant.
     * @return constant value converted to the string. 
     */
    public String constantToString(String variableName, Object constantValue) {
        if (getType(variableName) == TypeInfo.BYTE.getType()
            || getType(variableName) == TypeInfo.SHORT.getType()
            || getType(variableName) == TypeInfo.INT.getType()
            || getType(variableName) == TypeInfo.LONG.getType()
            || getType(variableName) == TypeInfo.FLOAT.getType()
            || getType(variableName) == TypeInfo.DOUBLE.getType()
            || getType(variableName) == TypeInfo.BOOLEAN.getType()) {
            return String.valueOf(constantValue);
        } else if (getType(variableName) == TypeInfo.CHAR.getType()) {
            return "'" + String.valueOf(constantValue) + "'";
        } else if (getType(variableName) == TypeInfo.STRING.getType()) {
            return "\"" + String.valueOf(constantValue) + "\"";
        }
        throw new IllegalArgumentException(
                "Can't cast constant [" + constantValue + "]" +
                " to string for variable [" + variableName + "]");
    }
    
    /**
     * Returns one of
     * <ol>
     *   <li>{@link TypeInfo#BYTE}</li>
     *   <li>{@link TypeInfo#SHORT}</li>
     *   <li>{@link TypeInfo#INT}</li>
     *   <li>{@link TypeInfo#LONG}</li>
     *   <li>{@link TypeInfo#FLOAT}</li>
     *   <li>{@link TypeInfo#DOUBLE}</li>
     *   <li>{@link TypeInfo#BOOLEAN}</li>
     *   <li>{@link TypeInfo#CHAR}</li>
     *   <li>{@link TypeInfo#FINIT_SET}</li>
     *   <li>{@link TypeInfo#DISCRETE}</li>
     * </ol> 
     * @param variableName full name of the variable.
     * @return type of the variable.
     */
    protected abstract int getType(String variableName);

    protected abstract Set getValuesSet(String variableName);
    
    /**
     * Returns Java class of the varaible values. All values corresponding to
     * the variable should be of this class.  
     * This method should be implemented only for the DISCRETE and FINIT_SET
     * variables.
     * @param variableName full name of the variable.
     * @return Returns Java class of the varaible values.
     */
    protected abstract Class getValueClass(String variableName);
}
