/*
 * Copyright (c) 1999-2005 eVelopers Corporation. All rights reserved.
 *
 * This is open source software; you can use, redistribute and/or modify 
 * it under the terms of the Open Software Licence v 2.1 as published by the Open 
 * Source Initiative.
 *
 * You should have received a copy of the Open Software Licence along with this
 * application; if not, contact the Open Source Initiative (http://opensource.org).
 */
package com.evelopers.unimod.analysis;

import java.util.Set;

public class TypeInfo {

    public static final TypeInfo BYTE       =  new TypeInfo(0, new Byte(Byte.MIN_VALUE), new Byte(Byte.MAX_VALUE), null, Byte.class) {
        public Object next(Object value) {
            Byte b = (Byte) value;
            return new Byte((byte) (b.byteValue() + 1));
        }
    };
    public static final TypeInfo SHORT      =  new TypeInfo(1, new Short(Short.MIN_VALUE), new Short(Short.MAX_VALUE), null, Short.class) {
        public Object next(Object value) {
            Short s = (Short) value;
            return new Short((short) (s.shortValue() + 1));
        }
    };
    public static final TypeInfo INT        =  new TypeInfo(2, new Integer(Integer.MIN_VALUE), new Integer(Integer.MAX_VALUE), null, Integer.class) {
        public Object next(Object value) {
            Integer i = (Integer) value;
            return new Integer(i.intValue() + 1);
        }
    };
    public static final TypeInfo LONG       =  new TypeInfo(3, new Long(Long.MIN_VALUE), new Long(Long.MAX_VALUE), null, Long.class) {
        public Object next(Object value) {
            Long l = (Long) value;
            return new Long(l.longValue() + 1);
        }
    };
    public static final TypeInfo FLOAT      =  new TypeInfo(4, new Float(Float.NEGATIVE_INFINITY), new Float(Float.POSITIVE_INFINITY), null, Float.class);
    public static final TypeInfo DOUBLE     =  new TypeInfo(5, new Double(Double.NEGATIVE_INFINITY), new Double(Double.POSITIVE_INFINITY), null, Double.class);
    public static final TypeInfo BOOLEAN    =  new TypeInfo(6, null, null, null, Boolean.class);
    public static final TypeInfo CHAR       =  new TypeInfo(7, new Character(Character.MIN_VALUE), new Character(Character.MAX_VALUE), null, Character.class) {
        public Object next(Object value) {
            Character c = (Character) value;
            return new Character((char) (c.charValue() + 1));
        }
    };
    public static final TypeInfo DISCRETE   =  new TypeInfo(8, null, null, null, null);
    public static final TypeInfo STRING     =  new TypeInfo(9, null, null, null, String.class);
    public static final TypeInfo FINIT_SET  =  new TypeInfo(10, null, null, null, null);

    private final int type;
    private final Comparable min;
    private final Comparable max;
    private final Set valuesSet; 
    private final Class valueClass; 

    public static final TypeInfo createFinitSetTypeInfo(Set valuesSet, Class valueClass) {
        return new TypeInfo(FINIT_SET.getType(), null, null, valuesSet, valueClass);
    }

    public static final TypeInfo createDiscreteTypeInfo(Class valueClass) {
        return new TypeInfo(DISCRETE.getType(), null, null, null, valueClass);
    }
    
    private TypeInfo(int type, Comparable min, Comparable max, Set valuesSet, Class valueClass) {
        this.type = type;
        this.max = max;
        this.min = min;
        this.valuesSet = valuesSet;
        this.valueClass = valueClass;
    }

    public int getType() {
        return type;
    }
    
    /**
     * For range values variables returns min value.
     * @throws OperationNotSupportedException 
     */
    public Comparable getMin() {
        if (min != null) {
            return min;
        } else {
            throw new UnsupportedOperationException("No min value is defined");
        }
    }
    
    /**
     * For range values variables returns max value.
     * @param variableName name of the variable
     * @return max value of the variable <code>variableName</b> 
     */
    public Comparable getMax() {
        if (max != null) {
            return max;
        } else {
            throw new UnsupportedOperationException("No max value is defined");
        }
    }

    /**
     * For variables of type {@link #FINIT_SET} returns set
     * of admitted values.
     * @return set of admitted values 
     */
    public Set getValuesSet() {
        if (valuesSet != null) {
            return valuesSet;
        } else {
            throw new UnsupportedOperationException("No set of values is defined");
        }
    }
 
    public Class getValueClass() {
        if (valueClass != null) {
            return valueClass;
        } else {
            throw new UnsupportedOperationException("No value class is defined");
        }
    }
    
    /**
     * For range values variables returns next after given value.
     * Range are varaible of types 
     * <ul>
     *   <li>{@link #BYTE}</li>
     *   <li>{@link #SHORT}</li>
     *   <li>{@link #INT}</li>
     *   <li>{@link #LONG}</li>
     *   <li>{@link #CHAR}</li>
     * </ul>
     * @param variableName name of the variable
     * @param value value to be incremented
     * @return incremented <code>value</code>
     */
    public Object next(Object value) {
        throw new UnsupportedOperationException("No next value is defined");
    }
}
