/*
 * Developed by eVelopers Corporation - April, 2005
 *
 * $Date: 19-Jul-05 13:22:41$
 *
 */
package com.evelopers.common.util.csv;

import java.io.*;
import java.lang.reflect.InvocationTargetException;
import java.util.*;

import org.apache.commons.beanutils.BeanUtils;
import org.apache.commons.beanutils.ConvertUtils;

import com.evelopers.common.exception.CommonException;
import com.evelopers.common.exception.SystemException;
import com.evelopers.common.util.csv.iterator.ExportIterator;

/**
 * Utility class for encoding Comma Separated Variable (CSV) files.
 * 
 * Provides methods to export arrays of objects, collections of objects,
 * and data from export iterators 
 * ({@link com.evelopers.common.util.csv.iterator.ExportIterator}) to file,
 * to stream or using specified writer.
 * 
 * Exporting with ExportIterator has two variants: when iterator's next() 
 * method retirns maps of properties and when it returns objects of 
 * specified class.
 * 
 * <p>Note: to convert objects to strings Apache Converter Utils are used
 * (<code>org.apache.commons.beanutils.ConvertUtils</code>). 
 *
 * @author: loukiana
 * @version $Revision: 1$
 */
public class CSVFileEncoder {
    
    public static final String HALT_PROCESS = "haltOnAnyException";
    public static final String FILE_FORMAT = "fileFormat";

    protected boolean haltProcess = true;
    String[] format = null;
    Properties properties;
    
    protected EncoderState state = new EncoderState();
    protected DataModel data = new DataModel();
    
    public CSVFileEncoder() {
        this.properties = new Properties();
    }
    
    public CSVFileEncoder(Properties properties) {
        setProperties(properties);
    }
    
    public void export(File file, Object[] objects) throws CommonException {
    	try {
    		export(new PrintWriter(new FileWriter(file)), objects);
    	} catch (IOException ioe) {
    		throw new SystemException(ioe, "Error exporting data.");
    	}
    }

    public void export(File file, ExportIterator iterator) throws CommonException {
    	try {
    		export(new PrintWriter(new FileWriter(file)), iterator);
    	} catch (IOException ioe) {
    		throw new SystemException(ioe, "Error exporting data.");
    	}
    }
    
    public void export(File file, ExportIterator iterator, Class beanClass) throws CommonException {
    	try {
    		export(new PrintWriter(new FileWriter(file)), iterator);
    	} catch (IOException ioe) {
    		throw new SystemException(ioe, "Error exporting data.");
    	}
    }

    public void export(OutputStream os, Object[] objects) throws CommonException {
        export(new PrintWriter(os), objects);
    }
    
    public void export(OutputStream os, Collection objects) throws CommonException {
        export(new PrintWriter(os), objects);
    }
    
    public void export(OutputStream os, ExportIterator iterator) throws CommonException {
        export(new PrintWriter(os), iterator);
    }
    
    public void export(OutputStream os, ExportIterator iterator, Class beanClass) throws CommonException {
        export(new PrintWriter(os), iterator);
    }
    
    public void export(PrintWriter writer, Object[] objects) throws CommonException {
        state.init();
        data.init();
        
        if (objects == null || objects.length == 0) return;
        if (format == null) throw new SystemException("Format is undefined");
        
        //export to DataModel        
        for (int i = 0, l = objects.length; i < l; i++) {
            try {
                String[] tokens = objectToTokens(objects[i]);
                data.addObject(tokens);
                state.objectExported();
            } catch (Exception e) {
                if (haltProcess) {
                    state.interrupt();
                    break;
                } else {
                    state.failedToExportObject(i+1, e);
                }
            }
        }
        
        //store data to file
        try {
        	CSVEncoder.export(data, writer);
        } catch (IOException ioe) {
        	throw new SystemException(ioe, "Error exporting data.");
        }
    }
    
    public void export(PrintWriter writer, Collection objects) throws CommonException {
        state.init();
        data.init();
        
        if (objects == null || objects.size() == 0) return;
        if (format == null) throw new SystemException("Format is undefined");
        
        //export to DataModel        
        int i = 0;
        for (Iterator iter = objects.iterator(); iter.hasNext(); i++) {            
            try {
                String[] tokens = objectToTokens(iter.next());
                data.addObject(tokens);
                state.objectExported();                
            } catch (Exception e) {
                if (haltProcess) {
                    state.interrupt();
                    throw new SystemException(e);                    
                } else {
                    state.failedToExportObject(i+1, e);
                }
            }
        }
        
        //store data to file
        try {
        	CSVEncoder.export(data, writer);
        } catch (IOException ioe) {
        	throw new SystemException(ioe, "Error exporting data.");
        }
    }
    
    /**
     * Export data from ExportIterator. It's next() method must return maps
     * of properties to export.
     *  
     * <p>Note: doesn't close ExportIterator.
     * @param writer writer used to export data
     * @param iterator used to obtain data
     * @throws CommonException
     */
    public void export(PrintWriter writer, ExportIterator iterator) throws CommonException {
        state.init();
        data.init();
        
        if (iterator == null) return;
        if (format == null) throw new SystemException("Format is undefined");
        
        //export to DataModel
        int i = 0;
        for (Map row = iterator.next(); row != null; row = iterator.next(), i++) {
            try {                
                String[] tokens = mapToTokens(row);
                data.addObject(tokens);
                state.objectExported();                
            } catch (Exception e) {
                if (haltProcess) {
                    state.interrupt();
                    throw new SystemException(e);
                } else {
                    state.failedToExportObject(i+1, e);
                }
            } 
        }
        
        //store data to file
        try {
        	CSVEncoder.export(data, writer);
        } catch (IOException ioe) {
        	throw new SystemException(ioe, "Error exporting data.");
        }
    }
    
    /**
     * Export data from ExportIterator. It's next(Class beanClass) method must 
     * return objects of <code>beanClass</code>.
     *  
     * <p>Note: doesn't close ExportIterator.
     * @param writer writer used to export data
     * @param iterator used to obtain data
     * @param beanClass class to pass into ExportIterator.next() method.
     * @throws CommonException
     */
    public void export(PrintWriter writer, ExportIterator iterator, Class beanClass) throws CommonException {
        state.init();
        data.init();
        
        if (iterator == null) return;
        if (format == null) throw new SystemException("Format is undefined");
        
        //export to DataModel
        Object row;       
        int i = 0;
        while ((row = iterator.next(beanClass)) != null) {
            try {
                String[] tokens = objectToTokens(row);
                data.addObject(tokens);
                state.objectExported();
                i++;
            } catch (Exception e) {
                if (haltProcess) {
                    state.interrupt();
                    throw new SystemException(e);
                } else {
                    state.failedToExportObject(i+1, e);
                }
            }
        }
        
        //store data to file
        try {
        	CSVEncoder.export(data, writer);
        } catch (IOException ioe) {
        	throw new SystemException(ioe, "Error exporting data.");
        }
    }

    public void setProperties(Properties properties) {
        this.properties = properties;
        
        String str = properties.getProperty(CSVFileEncoder.HALT_PROCESS);
        haltProcess = str != null && Boolean.valueOf(str).booleanValue();
        
        StringTokenizer st = new StringTokenizer(properties.getProperty(CSVFileEncoder.FILE_FORMAT), ",; ");
        format = new String[st.countTokens()];
        for (int i = 0; st.hasMoreTokens(); i++) {
            format[i] = st.nextToken();
        }
    }

    public EncoderState getState() {
        return state;
    }
    
    protected String[] objectToTokens(Object object) throws CommonException {
        String[] list = new String[format.length];

        for (int i = 0, l = format.length; i < l; i++) {
            try {
                list[i] = BeanUtils.getProperty(object, format[i]);
            } catch (IllegalAccessException iae) {
                throw new SystemException(iae, "Failed to get bean's properties, access error on property '"+format[i]+"'.");
            } catch (InvocationTargetException ite) {
                throw new SystemException(ite, "Failed to get bean's properties, invocation error on property '"+format[i]+"'.");
            } catch (NoSuchMethodException nsme) {
                throw new SystemException(nsme, "Failed to get bean's properties, no property '"+format[i]+"' specified.");
            }
        }
        
        return list;
    }
    
    protected String[] mapToTokens(Map map) throws CommonException {
        String[] list = new String[format.length];

        for (int i = 0, l = format.length; i < l; i++) {
            list[i] = ConvertUtils.convert(map.get(format[i]));
        }
        
        return list;
    }
    
    public class EncoderState {
        public static final int OK = 0;
        public static final int INTERRUPTED = 1;
        public static final int ERROR = 2;
        
        private int state = OK;
        private int objectsExported = 0;        
        private ArrayList errors = null;
        
        protected void init() {
            state = OK;
            objectsExported = 0;
            errors = null;
        }
        
        protected void objectExported() {
            objectsExported++; 
        }
        
        protected void failedToExportObject(int number, Exception e) {
            if (errors == null) errors = new ArrayList();
            errors.add(new ExportError(number, e));
            state = ERROR;
        }
        
        protected void interrupt() {
            state = INTERRUPTED;
        }
        
        public int getState() {
            return state;
        }
        
        public boolean isOK() {
            return state == OK;
        }
        
        public int objectsExported() {
            return objectsExported;
        }
        
        public ExportError[] getErrors() {
            return (ExportError[])errors.toArray(new ExportError[errors.size()]);
        }
    }
    
    public class ExportError extends Exception {
        int objectNumber = 0;
        Exception exception = null;
        
        protected ExportError(int number, Exception e) {
            objectNumber = number;
            exception = e;
        }
        
        public String getMessage() {
            StringBuffer sb = new StringBuffer("Failed to export object ");
            if (objectNumber >= 0) sb.append(objectNumber).append(" ");
            sb.append("due to error: "+(exception != null ? exception.getMessage(): "unknown"));
            return sb.toString();
        }
    }
    
    public class DataModel implements CSVEncoder.DataModel {
        private ArrayList lines = new ArrayList();
        
        protected void init() {
            lines = new ArrayList();
        }
        
        protected void addObject(String[] tokens) {
            lines.add(tokens);
        }
        
        public Object get(int column, int row) throws CommonException {
            if (row >= lines.size()) 
                throw new IndexOutOfBoundsException("Row number ("+row+") is out of range");
            
            String[] tokens = (String[])lines.get(row);
            
            if (tokens == null || tokens.length <= column)
                throw new IndexOutOfBoundsException("Column number ("+column+" is out of range");
            
            return tokens[column];
        }
        
        public int getColumnCount() throws CommonException {
            if (CSVFileEncoder.this.format != null)
                return CSVFileEncoder.this.format.length;
            else return -1;
        }
        public String getColumnName(int column) throws CommonException {
            if (CSVFileEncoder.this.format != null)
                return CSVFileEncoder.this.format[column];
            else return null;
        }
        public int getFlushCount() throws CommonException {
            return CSVEncoder.DataModel.NO_FLUSH;
        }
        public int getRowCount() throws CommonException {
            if (lines != null)
                return lines.size();
            else return 0;
        }
        public boolean needHeader() throws CommonException {
            return false;
        }
    }
}
