/*
 * Developed by eVelopers Corporation - April, 2005
 *
 * $Date: 19-Jul-05 13:22:41$
 *
 */
package com.evelopers.common.util.csv;

import java.util.*;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;

import com.evelopers.common.exception.CommonException;
import com.evelopers.common.exception.SystemException;
import com.evelopers.common.exception.ValidationError;
import com.evelopers.common.util.helper.BeanHelper;

import org.apache.commons.beanutils.BeanUtils;

/**
 * Utility class for decoding Comma Separated Variable (CSV) files.
 *
 * @author: loukiana
 * @version $Revision: 1$
 */
public class CSVFileDecoder {
    
    public static final String BEAN_CLASS = "bean_class_name";
    public static final String FILE_FORMAT = "file_format";
    public static final String VALIDATION_FALL_REPLY = "validationFallReply";
    public static final String DEFAULTS = "defaultValues";
    
    public interface Replies {
        /* with the very first validation error throw an exception, don't try to use any default value */
        public static final String HALT_PROCESS = "Halt process and throw exception";    
        public static final String SKIP = "Skip any invalid line";    
        /* if default value is set for token and validation fails, validation is applied to
         * the default value. if dafult is valid, it is used, overwise exception is thrown */
        public static final String USE_DEFAULT_OR_HALT = "Use default or halt and throw exception";
        /* same as in the previous case but skip the line instead of throwing exception */
        public static final String USE_DEFAULT_OR_SKIP = "Use default or halt and skip line";
        /* same as USE_DEFAULT_OR_HALT but try to use dafault only if token is empty */
        public static final String USE_DEFAULT_ON_EMPTY_OR_HALT = "Use default on empty token else halt";    
        /* same as in the previous case but skip the line instead of throwing exception */
        public static final String USE_DEFAULT_ON_EMPTY_OR_SKIP = "Use default on empty token else skip line";
    }
    
    protected boolean doValidate = false;
    
    // reply:
    protected boolean useDefault = true;
    protected boolean useOnEmpty = true;
    protected boolean lineHalt = false;
    
    protected Map validationRules = null;
    
    Properties properties;
    String[] format = null;
    String[] defaults = null;
    
    protected DecoderState state = new DecoderState();
    
    public CSVFileDecoder() {
        this.properties = new Properties();
    }
    
    public CSVFileDecoder(boolean doValidate) {
        this();
        this.doValidate = doValidate;
    }

    public CSVFileDecoder(Properties properties) throws CommonException {
        setProperties(properties);
    }

    public CSVFileDecoder(Properties properties, boolean doValidate) throws CommonException {
        this(properties);
        this.doValidate = doValidate;
    }
    
    public void setProperties(Properties properties) throws CommonException {
        this.properties = properties;
        
        //load reply
        String str = properties.getProperty(VALIDATION_FALL_REPLY);
        useDefault = !Replies.HALT_PROCESS.equals(str) && !Replies.SKIP.equals(str);
        useOnEmpty = useDefault && (Replies.USE_DEFAULT_ON_EMPTY_OR_SKIP.equals(str) 
        				|| Replies.USE_DEFAULT_ON_EMPTY_OR_HALT.equals(str));
        lineHalt = Replies.HALT_PROCESS.equals(str) 
        				|| Replies.USE_DEFAULT_OR_HALT.equals(str) 
        				|| Replies.USE_DEFAULT_ON_EMPTY_OR_HALT.equals(str);
        
        //load format
        StringTokenizer st = new StringTokenizer(properties.getProperty(FILE_FORMAT), ",; ");
        format = new String[st.countTokens()];
        for (int i = 0; st.hasMoreTokens(); i++) {
            format[i] = st.nextToken();
        }
        
        //load defaults
        String def = properties.getProperty(DEFAULTS);
        if (def != null && def.length() > 0)
            defaults = CSVDecoder.splitCSVString(def);
    }
    
    public void doValidate(boolean doValidate) {
        this.doValidate = doValidate; 
    }
    
    /*
     * Parameter <code>rules</code> here contains
     * pairs where key is a string - namne of property
     * to validate, values is an array of strings --
     * test names. 
     */
    public void setValidationRules(Map rules) {
       validationRules = rules; 
    }
        
    public Object[] readFile(java.io.Reader reader) throws CommonException {
        return readFile(reader, null);
    }
    
    public Object[] readFile(File file) throws CommonException {
    	try {
    		return readFile(new java.io.FileReader(file), null);
    	} catch (FileNotFoundException fnfe) {
    		throw new SystemException(fnfe, "Error importing data, file not found.");
    	}
    }
    
    public Object[] readFile(File file, Object[] objects) throws CommonException {
    	try {
    		return readFile(new java.io.FileReader(file), objects);
    	} catch (FileNotFoundException fnfe) {
    		throw new SystemException(fnfe, "Error importing data, file not found.");
    	}
    }
    
    public Object[] readFile(java.io.Reader reader, Object[] objects) throws CommonException {
        
        ArrayList result = null;

        if (objects == null)
            result = new ArrayList();

        state.init();

        java.io.BufferedReader csvReader = (reader instanceof java.io.BufferedReader)
        							? (java.io.BufferedReader)reader
        							: new java.io.BufferedReader(reader);
        
        String line = null;
        Object object = null;
        
        try {
	        while((line = csvReader.readLine()) != null) {
	
	            //skip comment line
	            if(line.trim().startsWith("*---")) continue;
	
	            //skip empty line
	            if(line.trim().length() < 1) continue;
	
	            //get bean to populate
	            if (objects != null) 
	                if (objects.length > state.getLinesRead())
	                    object = objects[state.getLinesRead()];
	                else
	                    break;
	            else 
	                object = null;
	
	            // increment number of lines
	            state.lineRead();
	
	            //validate line, populate bean
	            try {
	                object = readLine(line, object);
	
	                //increment number of lines validated
	                if (doValidate)
	                    state.lineValidated();
	                
	            } catch (ValidationError error) {
	                error.setNumber(state.getLinesRead());
	                if (lineHalt)
	                    throw error;
	                else
	                    state.invalidLine(error);                    
	            }
	            
	            if (objects == null && object != null)
	                result.add(object);
	            
	        }     
        }catch (IOException ioe) {
        	throw new SystemException(ioe, "Error reading imported data.");
        } finally {
            if(csvReader != null) {
                try {
                    csvReader.close();
                } catch(Exception e) {}
            }
        }
        
        if (objects != null)
            return objects;
        else return result.toArray();
    }
    
    /*
     * If <code>object</code> is not null, this method pupolates it with properties
     * values got from <code>line</code> using FILE_FORMAT string to identify 
     * names of properties.
     * 
     * If <code>object</code> is null than this method instantiates it using BEAN_CLASS
     * string as its type and that populates instantiated object. 
     * 
     * <p><strong>BEAN POPULATION.</strong> 
     * Property values got from the line are copied to the destination bean object,
     * performing any type conversion that is required. If the destination
     * bean does not have a property of the name got from FILE_FORMAT, skips the value.  
     * Implement org.apache.commons.beanutils.Converter and use the <code>register()</code>
     * method of org.apache.commons.beanutils.ConvertUtils to make this method work with
     * custom destination property types.</p>
     * 
     * <p>See 
     * org.apache.commons.beanutils.BeanUtils.copyProperty(Object bean, String name, Object value)
     * for details. 
     */
    public Object readLine(String line, Object object) throws CommonException {
        
        if (object == null) {
            String typeName = properties.getProperty(BEAN_CLASS);
            Class type = BeanHelper.beanClass(typeName);
        
        	if (type != null) {
        		try {
        			object = type.newInstance();
        		} catch (IllegalAccessException iae) {
        			throw new SystemException(iae, "Failed to instantiate bean, access error.");
        		} catch (InstantiationException ie) {
        			throw new SystemException(ie, "Failed to instantiate bean.");
        		}
        	}
    	}
        
        if (object == null)
            throw new SystemException("Failed to instantiate bean.");
        
        String[] lineTokens = CSVDecoder.splitCSVString(line);
        
        if (doValidate) {
            try {
                validate(lineTokens);
            } catch (ValidationError ve) {
                ve.setLine(line);                
                throw ve;
            }
        }
        
        if (lineTokens != null) {
       		BeanHelper.beanPopulate(object, lineTokens, format);
        	
        }
        
        return object;
    }
    
    public Object readLine(String line) throws CommonException {
        return readLine(line, null);
    }
    
    public DecoderState getState() {
        return state;
    }
    
    protected void validate(String[] values) throws CommonException {
        if (validationRules == null) 
            throw new SystemException("Validation rules haven't been defined.");
        
        for (int i = 0, l = values.length, l2 = format.length; i < l && i < l2; i++) {
            String[] tests = (String[])validationRules.get(format[i]);
            if (tests == null) continue;
            
            for (int k = 0, l3 = tests.length; k < l3; k++) {
                String test = test(tests[k]);
                
                TokenValidator validator = TokenValidatorUtils.lookup(test);
                
                if (validator == null)
                    throw new ValidationError(ValidationError.TEST_NOT_FOUND, test);
                
                try {
                    try {
                        validator.validate(test, values[i], parameters(tests[k]));
                    } catch (ValidationError ex) {
                        //try default
                        if (useDefault && (values[i] == null || values[i].length() <= 0 
                                || !useOnEmpty) && defaults != null && defaults.length > i) {                                
                            validator.validate(test, defaults[i], parameters(tests[k]));
                            //default is a valid value, use it
                            values[i] = defaults[i];
                        } else {
                            throw ex;
                        } 
                    }
                } catch (ValidationError ex) {
                    ex.setPropertyName(format[i]);
                    ex.setTokenNumber(i+1); //start with 1                        
                    throw ex;
                }                     
            }
        }
    }
    
    protected String test(String full) {
        if (full.indexOf(':') >= 0)
            return full.trim().substring(0, full.trim().indexOf(':'));
        else return full;
    }
    
    protected Properties parameters(String full) {
        try {
            String params = full.trim().substring(full.trim().indexOf(':')+1);
            
            StringTokenizer st = new StringTokenizer(params, ",; "); 
            Properties p = new Properties();
            while (st.hasMoreTokens()) {
                String param = st.nextToken();
                p.setProperty(param.substring(0, param.indexOf('=')),
                        param.substring(param.indexOf("=")+1));
            }
            return p;
        } catch (Exception ex) {
            return null;
        }        
    }
    
    public class DecoderState {
        
        private int linesRead = 0;
        private int linesValidated = 0;
        private ArrayList invalidLines = null;
        
        protected void init() {
            linesRead = 0;
            linesValidated = 0;
            invalidLines = null;
        }
        
        protected void lineRead() {
            linesRead++;
        }
        protected void lineValidated() {
            linesValidated++;
        }
        protected void invalidLine(ValidationError error) {
            if (invalidLines == null)
                invalidLines = new ArrayList();
            invalidLines.add(error);
        }
        
        public ValidationError[] getInvalidLines() {
            if (invalidLines == null) return new ValidationError[0];
            return (ValidationError[])(invalidLines.toArray(new ValidationError[invalidLines.size()]));
        }
        public int getLinesRead() {
            return linesRead;
        }
        public int getLinesValidated() {
            return linesValidated;
        }
    }
        
}
