 /*
 * Developed by eVelopers Corporation
 *
 * Copyright (c) 1999-2003 eVelopers Corporation. All rights reserved.
 * This software is the confidential and proprietary information of
 * eVelopers Corporation. You shall not disclose such Confidential
 * Information and shall use it only in accordance with the terms of
 * the license agreement you entered into with eVelopers.
 *
 * $Date: 08-Apr-04 17:59:26$
 *
 */
package com.evelopers.common.exception;

import java.io.Serializable;
import com.evelopers.common.util.helper.*;

/**
 * <p>
 * Common abstract exception is a base class for all exceptions that is being thrown
 * by any methods of all depended projects.
 * </p>
 * <p>
 * Conceptually all exceptions are divided on the following groups:
 * <li>Logic Exceptions - logic errors in problem domain;</li>
 * <li>System Exceptions - unrecoverable error inside component because of computer system.
 * Component must throw this exception only in unrecovereable cases from the component point of view;</li>
 * <li>Service Exceptions - used to manage some process as alternative to "goto".
 * Classical usage - jump to the end of some process execution. Component should NEVER throw this
 * exception. <b>TRY DO NOT USE THIS EXCEPTION.</b></li>
 * <li>Uncheked Exceptions - are not a subclasses of CommonException, but are subclasses of
 * RuntimeExceptions. To be thrown due to application bugs - not expected runtime states or cases.</li>
 * </p>
 *
 * @author      kex
 * @version     $Revision: 9$
 */
public abstract class CommonException extends Exception implements Serializable {

  /**
   * Parent exception of this one
   */
  protected Throwable parentException = null;

  /**
   * Create exception with given error message
   *
   * @param message error message
   */
  public CommonException(String message) {
    this(null, message);
  }

  /**
   * Create exception with given parent exception
   *
   * @param e parent exception
   */
  public CommonException(Throwable e) {
    this(e, null);
  }

  /**
   * Create exception with given parent exception and error message
   *
   * @param e parent exception
   * @param message error message
   */
  public CommonException(Throwable e, String message) {
    super(message);
    parentException = e;
  }

  /**
   * Returns parent exception
   *
   * @return parent exception
   */
  public Throwable getParentException() {
    return parentException;
  }

  /**
   * Returns exception stack trace in string representation
   *
   * @return stack trace as string
   */
  public String getTrace() {
    return StringHelper.stackTraceToString(this);
  }

  /**
   * Returns parent exception stack trace in string representation
   *
   * @return parent stack trace as string
   */
  public String getParentTrace() {
    return StringHelper.stackTraceToString(getParentException());
  }

  /**
   * Returns root exception stack trace in string representation
   *
   * @return root stack trace as string
   */
  public String getRootTrace() {
    return StringHelper.stackTraceToString(getRootException());
  }

  /**
   * Returns root exception in this exception's chain
   *
   * @return root exception
   */
  public Throwable getRootException() {
    Throwable parent = getParentException();

    if ( parent != null ) {
      if ( parent instanceof CommonException ) {
        return ((CommonException)parent).getRootException();
      } else {
        return parent;
      }
    } else {
      return this;
    }

  }

  /**
   * Returns chain of messages from current till root in form
   * [ExceptionType]: Message \n (<- [ExceptionType]: Message)*
   *
   * @return chanined message
   */
  public String getChainedMessage() {
    Throwable parent = getParentException();
    String parentChainedMessage;

    if ( parent != null ) {
      if ( parent instanceof CommonException ) {
        parentChainedMessage = ((CommonException)parent).getChainedMessage();
      } else {
        parentChainedMessage = "[" + parent.getClass().getName() + "]: " + parent.getMessage();
      }
    } else {
      parentChainedMessage = null;
    }
    
    return "[" + this.getClass().getName() + "]: " + 
           (getMessage() != null ? getMessage() : "<no message>") + 
           (parentChainedMessage != null ? "\n<- " + parentChainedMessage : "");
  }

  /**
   * Returns chain of exception traces from current till root in form
   * <Current Trace>[\n Parent Trace [\n Parent Trace [...]]]
   * 
   * @return chain of exception traces from current till root
   */
  public String getChainedTrace() {
	 Throwable parent = getParentException();
	 String parentTrace;

	 if ( parent != null ) {
	   if ( parent instanceof CommonException ) {
		 parentTrace = ((CommonException)parent).getChainedTrace();
	   } else {
		 parentTrace = StringHelper.stackTraceToString(parent);
	   }
	 } else {
	   parentTrace = null;
	 }

	 return (parentTrace != null) ? getTrace() + "\n" + parentTrace : getTrace();
    
   }
}
