/*
 * Developed by eVelopers Corporation
 *
 * $Date: 18-Feb-04 13:14:04$
 *
 */
package com.evelopers.common.concurrent;

/**
 * <p>This class is intended to provide the functionality of Read-Write locks.
 * Read-write lock allows multiple threads to acquire a read lock provided
 * no other thread currently has a write lock on the same object. A thread
 * can acquire a write lock if no other thread owns either a read lock or
 * a write lock. Java does not provide a ready-made solution for Read-Write
 * Locks.</p>
 *
 * <p> Methods {@link #getReadLock()} and {@link #getWriteLock()} will return
 * immediately with a lock or else block the calling thread until the lock is
 * available. This implementation of RWLock class will give preference to the
 * waiting writer threads over the readers, through the variable waitingWriters.
 * That means if we have two threads waiting to acquire the lock, the one
 * waiting for write lock will get the lock first.</p>
 *
 * Author: L.V.A
 * @version $Revision: 1$
 */
public class RWLock {

    /**
     * The number of threads holding the read lock; it will be -1 if there is
     *   a write lock and zero if no thread is holding the lock.
     */
    private int givenLocks;

    /** The number of write threads waiting for the lock. */
    private int waitingWriters;

    /** Variable 'mutex' is used to synchronize access to the RWLock object. */
    private Object mutex;


    /**
     * Default constructor
     *
     */
    public RWLock() {

        mutex = new Object();
        givenLocks = 0;
        waitingWriters = 0;
    }


    /**
     * Request for read lock. This method returns immediately if
     * there are no write locks. Otherwise method will wait until
     * all write threads complete their work. It is possible to
     * have several concurrent read locks: read lock is not exclusive.
     *
     * @return <code>true</code> if read lock is set, <code>false</code>
     * if <code>java.lang.InterruptedException</code> is thrown.
     */
    public boolean getReadLock() {

        synchronized (mutex) {
            try {
                while ((givenLocks == -1) || (waitingWriters != 0))
                    mutex.wait();

            } catch (InterruptedException e) {
                return false;
            }

            givenLocks++;
        }
        return true;
    }

    /**
     * Request for write lock. This method returns immediately if
     * there are no read locks. Otherwise method will wait until
     * all read threads complete their work. Write lock is exclusive:
     * only one write lock may be set at the same time.
     *
     * @return <code>true</code> if read lock is set, <code>false</code>
     * if <code>java.lang.InterruptedException</code> is thrown.
     */
    public boolean getWriteLock() {

        synchronized (mutex) {
            waitingWriters++;
            try {
                while (givenLocks != 0)
                    mutex.wait();

            } catch (InterruptedException e) {
                return false;
            } finally {
              waitingWriters--;
            }

            givenLocks = -1;
        }
        return true;
    }


    /**
     * Release the lock. This method works for both read and write locks.
     */
    public void releaseLock() {

        synchronized (mutex) {
            if (givenLocks == 0)
                return;

            if (givenLocks == -1)
                givenLocks = 0;
            else
                givenLocks--;

            mutex.notifyAll();
        }
    }

}



