/*
 * Developed by eVelopers Corporation - May 31, 2003
 *
 * Copyright (c) 1999-2003 eVelopers Corporation. All rights reserved.
 * This software is the confidential and proprietary information of
 * eVelopers Corporation. You shall not disclose such Confidential
 * Information and shall use it only in accordance with the terms of
 * the license agreement you entered into with eVelopers.
 *
 * $Date: 26-Jun-03 11:29:30$
 *
 */
package com.evelopers.common.concurrent;

/**
 * This is a latch. It used by players of two roles: a walker and a guard.
 * Walker calls {@link #pass()} method trying to pass the gate. Guard uses
 * {@link #close()} to close the gate resulting walkes to be crowded at the
 * entrance. The other message a guard can issue is {@link #open()} which
 * opens the gate and all the crowd pass through it. Only {@link #pass()}
 * method is blocking.
 *
 * @author danis
 * @version $Revision: 3$
 */
public class Gate {

    private GateState state;

    /**
     * Constructs an opened gate.
     */
    public Gate() { this(false); }

    /**
     * Constructs the gate with specified initial state.
     *
     * @param opened <code>true</code> if the gate is opened initially;
     * <code>false</code> otherwise.
     */
    public Gate(boolean opened) {
        this.state = new GateState(opened);
    }

    /**
     * Opens the gate resulting all walkers awaited at the entrance
     * passes through.
     */
    public synchronized void open() {
        state.opened = true;
        notifyAll();
    }

    /**
     * Closes the gate resulting all walkers called to {@link #pass()}
     * crowded at the entrance.
     */
    public synchronized void close() { state.opened = false; }

    /**
     * Tries to pass through the gate. The call is blocked if the gate is
     * closed and stays blocked until the gate opens. If the gate is opened
     * at the moment of invocation the call returns immediately.
     *
     * @throws InterruptedException if the thread is interrupted while
     * the gate is awaited for being opened.
     */
    public synchronized void pass() throws InterruptedException {
        while(!opened()) wait();
    }

    /**
     * Returns <code>true</code> if the gate is opened at the invocation
     * moment; <code>false</code> otherwise.
     *
     * @return <code>true</code> if the gate is opened;
     * <code>false</code> otherwise
     */
    public synchronized boolean opened() { return state.opened; }


    private static final class GateState {
        public boolean opened;
        public GateState(boolean opened) { this.opened = opened; }
    }
} ///:~