/*
 * Developed by eVelopers Corporation - May 31, 2003
 *
 * Copyright (c) 1999-2003 eVelopers Corporation. All rights reserved.
 * This software is the confidential and proprietary information of
 * eVelopers Corporation. You shall not disclose such Confidential
 * Information and shall use it only in accordance with the terms of
 * the license agreement you entered into with eVelopers.
 *
 * $Date: 26-Oct-04 17:00:28$
 *
 */
package com.evelopers.common.concurrent;

import java.util.LinkedList;

/**
 * Queue of messages. Put a message in the queue without taking care about its
 * future fate. The queue guaranties the message will be delivered to processor
 * as soon as the previuos message gets prcessed by the processor.
 *
 * @author kex
 * @version $Revision: 6$
 */
public class AbstractQueue {
    private Queue queue;
    private boolean stopped = false;

    /**
     * Creates a queue with specified processor wich is delegated message
     * processing to. Number of message pupm is 1.
     *
     * @param processor object that is to processMessage messages.
     */
    public AbstractQueue(QueueMessageProcessor processor) {
        this(processor, 1);
    }

    /**
     * Creates a queue with specified processor wich is delegated message
     * processing to. Number of message pupm is specified by
     * <code>pumpNo</code>.
     *
     * @param processor object that is to processMessage messages.
     * @param pumpNo    number of message pumps.
     */
    public AbstractQueue(QueueMessageProcessor processor, int pumpNo) {

        if (pumpNo < 1)
            throw new IllegalArgumentException("Number of pumps cannot be less than 1.");

        this.queue = new Queue();
        for (int i = 0; i < pumpNo; i++)
            new EventPump(queue, processor, i).start();
    }

    /**
     * Used to push a message to the queue.
     *
     * @param msg message.
     */
    public void push(Object msg) {
        queue.push(msg);
    }

    /**
     * Stops queue thread
     */
    public void stop() {
        stopped = true;
	
	synchronized (queue) {
	        queue.notify();
	}
    }

    private final class Queue {

        private LinkedList queue = new LinkedList();

        public synchronized void push(Object msg) {
            queue.addLast(msg);
            notify();
        }

        public synchronized Object pull() {
            while (queue.isEmpty() && !stopped) {
                try {
                    wait();
                } catch (InterruptedException ex) {}
            }
            return stopped ? null : queue.removeFirst();
        }
    }

    private final class EventPump extends Thread {
        private Queue queue;
        private QueueMessageProcessor processor;

        public EventPump(Queue queue, QueueMessageProcessor processor, int no) {
            super("Messages queue");

            this.queue = queue;
            this.processor = processor;

            setName(getClass().getName() + "-" + no);
            setDaemon(true);
        }

        public void run() {
            while (!stopped) {
                Object msg = queue.pull();
                if (!stopped) {
                    processor.processMessage(msg);
                }
            }
        }
    }
} ///:~