/*
 * Developed by eVelopers Corporation
 *
 * $Date: 17-Mar-05 17:41:04$
 *
 */

package com.evelopers.common.cache;

import java.util.Date;

import org.apache.commons.collections.map.ReferenceMap;
import com.evelopers.common.concurrent.RWLock;

/**
 * <p>This class implements memory and time sensitive cache with separate read/write locks
 * ({@link com.evelopers.common.RWLock}) to increase performance. Memory sensitivity
 * is implemented using "soft" references. Time sensitivity is implemented providing
*  {@link get(Object, int)} method. </p>
 *
 * @author: L.V.A
 * @version $Revision: 2$
 */
public class TimeSensitiveCache {

    private ReferenceMap cache;
    private RWLock lock;


    /**
     * Default constructor
     */
    public TimeSensitiveCache() {
        cache = new ReferenceMap();
        lock = new RWLock();
    }

    /**
     * Gets object from cache regardless of value expiration
     *
     * @param <code>key</code> the key to be found
     * @return object cached with given key or <code>null</code> if such
     * a key doesn't exist
     */
    public Object get(Object key) {
        return get(key, -1);
    }

    /**
     * <p>Gets object from cache taking into consideration expiration period
     * passed as second parameter. If time stamp stored with requested
     * object is less than current date for more than given expiration period
     * (in seconds) this object will be removed from the cache and
     * <code>null</code> value will be returned.</p>
     *
     * @param <code>key</code> the key to be found
     * @param <code>expirationPeriod</code> value of expiration period in seconds
     * @return object cached with given key or <code>null</code> if such a key
     * doesn't exist or is expired
     */
    public Object get(Object key, long expirationPeriod) {

        if (lock.getReadLock()) {
            try {
                Object[] item = (Object[])cache.get(key);
                if (item == null)
                    return null;
                if (expirationPeriod*1000 > 0) { // check expiration
                    if (new Date().getTime() - ((Date)item[0]).getTime() > expirationPeriod) {
                        cache.remove(key);
                        return null;
                    }
                    return item[1];
                }
                return item[1];
            } finally {
                lock.releaseLock();
            }
        }
        return null;
    }

    /**
     * Cache object
     *
     * @param <code>key</code>
     * @param <code>value</code>
     */
    public void put(Object key, Object value) {

        if (lock.getWriteLock()) {
            try {
                cache.put(key, new Object[]{ new Date(), value });
            } finally {
                lock.releaseLock();
            }
        }
    }

    /**
     * Clears the cache
     *
     */
    public void clear() {

        if (lock.getWriteLock()) {
            try {
                cache.clear();
            } finally {
                lock.releaseLock();
            }
        }
    }
}
