package org.cleversafe.storage.ss.handlers.slices;

import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;

import java.util.UUID;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.TimeUnit;

import org.apache.log4j.BasicConfigurator;
import org.cleversafe.layer.grid.DataSlice;
import org.cleversafe.layer.grid.SliceName;
import org.cleversafe.layer.protocol.BeginTransactionRequest;
import org.cleversafe.layer.protocol.BeginTransactionResponse;
import org.cleversafe.layer.protocol.CommitTransactionRequest;
import org.cleversafe.layer.protocol.CommitTransactionResponse;
import org.cleversafe.layer.protocol.ListBeginRequest;
import org.cleversafe.layer.protocol.ListBeginResponse;
import org.cleversafe.layer.protocol.MultipleWriteRequest;
import org.cleversafe.layer.protocol.MultipleWriteResponse;
import org.cleversafe.server.ClientSession;
import org.cleversafe.storage.ss.SliceServerApplication;
import org.cleversafe.storage.ss.handlers.SliceStorePerformanceBase;
import org.cleversafe.util.BoundedThreadPoolExecutor;
import org.junit.BeforeClass;
import org.junit.Test;

public class ListBeginHandlerPerformanceTest extends SliceStorePerformanceBase
{

	// TODO: Create a test combining ListBegin/ListContinue handlers.
	
	// How long test should be run. This is mutually exclusive with testIterations
	// Only one of them is used.
	private static int testDuration;
	// I/O Request Size in KB
	private static int requestSize;
	// Number of concurrent requests
	private static int numConcurrentRequests;

	// how many operations to be performed. This is mutually exclusive with testDuration.
	// Only one of them is used.
	private static int testIterations;
	private static final int txnID = 1;
	
	private static UUID accountUUID;
	private SliceServerApplication sliceServer = null;

	private static ClientSession session = null;

	public  byte[] data;
	
	@BeforeClass
	public static void initializeTests()
	{
		// Only one of the testIterations or testDuration is used.
		testIterations = Integer.getInteger("test.operations", 1000); //
		testDuration = Integer.getInteger("test.duration", 1); // 1 minutes
		numConcurrentRequests = Integer.getInteger("test.concurrency", 5); //
		requestSize = Integer.getInteger("test.requestsize", 4); // 4K

		// Setup basic logging
		BasicConfigurator.configure();
	}
	
	/*
	 * Writes number of slices equal to the number of testIterations.
	 * 
	 */
	private ClientSession writeSlices()
	{
		
		session = null;
		// Create a session
		if (session == null) {
			try {
				// Create a session with a valid SliceStore
				session = createSession();
			} catch (Exception ex) {
				ex.printStackTrace();
				fail("Unable to start an operational slice store");
			}
		}
		
		// Create a transaction
		try {
			// Create Transaction
			BeginTransactionResponse beginTransactionResponse = (BeginTransactionResponse) this.sliceServer
					.service(new BeginTransactionRequest(txnID), session);
			// Ensure no exception flag was set
			assertTrue("An exception was thrown by the SliceServer",
					beginTransactionResponse.getExceptionFlag() == false);
		} catch (ClassCastException ex) {
			fail("SliceServer returned an inproper Response type");
		}
		/* 
		 * Create data once and write the same stuff over and over again.
		 * 
		 */
		data = new byte[requestSize * 1024];
		randomizeBuffer(data);
		
		int i;
		for (i=1; i< testIterations; i++) {
			writeSlice(Integer.toString(i), data);
		}
		
		// Commit the transaction.
		try {
			// Commit Transaction
			CommitTransactionResponse commitTransactionResponse = (CommitTransactionResponse) this.sliceServer
					.service(new CommitTransactionRequest(txnID), session);
			// Ensure no exception flag was set
			assertTrue("An exception was thrown by the SliceServer",
					commitTransactionResponse.getExceptionFlag() == false);
		} catch (ClassCastException ex) {
			fail("SliceServer returned an inproper Response type");
		}

		return session;
	}
	
	
	@Test
	public void listBeginHandlerPerformance_n()
	{
	
		// Create a session in which a slices have been written. No need to do in threads.
		session = writeSlices();
	
		class WorkItem implements Runnable
		{
			int id;
			public WorkItem(int slice)
			{
				id = slice;
			}
			public void run()
			{
				ListBeginHandlerPerformanceTest test = ListBeginHandlerPerformanceTest.this;;
				try {
					// Create Request and ask server to service it.
					ListBeginResponse response = (ListBeginResponse) test.sliceServer
					.service(new ListBeginRequest(), session);
				
					if (response.getExceptionFlag())
					{
						response.getException().printStackTrace();
						fail("Exception in responce.");
					}
				} catch (ClassCastException ex) {
					ex.printStackTrace();
					fail("SliceServer returned an improper Response type");
				}
				
			}
		}
		
		ExecutorService executor = new BoundedThreadPoolExecutor("ListBegin handler Test", numConcurrentRequests);

		// Create a transaction
		try {
			// Create Transaction
			BeginTransactionResponse beginTransactionResponse = (BeginTransactionResponse) this.sliceServer
					.service(new BeginTransactionRequest(txnID), session);
			// Ensure no exception flag was set
			assertTrue("An exception was thrown by the SliceServer",
					beginTransactionResponse.getExceptionFlag() == false);
		} catch (ClassCastException ex) {
			fail("SliceServer returned an inproper Response type");
		}
		
		// Start Operations.
		int operations;
		long beginTest, endTest;
		beginTest = System.currentTimeMillis();
		System.out.println("Begin test.");
		for (operations = 1; operations < testIterations; operations++) {
			// execute in an available thread.
			executor.execute( new WorkItem(operations) );
			// (new WorkItem(operations)).run();
		}
		      
		executor.shutdown();
		System.out.println("Waiting for threads to stop.");
		boolean finished = false;
		do {
			try {
				finished = executor.awaitTermination( 10, TimeUnit.SECONDS );
			} catch (InterruptedException e){
				e.printStackTrace();
			}
		} while ( !finished );

		endTest = System.currentTimeMillis();
		
		try {
			// Create Transaction
			CommitTransactionResponse commitTransactionResponse = (CommitTransactionResponse) this.sliceServer
					.service(new CommitTransactionRequest(txnID), session);
			// Ensure no exception flag was set
			if (commitTransactionResponse.getExceptionFlag()) {
				commitTransactionResponse.getException().printStackTrace();
			}
			assertTrue("An exception was thrown by the SliceServer",
					commitTransactionResponse.getExceptionFlag() == false);
		} catch (ClassCastException ex) {
			fail("SliceServer returned an inproper Response type");
		}
		
		PerformanceResults results =  new PerformanceResults("ListBeginHandler", testIterations, 0, beginTest, endTest, numConcurrentRequests);
		results.print(true);
		
	}
	
	/**
	 * Sends an Write request to the server and verifies that the slice is
	 * stored
	 * 
	 */
	private void writeSlice(String sliceName, byte[] data) {

		// Create a begin session request
	   MultipleWriteRequest request = new MultipleWriteRequest(new DataSlice(new SliceName(
				sliceName, 0), txnID, data));

		try {
			// Process the request on our custom ClientSession object
		   MultipleWriteResponse response = (MultipleWriteResponse) this.sliceServer.service(
					request, session);

			// Ensure no exception flag was set
			assertTrue("An exception was thrown by the SliceServer", response
					.getExceptionFlag() == false);
		} catch (ClassCastException ex) {
			fail("SliceServer returned an inproper Response type");
		}
	}

}
