//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// Author: Jason
//
// Date: May 15, 2007
//---------------------

package org.cleversafe.storage.ss.handlers.session;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.UUID;

import org.apache.log4j.Logger;
import org.cleversafe.layer.protocol.BeginSessionRequest;
import org.cleversafe.layer.protocol.BeginSessionResponse;
import org.cleversafe.layer.protocol.Request;
import org.cleversafe.layer.protocol.Response;
import org.cleversafe.layer.slicestore.SliceStore;
import org.cleversafe.layer.slicestore.SliceStoreDescriptor;
import org.cleversafe.layer.slicestore.SliceStoreDescriptorManager;
import org.cleversafe.layer.slicestore.SliceStoreManager;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreNotFoundException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreIOException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreLayerException;
import org.cleversafe.layer.slicestore.exceptions.UnsupportedSliceStoreTypeException;
import org.cleversafe.layer.slicestore.exceptions.UnsupportedVaultTypeException;
import org.cleversafe.server.ApplicationType;
import org.cleversafe.server.ClientSession;
import org.cleversafe.server.RequestHandler;
import org.cleversafe.server.exceptions.IllegalSessionStateException;

/**
 * Handler for BeginSessionRequest message
 * 
 * @author Jason Resch
 */
public class BeginSessionHandler implements RequestHandler
{
   private static Logger _logger = Logger.getLogger(BeginSessionHandler.class);

   private final List<ApplicationType> _allowedApplications;
   private final SliceStoreDescriptorManager sliceStoreDescriptorManager;
   private final Map<String, SliceStoreManager> vaultSliceStoreManagerMap;

   /**
    * Constructs object, creates allowed applications access control list and saves reference to the
    * SliceServerConfiguration object
    * 
    * @param config
    */
   public BeginSessionHandler(
         SliceStoreDescriptorManager sliceStoreDescriptorManager,
         Map<String, SliceStoreManager> vaultSliceStoreManagerMap)
   {
      this._allowedApplications = new ArrayList<ApplicationType>();
      this._allowedApplications.add(ApplicationType.TYPE_AUTHENTICATED);
      this.sliceStoreDescriptorManager = sliceStoreDescriptorManager;
      this.vaultSliceStoreManagerMap = vaultSliceStoreManagerMap;
   }

   /**
    * Returns class for the type of message this handler services
    * 
    * @see RequestHandler.getRequestClass()
    */
   public Class<BeginSessionRequest> getRequestClass()
   {
      return BeginSessionRequest.class;
   }

   /**
    * Returns a list of application types that can access this handler
    * 
    * @see RequestHandler.getAllowedApplications()
    */
   public List<ApplicationType> getAllowedApplications()
   {
      return this._allowedApplications;
   }

   /**
    * Loads the SliceStore that was bound to in the VaultBindRequest and calls
    * SliceStore.beginSession()
    * 
    * @see RequestHandler.service(Request request, ClientSession session)
    */
   public Response service(Request request, ClientSession session)
         throws IllegalSessionStateException, SliceStoreNotFoundException,
         UnsupportedVaultTypeException, UnsupportedSliceStoreTypeException, SliceStoreIOException
   {
      BeginSessionResponse response = new BeginSessionResponse(0);

      // Check pre-conditions
      if (session.containsKey(ClientSession.SLICE_STORE))
      {
         throw new IllegalSessionStateException(
               "A slice store has already been created for this session.", request);
      }
      if (!session.containsKey(ClientSession.VAULT_UUID))
      {
         throw new IllegalSessionStateException("Vault UUID has not been set in the session",
               request);
      }

      // Process request

      // Get Vault ID
      UUID vaultID = (UUID) session.get(ClientSession.VAULT_UUID);

      SliceStoreDescriptor descriptor = this.sliceStoreDescriptorManager.loadDescriptor(vaultID);

      SliceStoreManager manager = this.vaultSliceStoreManagerMap.get(descriptor.getVaultType());
      if (manager == null)
      {
         throw new SliceStoreIOException("No SliceStoreManager found for vault type "
               + descriptor.getVaultType());
      }

      SliceStore sliceStore = manager.loadSliceStore(vaultID);

      try
      {
         sliceStore.startSession();
      }
      catch (SliceStoreLayerException ex)
      {
         response.setException(ex);
      }

      // Store slice store in the session
      session.put(ClientSession.SLICE_STORE, sliceStore);

      _logger.debug("Begin slice store session for vault ID " + vaultID);

      return response;
   }

}
