//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// @author: dhendrickson
//
// Date: Nov 11, 2007
//---------------------

package org.cleversafe.storage.ss.configuration;

import java.io.BufferedReader;
import java.io.ByteArrayInputStream;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Map;
import java.util.Properties;

import org.apache.xmlbeans.XmlError;
import org.apache.xmlbeans.XmlException;
import org.apache.xmlbeans.XmlOptions;
import org.cleversafe.authentication.AuthenticationFacilitatorConfiguration;
import org.cleversafe.authentication.exceptions.AuthenticationConfigurationException;
import org.cleversafe.config.ConfigGenerator;
import org.cleversafe.config.ConfigurationFactory;
import org.cleversafe.config.evaluator.Reference;
import org.cleversafe.config.evaluator.XMLEvaluator;
import org.cleversafe.config.exceptions.ConfigurationItemNotDefinedException;
import org.cleversafe.config.exceptions.ConfigurationLoadException;
import org.cleversafe.config.exceptions.IllegalConfigurationContentException;
import org.cleversafe.config.exceptions.ObjectInitializationException;
import org.cleversafe.config.exceptions.ObjectInstantiationException;
import org.cleversafe.config.exceptions.ObjectValidationException;
import org.cleversafe.layer.communication.Acceptor;
import org.cleversafe.layer.slicestore.SliceStoreDescriptorManager;
import org.cleversafe.layer.slicestore.SliceStoreManager;
import org.cleversafe.serialization.ProtocolMessageFactory;
import org.cleversafe.server.exceptions.ServerConfigurationLoadException;
import org.cleversafe.sliceServer.SliceServerDocument;
import org.cleversafe.sliceServer.Authentication.Method;
import org.cleversafe.sliceServer.Authentication.Method.Option;
import org.cleversafe.sliceServer.VaultTypes.VaultType;
import org.cleversafe.storage.ss.SliceServerConfiguration;

public class XMLConfigurationLoader implements ConfigurationLoader
{
   public static String SLICE_SERVER_CONFIGURATION_PROPERTY =
         "org.cleversafe.storage.ss.xml.configuration";

   private SliceServerConfiguration configuration;
   private String xmlContent;

   private void loadConfiguration(String xmlFileName) throws ServerConfigurationLoadException
   {
      this.configuration = new SliceServerConfiguration();

      try
      {
         BufferedReader input = new BufferedReader(new FileReader(xmlFileName));
         StringBuffer buffer = new StringBuffer();

         String line;
         while ((line = input.readLine()) != null)
         {
            buffer.append(line);
         }
         this.xmlContent = buffer.toString();
      }
      catch (FileNotFoundException ex)
      {
         throw new ServerConfigurationLoadException(
               "Cannot read configuration file " + xmlFileName, ex);
      }
      catch (UnsupportedEncodingException ex)
      {
         throw new ServerConfigurationLoadException(
               "Cannot read configuration file " + xmlFileName, ex);
      }
      catch (IOException ex)
      {
         throw new ServerConfigurationLoadException("Error during read of configuration file "
               + xmlFileName, ex);
      }
   }

   public XMLConfigurationLoader() throws ServerConfigurationLoadException
   {
      if (System.getProperty(SLICE_SERVER_CONFIGURATION_PROPERTY) != null)
      {
         loadConfiguration(System.getProperty(SLICE_SERVER_CONFIGURATION_PROPERTY));
      }
      else
      {
         throw new ServerConfigurationLoadException("Property not set: "
               + SLICE_SERVER_CONFIGURATION_PROPERTY);
      }
   }

   public XMLConfigurationLoader(String xmlFileName) throws ServerConfigurationLoadException
   {
      loadConfiguration(xmlFileName);
   }

   public SliceServerConfiguration getConfiguration(Properties properties)
         throws ServerConfigurationLoadException
   {
      // Replace properties if necessary 
      String realContent = this.xmlContent;
      if (properties != null)
      {
         ConfigGenerator generator = new ConfigGenerator(this.xmlContent, properties);
         realContent = generator.generate();
      }

      // Create input stream from XML string
      InputStream xmlInputStream = new ByteArrayInputStream(realContent.getBytes());

      // Validate schema
      SliceServerDocument.SliceServer sliceServerElement;
      try
      {
         SliceServerDocument sliceServerDocument =
               SliceServerDocument.Factory.parse(xmlInputStream);

         XmlOptions validateOptions = new XmlOptions();
         ArrayList<XmlError> errorList = new ArrayList<XmlError>();

         validateOptions.setErrorListener(errorList);

         if (!sliceServerDocument.validate(validateOptions))
         {
            StringBuffer errorBuffer = new StringBuffer();
            errorBuffer.append("Number of errors: " + errorList.size() + ". ");
            int i = 1;
            for (XmlError xmlError : errorList)
            {
               errorBuffer.append("XML Error " + i++ + " - " + xmlError.getMessage()
                     + " at location " + xmlError.getCursorLocation().xmlText() + ";");
            }

            throw new ServerConfigurationLoadException("Could not validate configuration schema: "
                  + errorBuffer.toString());
         }

         sliceServerElement = sliceServerDocument.getSliceServer();
      }
      catch (XmlException ex)
      {
         throw new ServerConfigurationLoadException("Could not load XML configuration", ex);
      }
      catch (IOException ex)
      {
         throw new ServerConfigurationLoadException("Unable to read XML stream", ex);
      }

      // Parse XML
      try
      {
         Reference reference;

         // Get Acceptor
         reference = sliceServerElement.getAcceptor();
         Acceptor acceptor = (Acceptor) XMLEvaluator.parseReference(reference, null).evaluate();
         this.configuration.setAcceptor(acceptor);

         // Get ProtocolMessageFactory
         reference = sliceServerElement.getProtocolMessageFactory();
         ProtocolMessageFactory protocolMessageFactory =
               (ProtocolMessageFactory) XMLEvaluator.parseReference(reference, null).evaluate();
         this.configuration.setProtocolMessageFactory(protocolMessageFactory);

         // Get Authentication configuration
         AuthenticationFacilitatorConfiguration authConfiguration =
               ConfigurationFactory.getBindingsProvider(ConfigurationFactory.XML_CONFIG_TYPE).getDefaultImplementation(
                     AuthenticationFacilitatorConfiguration.class);
         authConfiguration.setDefaultMethod(sliceServerElement.getAuthentication().getDefault());
         Method[] methods = sliceServerElement.getAuthentication().getMethodArray();
         for (int a = 0; a < methods.length; a++)
         {
            String methodName = methods[a].getName();

            Class<?> loginModule = Class.forName(methods[a].getModule());

            String control = methods[a].getControl();

            AuthenticationFacilitatorConfiguration.ModuleControlFlag controlFlag = null;
            if (control.equalsIgnoreCase("OPTIONAL"))
            {
               controlFlag = AuthenticationFacilitatorConfiguration.ModuleControlFlag.OPTIONAL;
            }
            else if (control.equalsIgnoreCase("REQUIRED"))
            {
               controlFlag = AuthenticationFacilitatorConfiguration.ModuleControlFlag.REQUIRED;
            }
            else if (control.equalsIgnoreCase("REQUISITE"))
            {
               controlFlag = AuthenticationFacilitatorConfiguration.ModuleControlFlag.REQUISITE;
            }
            else if (control.equalsIgnoreCase("SUFFICIENT"))
            {
               controlFlag = AuthenticationFacilitatorConfiguration.ModuleControlFlag.SUFFICIENT;
            }

            Map<String, String> options = new HashMap<String, String>();
            Option[] params = methods[a].getOptionArray();
            for (int b = 0; b < params.length; b++)
            {
               options.put(params[b].getName(), params[b].getValue());
            }

            authConfiguration.addAuthenticationMethod(methodName, loginModule, controlFlag, options);
         }
         this.configuration.setAuthenticationFacilitatorConfiguration(authConfiguration);

         // Get SliceStoreDescriptorManager
         reference = sliceServerElement.getSliceStoreDescriptorManager();
         SliceStoreDescriptorManager descriptorManager =
               (SliceStoreDescriptorManager) XMLEvaluator.parseReference(reference, null).evaluate();
         this.configuration.setSliceStoreDescriptorManager(descriptorManager);

         // Get SliceStoreManagers
         VaultType[] vaultTypes = sliceServerElement.getVaultTypes().getVaultTypeArray();
         for (int a = 0; a < vaultTypes.length; a++)
         {
            String vaultType = vaultTypes[a].getName();

            SliceStoreManager sliceStoreManager;
            reference = vaultTypes[a].getSliceStoreManager();
            sliceStoreManager =
                  (SliceStoreManager) XMLEvaluator.parseReference(reference, null).evaluate();

            this.configuration.addSliceStoreManager(vaultType, sliceStoreManager);
         }
      }
      catch (ObjectInitializationException ex)
      {
         throw new ServerConfigurationLoadException("Unable to initialize object", ex);
      }
      catch (IllegalConfigurationContentException ex)
      {
         throw new ServerConfigurationLoadException("Invalid configuration", ex);
      }
      catch (ConfigurationItemNotDefinedException ex)
      {
         throw new ServerConfigurationLoadException("Configuration item not defined", ex);
      }
      catch (ObjectValidationException ex)
      {
         throw new ServerConfigurationLoadException("Unable to validate object", ex);
      }
      catch (ObjectInstantiationException ex)
      {
         throw new ServerConfigurationLoadException("Unable to instantiate object", ex);
      }
      catch (ClassNotFoundException ex)
      {
         throw new ServerConfigurationLoadException("Unable to load specified login module", ex);
      }
      catch (ConfigurationLoadException ex)
      {
         throw new ServerConfigurationLoadException(
               "Could not instantiate AuthenticationFacilitatorConfiguration", ex);
      }
      catch (AuthenticationConfigurationException ex)
      {
         throw new ServerConfigurationLoadException("Authentication configuration failed", ex);
      }

      return this.configuration;
   }
}
