//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// @author: ivolvovski
//
// Date: Jan 18, 2008
//---------------------

package org.cleversafe.storage.ss.cli;

import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.Properties;

import org.apache.log4j.LogManager;
import org.apache.log4j.Logger;
import org.apache.log4j.xml.DOMConfigurator;
import org.apache.xmlbeans.XmlException;
import org.cleversafe.config.ConfigurationFactory;
import org.cleversafe.layer.slicestore.RepairAdvice;
import org.cleversafe.layer.slicestore.RepairAdvice.StoreCondition;
import org.cleversafe.layer.slicestore.exceptions.RepairException;
import org.cleversafe.server.exceptions.ServerConfigurationLoadException;
import org.cleversafe.storage.ss.SliceServerConfiguration;
import org.cleversafe.storage.ss.SliceStoreRepairer;
import org.cleversafe.storage.ss.configuration.ConfigurationLoader;
import org.cleversafe.storage.ss.configuration.XMLConfigurationLoader;
import org.cleversafe.util.JSAPCommandLineParser;
import org.cleversafe.vault.exceptions.VaultException;

import com.martiansoftware.jsap.JSAPException;
import com.martiansoftware.jsap.JSAPResult;

public class SliceStoreRepairCLI
{
   private static Logger _logger = Logger.getLogger(SliceStoreRepairCLI.class);

   private static final String COMMAND_NAME = "slicestor-repair";

   private static final String CHECK_FLAG_OPTION = "CHECK_FLAG";

   private static final String SERVER_CONFIGURATION_OPTION = "SERVER_CONFIGURATION";
   private static final String DESCRIPTOR_OPTION = "DESCRIPTOR";
   private static final String ACL_FILE_OPTION = "ACL_FILE";
   private static final String DAEMON_ID_OPTION = "DAEMON_ID";

   public static final int UNKNOWN_ERROR = -1;
   public static final int OPTION_ERROR = -2;
   public static final int IO_ERROR = -3;
   public static final int CONFIGURATION_ERROR = -4;

   /**
    * @param args
    */
   public static void main(String[] args)
   {
      //      System.out.println(System.getProperty("java.class.path"));
      String loggingConfiguration = System.getProperty(LogManager.DEFAULT_CONFIGURATION_KEY);
      if (loggingConfiguration != null)
      {
         DOMConfigurator.configure(loggingConfiguration);
      }
      else
      {
         System.out.print("Logging configuration is not specified");
      }
      JSAPResult result = null;
      try
      {
         final JSAPCommandLineParser parser =
               new JSAPCommandLineParser(SliceStoreRepairCLI.class, "store_repair_options.jsap",
                     COMMAND_NAME + " [OPTIONS]... ",
                     "Checks for consistency and repairs slice store ");

         result = parser.parse(args);
      }
      catch (final IOException e)
      {
         System.err.println("error: could not parse command line arguments: " + e.getMessage());
         System.exit(IO_ERROR);
      }
      catch (final JSAPException e)
      {
         System.err.println("error: could not parse command line arguments: " + e.getMessage());
         System.exit(OPTION_ERROR);
      }

      if (result == null)
      {
         // Parser handled argument exception
         System.exit(OPTION_ERROR);
      }
      try
      {
         final SliceStoreRepairer repairer = buildRepairer(result);

         final RepairAdvice advice = repairer.verify();
         System.out.println("Advice: ");
         System.out.println(advice.toString());
         if (!repairer.isCheckOnly())
         {
            if (!advice.getStoreCondition().equals(StoreCondition.GOOD))
            {
               System.out.println("Starting repair...");
               repairer.repair(advice);
               System.out.println("Repair complete.");
            }
            else
            {
               System.out.println("Store is in good condition. No repair required.");
            }
         }
         else
         {
            System.out.println("No action, check was requested");
         }
         System.exit(advice.getStoreCondition().getValue());
      }
      catch (final RepairException ex)
      {
         System.out.println("Failed to perform repair" + ex.getMessage());
         System.exit(UNKNOWN_ERROR);
      }
   }

   /**
    * Builds a repairer object from command arguments Called only by main()
    * 
    * @param result
    * @return
    */
   private static SliceStoreRepairer buildRepairer(JSAPResult result)
   {
      boolean checkOnlyFlag = false;

      if (!result.userSpecified(CHECK_FLAG_OPTION))
      {
         checkOnlyFlag = false;
      }
      else
      {
         checkOnlyFlag = result.getBoolean(CHECK_FLAG_OPTION);
      }

      final String bindingsConfiguration =
            System.getProperty(ConfigurationFactory.XML_BINDINGS_CONFIG_PROPERTY);
      if (bindingsConfiguration == null)
      {
         System.out.println("Bindings are not specified");
         System.exit(OPTION_ERROR);
      }
      // Load server configuration, JSAP guarantees that the value is provided
      try
      {
         final String sliceServerConfigurationPath =
               result.getFile(SERVER_CONFIGURATION_OPTION).getPath();
         final ConfigurationLoader configLoader =
               new XMLConfigurationLoader(sliceServerConfigurationPath);
         Properties daemonProps = null;
         if (result.userSpecified(DAEMON_ID_OPTION))
         {
            daemonProps = new Properties();
            final int daemonId = result.getInt(DAEMON_ID_OPTION);
            daemonProps.put("DAEMON.ID", new Integer(daemonId).toString());
         }
         final SliceServerConfiguration serverConfig = configLoader.getConfiguration(daemonProps);
         final InputStream aclDefinitionStream =
               new FileInputStream(result.getFile(ACL_FILE_OPTION));

         final InputStream vaultDefinitionStream =
               new FileInputStream(result.getFile(DESCRIPTOR_OPTION));

         return new SliceStoreRepairer(checkOnlyFlag, serverConfig, vaultDefinitionStream,
               aclDefinitionStream);
      }
      catch (final ServerConfigurationLoadException e)
      {
         System.out.println("Invalid server configuration: " + e.getMessage());
         System.exit(OPTION_ERROR);
      }
      catch (final IOException e)
      {
         System.out.println("Failed to open file: " + e.getMessage());
         System.exit(IO_ERROR);
      }
      catch (final VaultException e)
      {
         System.out.println("Failed to create vault object file: " + e.getMessage());
         System.exit(CONFIGURATION_ERROR);
      }
      catch (final XmlException e)
      {
         System.out.println("Failed to open file: " + e.getMessage());
         System.exit(CONFIGURATION_ERROR);
      }
      throw new IllegalStateException("Should not be here!!");
   }

}
