//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// @author: ivolvovski
//
// Date: Jan 26, 2008
//---------------------

package org.cleversafe.storage.ss;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.List;

import org.apache.log4j.Logger;
import org.apache.xmlbeans.XmlException;
import org.cleversafe.config.ExecutionContext;
import org.cleversafe.layer.slicestore.RepairAdvice;
import org.cleversafe.layer.slicestore.SliceStoreDescriptor;
import org.cleversafe.layer.slicestore.SliceStoreManager;
import org.cleversafe.layer.slicestore.exceptions.RepairException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreExistsException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreIOException;
import org.cleversafe.vault.Vault;
import org.cleversafe.vault.VaultACL;
import org.cleversafe.vault.VaultACLFactory;
import org.cleversafe.vault.VaultDescriptor;
import org.cleversafe.vault.XMLVaultDescriptor;
import org.cleversafe.vault.exceptions.VaultException;

public class SliceStoreRepairer
{
   private final SliceServerConfiguration sliceServerConfiguration;

   private SliceStoreManager manager = null;

   private Vault vault = null;
   private VaultACL acl = null;

   private byte[] vaultDescriptorContent = null;

   private boolean checkOnlyFlag = false;

   private static Logger _logger = Logger.getLogger(SliceStoreRepairer.class);

   /**
    * Creates a repair object Streams passed will be closed by repairer after construction
    * 
    * @param checkOnlyFlag
    *           if false don't repair just give a repair advice
    * @param serverConfiguration
    *           server configuration file path
    * @param vaultDefinitionStream
    *           vault definition stream
    * @param aclDefinitionStream
    *           ACL definition stream
    * @throws VaultException
    *            Failed to instantiate VVault object from a given stream
    * @throws IOException
    *            IO error opening server configuration
    * @throws XmlException
    *            XML error encountered parsing vault definition stream
    */
   public SliceStoreRepairer(
         boolean checkOnlyFlag,
         SliceServerConfiguration serverConfiguration,
         InputStream vaultDefinitionStream,
         InputStream aclDefinitionStream) throws VaultException, IOException, XmlException
   {
      assert (serverConfiguration != null);
      assert (vaultDefinitionStream != null);
      assert (aclDefinitionStream != null);

      this.checkOnlyFlag = checkOnlyFlag;
      this.sliceServerConfiguration = serverConfiguration;

      try
      {

         final VaultACLFactory aclFactory = new VaultACLFactory();

         this.acl = aclFactory.getInstance(aclDefinitionStream);

         final class DummyACLEntry extends VaultACL.Entry
         {
            public DummyACLEntry()
            {
               super(null, null, null);
            }
         }

         final ExecutionContext ctx = new ExecutionContext();
         ctx.add(VaultDescriptor.ACL_CTX_STRING, this.acl);
         ctx.add(VaultDescriptor.ACL_ENTRY_CTX_STRING, new DummyACLEntry());
         //ctx.add(VaultDescriptor.CREDENTIALS_CTX_STRING, credentials);

         this.vaultDescriptorContent = accumulateBytes(vaultDefinitionStream);

         ByteArrayInputStream newVaultStream =
               new ByteArrayInputStream(this.vaultDescriptorContent);
         // Load Vault definition
         this.vault = new XMLVaultDescriptor(newVaultStream, ctx).createNoSliceStoreVaultObject();
         this.vault.setVaultIdentifier(this.acl.getVaultIdentifier());
         this.manager =
               this.sliceServerConfiguration.getVaultSliceStoreManagerMap().get(
                     this.vault.getType());
      }
      catch (IOException e)
      {
         _logger.error("Error creating vault ", e);
         throw e;
      }
      catch (XmlException e)
      {
         _logger.error("Error creating vault ", e);
         throw e;
      }
      catch (VaultException e)
      {
         _logger.error("Error creating vault ", e);
         throw e;
      }
      finally
      {
         vaultDefinitionStream.close();
         aclDefinitionStream.close();
      }
   }

   /**
    * Perform a prescribed repair procedure Returns verification advice
    */
   public RepairAdvice perform() throws RepairException
   {
      final RepairAdvice advice = verify();
      _logger.info("Advice:" + advice.toString());
      if (!isCheckOnly())
      {
         repair(advice);
      }
      else
      {
         _logger.info("No action, check was requested");
      }
      return advice;
   }

   /**
    * Check only mode
    * 
    * @return
    */
   public boolean isCheckOnly()
   {
      return this.checkOnlyFlag;
   }

   /**
    * Verify what repairs are needed and returns an advice
    * 
    * @return
    * @throws RepairException
    */
   public RepairAdvice verify() throws RepairException
   {
      RepairAdvice advice = this.manager.checkIntegrity(this.vault, this.acl);
      // Check descriptor file
      try
      {
         this.sliceServerConfiguration.getSliceStoreDescriptorManager().loadDescriptor(
               this.vault.getVaultIdentifier());
      }
      catch (Exception ex)
      {
         advice.setMissingDescriptorDefinition();
         advice.setProbableCondition(RepairAdvice.StoreCondition.PARTIAL_REPAIR);
      }
      return advice;
   }

   /**
    * Perform repair according to an advice
    * 
    * @param advice
    * @throws RepairException
    */
   public void repair(final RepairAdvice advice) throws RepairException
   {
      try
      {
         this.manager.repair(advice, this.vault, this.vaultDescriptorContent, this.acl);
         if (advice.getMissingDescriptorDefinition())
         {
            try
            {
               _logger.info("Restoring slice store descriptor");
               this.sliceServerConfiguration.getSliceStoreDescriptorManager().saveDescriptor(
                     this.vault.getVaultIdentifier(),
                     new SliceStoreDescriptor(this.vault.getType()));
            }
            catch (SliceStoreIOException e)
            {
               _logger.error(e.getMessage());
               throw new RepairException("Could not repair!", e);
            }
         }
      }
      catch (final SliceStoreExistsException e)
      {
         final String errorMesasage =
               "Errror during repair, store already exists for vault:"
                     + this.vault.getVaultIdentifier();
         _logger.error(errorMesasage);
         throw new RepairException("Could not repair!", e);
      }
   }

   private byte[] accumulateBytes(InputStream stream) throws IOException
   {
      // Accumulate stream content in memory, we expect
      // TODO: Don't use byte arrays in this interfaces, use stream instead
      final int readSize = 1024;
      final List<byte[]> sequence = new ArrayList<byte[]>();
      while (true)
      {
         byte[] buffer = new byte[readSize];
         final int bytesRead = stream.read(buffer);
         if (bytesRead == -1)
         {
            break;
         }
         else
         {
            // correct size if necessary
            if (bytesRead != readSize)
            {
               byte[] truncBuffer = new byte[bytesRead];
               System.arraycopy(buffer, 0, truncBuffer, 0, bytesRead);
               buffer = truncBuffer;
            }
            sequence.add(buffer);
         }
      }
      // Calculate length
      int bufferLength = 0;
      for (final byte[] buffer : sequence)
      {
         bufferLength += buffer.length;
      }
      final byte[] streamContent = new byte[bufferLength];
      int curPos = 0;
      for (final byte[] buffer : sequence)
      {
         System.arraycopy(buffer, 0, streamContent, curPos, buffer.length);
         curPos += buffer.length;
      }
      return streamContent;
   }

}
