//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// @author: jquigley
//
// Date: Sep 17, 2007
//---------------------

package org.cleversafe.storage.ss;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.util.Properties;

import org.apache.log4j.Logger;
import org.apache.mina.common.ByteBuffer;
import org.cleversafe.config.ConfigurationFactory;
import org.cleversafe.config.exceptions.ConfigurationException;
import org.cleversafe.exceptions.InitializationException;
import org.cleversafe.exceptions.NotImplementedException;
import org.cleversafe.layer.communication.exceptions.CommunicationIOException;
import org.cleversafe.layer.communication.exceptions.CommunicationInterruptedException;
import org.cleversafe.layer.communication.exceptions.CommunicationResponseException;
import org.cleversafe.layer.communication.exceptions.CommunicationTransmissionException;
import org.cleversafe.layer.communication.exceptions.NotConnectedException;
import org.cleversafe.layer.communication.memory.MemoryConnector;
import org.cleversafe.layer.protocol.GridProtocolHeaderCodec;
import org.cleversafe.layer.protocol.ProtocolMessage;
import org.cleversafe.layer.protocol.Request;
import org.cleversafe.layer.protocol.Response;
import org.cleversafe.layer.protocol.SequencedProtocolMessage;
import org.cleversafe.layer.protocol.VersionedProtocolMessage;
import org.cleversafe.layer.protocol.exceptions.HeaderFormatException;
import org.cleversafe.layer.protocol.exceptions.OperationNotRegisteredException;
import org.cleversafe.layer.protocol.exceptions.ProtocolDeserializationException;
import org.cleversafe.layer.protocol.exceptions.ProtocolSerializationException;
import org.cleversafe.serialization.GridProtocolMessageFactory;
import org.cleversafe.serialization.ProtocolMessageFactory;
import org.cleversafe.server.ClientSession;
import org.cleversafe.server.RequestHandler;
import org.cleversafe.server.ServerApplication;
import org.cleversafe.server.exceptions.ServerConfigurationLoadException;
import org.cleversafe.storage.ss.configuration.XMLConfigurationLoader;

/**
 * Used to simulate network communication without going through a network stack all transfers of
 * messages are conducted in-memory.
 * 
 * @author John Quigley
 */
public class SliceServerMemoryConnector extends MemoryConnector
{
   private static Logger _logger = Logger.getLogger(SliceServerMemoryConnector.class);
   private static final String JAAS_PROFILE_OPTION = "jaas_profile";

   private ServerApplication _serverApplication = null;
   private boolean doSerialization = true;

   private ProtocolMessageFactory _protocolMessageFactory = null;

   private final Properties properties = new Properties();

   /**
    * Constructor used by framework
    */
   public SliceServerMemoryConnector()
   {
   }

   /**
    * Constructor used programmatically
    * 
    * @param config
    */
   public SliceServerMemoryConnector(
         SliceServerConfiguration config,
         ProtocolMessageFactory pmFactory)
   {
      this(config, true, pmFactory);
   }

   /**
    * Constructor used programmatically
    * 
    * @param config
    * @param doSerialization
    */
   public SliceServerMemoryConnector(
         SliceServerConfiguration config,
         boolean doSerialization,
         ProtocolMessageFactory pmFactory)
   {
      this.doSerialization = true;
      this._serverApplication = new SliceServerApplication(config);
      this._protocolMessageFactory = pmFactory;
      super.initialize();
   }

   /**
    * @return the doSerialization
    */
   public boolean isDoSerialization()
   {
      return this.doSerialization;
   }

   /**
    * @param doSerialization
    *           the doSerialization to set
    */
   public void setDoSerialization(boolean doSerialization)
   {
      this.doSerialization = doSerialization;
   }

   public void setID(String value)
   {
      this.properties.put("DAEMON.ID", value);
   }

   @Override
   public void initialize()
   {
      try
      {
         this._protocolMessageFactory =
               ConfigurationFactory.getBindingsProvider(ConfigurationFactory.XML_CONFIG_TYPE).getDefaultImplementation(
                     ProtocolMessageFactory.class);
      }
      catch (ConfigurationException e)
      {
         throw new InitializationException(
               "Could not instantiate ProtocolMessageFactory from configuration", e);
      }

      try
      {
         XMLConfigurationLoader loader = new XMLConfigurationLoader();
         this._serverApplication =
               new SliceServerApplication(loader.getConfiguration(this.properties));
         super.initialize();
      }
      catch (ServerConfigurationLoadException e)
      {
         throw new InitializationException("unable to initialize SliceServer configuration", e);
      }
   }

   @Override
   public Response exchange(Request request) throws CommunicationIOException,
         NotConnectedException, CommunicationInterruptedException, CommunicationResponseException,
         CommunicationTransmissionException
   {
      return exchangeImpl(request, 0);
   }

   @Override
   public Response exchange(Request request, int timeout) throws CommunicationIOException,
         NotConnectedException, CommunicationInterruptedException, CommunicationResponseException,
         CommunicationTransmissionException
   {
      return exchangeImpl(request, timeout);
   }

   private Response exchangeImpl(Request request, int timeout) throws CommunicationIOException,
         NotConnectedException, CommunicationInterruptedException, CommunicationResponseException,
         CommunicationTransmissionException
   {

      // Simulate latency in processing the request
      if (System.getProperty("test.latency") != null)
      {
         int latency = Integer.getInteger("test.latency");
         if (latency > 0)
         {
            try
            {
               Thread.sleep(latency);
            }
            catch (InterruptedException e)
            {
               // ignore
            }
         }
      }

      if (!this.doSerialization)
      {
         return service(request, this._session);
      }
      else
      {
         try
         {
            // Client request serialization
            ByteBuffer in = _encode(request);
            // Server request de-serialization
            Request serverRequest = (Request) _decode(in);

            Response serverResponse = service(serverRequest, this._session);

            // Server response serialization
            ByteBuffer out = _encode(serverResponse);
            // Client response de-serialization
            Response clientResponse = (Response) _decode(out);

            return clientResponse;
         }

         catch (HeaderFormatException e)
         {
            throw new CommunicationTransmissionException("malformed message ", e);
         }
         catch (OperationNotRegisteredException e)
         {
            // TODO Auto-generated catch block
            e.printStackTrace();
         }
         catch (ProtocolDeserializationException e)
         {
            throw new CommunicationTransmissionException("Operation not registsred", e);
         }
         catch (ProtocolSerializationException e)
         {
            throw new CommunicationTransmissionException("Serialization error", e);
         }
         return null; // TODO: get rid of it
      }
   }

   @Override
   public void addHandler(RequestHandler handler)
   {
      this._serverApplication.addHandler(handler);

   }

   @Override
   public void removeHandler(RequestHandler handler)
   {
      this._serverApplication.removeHandler(handler);

   }

   @Override
   public Response service(Request request, ClientSession session)
   {
      return this._serverApplication.service(request, session);
   }

   private ProtocolMessage _decode(ByteBuffer in) throws HeaderFormatException,
         OperationNotRegisteredException, ProtocolDeserializationException
   {
      byte headerBuffer[] = new byte[GridProtocolHeaderCodec.LENGTH];

      in.get(headerBuffer);

      GridProtocolHeaderCodec header = new GridProtocolHeaderCodec(headerBuffer);

      // TODO: Look into eliminating this memory allocation
      byte payload[] = new byte[header.getPayloadLength()];

      in.get(payload);

      ProtocolMessage message =
            this._protocolMessageFactory.deserialize(header.getOperationCode(),
                  new DataInputStream(new ByteArrayInputStream(payload)));

      if (message instanceof SequencedProtocolMessage)
      {
         ((SequencedProtocolMessage) message).setSequenceNumber(header.getMessageID());
      }
      if (message instanceof VersionedProtocolMessage)
      {
         ((VersionedProtocolMessage) message).setProtocolVersion(header.getProtocolVersion());
         ((VersionedProtocolMessage) message).setProtocolIdentifier(header.getProtocolID());
      }

      return message;
   }

   byte[] serialize(ProtocolMessage message) throws ProtocolSerializationException
   {
      ByteArrayOutputStream bytes = new ByteArrayOutputStream();
      DataOutputStream out = new DataOutputStream(bytes);
      this._protocolMessageFactory.serialize(out, message);
      return bytes.toByteArray();
   }

   private ByteBuffer _encode(Object message) throws ProtocolSerializationException
   {
      SequencedProtocolMessage pm = (SequencedProtocolMessage) message;

      int messageLength = this._protocolMessageFactory.getMaxSerializedSize(pm);

      // Allocate single byte array for serializing the header and message
      ByteArrayOutputStream bos =
            new ByteArrayOutputStream(GridProtocolHeaderCodec.LENGTH + messageLength);
      DataOutputStream outputStream = new DataOutputStream(bos);

      // Get protocol version number
      short protocolVersion = -1;
      if (pm instanceof VersionedProtocolMessage)
      {
         ((VersionedProtocolMessage) pm).setProtocolVersion(GridProtocolMessageFactory.getGridProtocolVersion());
         protocolVersion = ((VersionedProtocolMessage) pm).getProtocolVersion();
      }
      else
      {
         protocolVersion = GridProtocolHeaderCodec.PROTOCOL_VERSION;
      }

      // Serialize the Protocol Header
      GridProtocolHeaderCodec header =
            new GridProtocolHeaderCodec(protocolVersion, pm.getSequenceNumber(),
                  pm.getOperation().getOperationCode(), 0 /* This is updated later */);
      header.serialize(outputStream);

      this._protocolMessageFactory.serialize(outputStream, pm);

      byte[] bytes = bos.toByteArray();

      // need to set the length on the header
      header.fixLength(bytes);

      return ByteBuffer.wrap(bytes);
   }

   @Override
   public String getIdentification()
   {
      return "In-Memory connector : " + getRemoteAddress();
   }

   // TODO: make this paramaterizable
   public ProtocolMessageFactory getProtocolMessageFactory()
   {
      return this._protocolMessageFactory;
   }

   public void setProtocolMessageFactory(ProtocolMessageFactory protocolMessageFactory)
   {
      this._protocolMessageFactory = protocolMessageFactory;
   }

   public int getNumOutstandingExchanges()
   {
      throw new NotImplementedException("not implemented");
   }
}
