//Cleversafe open-source code header - Version 1.1 - December 1, 2006
//
//Cleversafe Dispersed Storage(TM) is software for secure, private and
//reliable storage of the world's data using information dispersal.
//
//Copyright (C) 2005-2007 Cleversafe, Inc.
//
//This program is free software; you can redistribute it and/or
//modify it under the terms of the GNU General Public License
//as published by the Free Software Foundation; either version 2
//of the License, or (at your option) any later version.
//
//This program is distributed in the hope that it will be useful,
//but WITHOUT ANY WARRANTY; without even the implied warranty of
//MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//GNU General Public License for more details.
//
//You should have received a copy of the GNU General Public License
//along with this program; if not, write to the Free Software
//Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
//USA.
//
//Contact Information: 
// Cleversafe, 10 W. 35th Street, 16th Floor #84,
// Chicago IL 60616
// email: licensing@cleversafe.org
//
//END-OF-HEADER
//-----------------------
//@author: John Quigley <jquigley@cleversafe.com>
//@date: January 1, 2008
//---------------------

package org.jscsi.scsi.tasks;

import java.nio.ByteBuffer;
import java.util.concurrent.atomic.AtomicInteger;

import org.apache.log4j.Logger;
import org.jscsi.scsi.tasks.Status;
import org.jscsi.scsi.protocol.Command;
import org.jscsi.scsi.protocol.cdb.CDB;
import org.jscsi.scsi.protocol.cdb.ParameterCDB;
import org.jscsi.scsi.protocol.cdb.TransferCDB;
import org.jscsi.scsi.protocol.inquiry.InquiryDataRegistry;
import org.jscsi.scsi.protocol.mode.ModePageRegistry;
import org.jscsi.scsi.protocol.sense.exceptions.SenseException;
import org.jscsi.scsi.protocol.sense.exceptions.SynchronousDataTransferErrorException;
import org.jscsi.scsi.transport.TargetTransportPort;

//TODO: Describe class or interface
public abstract class AbstractTask implements Task
{
   private static Logger _logger = Logger.getLogger(AbstractTask.class);

   private TargetTransportPort targetTransportPort;
   private Command command;
   private ModePageRegistry modePageRegistry;
   private InquiryDataRegistry inquiryDataRegistry;
   private String name = "DefaultTaskName";

   /**
    * Status indicates if the task is in progress, aborted, or completed. If the task is not
    * in progress, abort(), writeResponse(), and other methods cannot proceed (only one of these
    * can be called once).
    * 
    * This variable protects abort() and writeResponse() from being called at the same time while
    * still allowing the actual status of the task to be communicated at the end.
    */
   private volatile AtomicInteger status = new AtomicInteger(IN_PROGRESS_INT);

   private final static int IN_PROGRESS_INT = 0;
   private final static int ABORTED_INT = 1;
   private final static int COMPLETED_INT = 2;

   /////////////////////////////////////////////////////////////////////////////
   // abstract methods

   protected abstract void execute() throws InterruptedException, SenseException;

   /////////////////////////////////////////////////////////////////////////////
   // constructors

   protected AbstractTask()
   {
   }

   protected AbstractTask(String name)
   {
      this.name = name;
   }

   protected AbstractTask(
         String name,
         TargetTransportPort targetPort,
         Command command,
         ModePageRegistry modePageRegistry,
         InquiryDataRegistry inquiryDataRegistry)
   {
      this.name = name;
      this.targetTransportPort = targetPort;
      this.command = command;
      this.modePageRegistry = modePageRegistry;
      this.inquiryDataRegistry = inquiryDataRegistry;
   }

   /////////////////////////////////////////////////////////////////////////////
   // operations

   protected final Task load(
         TargetTransportPort targetPort,
         Command command,
         ModePageRegistry modePageRegistry,
         InquiryDataRegistry inquiryDataRegistry)
   {
      this.command = command;
      this.targetTransportPort = targetPort;
      this.modePageRegistry = modePageRegistry;
      this.inquiryDataRegistry = inquiryDataRegistry;
      return this;
   }

   private boolean markCompleted()
   {
      synchronized (status)
      {
         status.notify();
         return this.status.compareAndSet(IN_PROGRESS_INT, COMPLETED_INT);
      }
   }

   private boolean markAborted()
   {
      synchronized (status)
      {
         status.notify();
         return this.status.compareAndSet(IN_PROGRESS_INT, ABORTED_INT);
      }
   }

   public TaskStatus waitForFinish()
   {
      synchronized (this.status)
      {
         while (this.status.get() == IN_PROGRESS_INT)
         {
            try
            {
               this.status.wait();
            }
            catch (InterruptedException e)
            {
            }
         }
      }

      switch (this.status.get())
      {
         case ABORTED_INT :
            return TaskStatus.ABORTED;
         case COMPLETED_INT :
            return TaskStatus.COMPLETED;
         default :
            throw new RuntimeException("Unexpected status integer");
      }
   }

   public final boolean isAborted()
   {
      return this.status.get() == ABORTED_INT;
   }

   public final boolean abort()
   {
      _logger.debug("Aborting task: " + this.getCommand());

      if (this.markAborted())
      {
         // If abort is false the task can be aborted because it is neither
         // already aborted nor in the writeResponse() phase. Abort is now set as true.

         this.targetTransportPort.terminateDataTransfer(this.command.getNexus(),
               this.command.getCommandReferenceNumber());

         return true;
      }
      else
      {
         // If abort is true the task is already aborted or is in the writeResponse() phase. We
         // can no longer abort and the abort value remains set as true.
         return false;
      }
   }

   public final void run()
   {
      try
      {
         this.execute();
      }
      catch (SenseException e)
      {
         _logger.debug("sense exception caught handling command: " + command);
         // Write response with a CHECK CONDITION status.
         if (this.markCompleted())
         {
            // We are irreversibly committed to responding now, so we set abort to true to
            // avoid abort() calls from having any further effect.
            this.targetTransportPort.writeResponse(this.command.getNexus(),
                  this.command.getCommandReferenceNumber(), Status.CHECK_CONDITION,
                  ByteBuffer.wrap(e.encode()));
         }
      }
      catch (InterruptedException e)
      {
         _logger.info("Task " + name + " was aborted.");
         // Task was aborted, don't do anything
      }
      catch (Exception e)
      {
         _logger.error("Task " + name + " encountered an exception while executing", e);
      }

      if (this.markCompleted())
      {
         _logger.error("INTERNAL ERROR: task failed to call writeResponse on success. DON'T PANIC EVERYTHING IS COOL, DATA IS GOOD");
      }
   }

   protected final boolean readData(ByteBuffer output) throws InterruptedException,
         SynchronousDataTransferErrorException
   {
      if (this.status.get() != IN_PROGRESS_INT)
      {
         return false;
      }
      return this.targetTransportPort.readData(this.command.getNexus(),
            this.command.getCommandReferenceNumber(), output);
   }

   protected final boolean writeData(ByteBuffer input) throws InterruptedException,
         SynchronousDataTransferErrorException
   {
      if (this.status.get() != IN_PROGRESS_INT)
      {
         return false;
      }

      return this.targetTransportPort.writeData(this.command.getNexus(),
            this.command.getCommandReferenceNumber(), input);
   }

   protected final boolean writeData(byte[] input) throws InterruptedException,
         SynchronousDataTransferErrorException
   {
      // check how much data may be returned
      CDB cdb = this.command.getCommandDescriptorBlock();
      long transferLength = 0;
      if (cdb instanceof TransferCDB)
      {
         transferLength = ((TransferCDB) cdb).getTransferLength();
      }
      else if (cdb instanceof ParameterCDB)
      {
         transferLength = ((ParameterCDB) cdb).getAllocationLength();
      }
      // If the CDB is not a transfer or parameter CDB no data should be transfered

      // We allocate a byte buffer of transfer length and write either all input data
      // or up to the transfer length in input data.
      ByteBuffer data = ByteBuffer.allocate((int) Math.min(transferLength, input.length));
      data.put(input, 0, (int) Math.min(transferLength, input.length));
      data.rewind();

      return writeData(data);
   }

   protected final void writeResponse(Status status, ByteBuffer senseData)
   {
      if (this.markCompleted())
      {
         // If abort is false the task can enter the writeResponse() phase. Abort is now
         // set as true to indicate that abort() can no longer succeed.
         this.getTargetTransportPort().writeResponse(command.getNexus(),
               command.getCommandReferenceNumber(), status, senseData);
      }
      else
      {
         // If abort is true the task has been aborted and no data shall be written to
         // the target transport port. Abort remains set as true.
         _logger.debug("task was aborted, no response to be written: " + this);
         return;
      }
   }
   
   public TaskStatus getStatus()
   {
	   switch (this.status.get())
	   {
		   case ABORTED_INT :
			   return TaskStatus.ABORTED;
		   case COMPLETED_INT :
			   return TaskStatus.COMPLETED;
		   case IN_PROGRESS_INT :
			   return TaskStatus.IN_PROGRESS;
		   default :
			   throw new RuntimeException("Unexpected status integer");
	   }
   }

   /////////////////////////////////////////////////////////////////////////////
   // getters/setters

   public final Command getCommand()
   {
      return this.command;
   }

   public final TargetTransportPort getTargetTransportPort()
   {
      return this.targetTransportPort;
   }

   protected final InquiryDataRegistry getInquiryDataRegistry()
   {
      return this.inquiryDataRegistry;
   }

   protected final ModePageRegistry getModePageRegistry()
   {
      return this.modePageRegistry;
   }

   public final String getName()
   {
      return this.name;
   }

   public void setName(final String name)
   {
      this.name = name;
   }

   @Override
   public String toString()
   {
      return String.format("<SCSITask name: %s command: %s", this.getName(), this.command);
   }
}
