//Cleversafe open-source code header - Version 1.1 - December 1, 2006
//
//Cleversafe Dispersed Storage(TM) is software for secure, private and
//reliable storage of the world's data using information dispersal.
//
//Copyright (C) 2005-2007 Cleversafe, Inc.
//
//This program is free software; you can redistribute it and/or
//modify it under the terms of the GNU General Public License
//as published by the Free Software Foundation; either version 2
//of the License, or (at your option) any later version.
//
//This program is distributed in the hope that it will be useful,
//but WITHOUT ANY WARRANTY; without even the implied warranty of
//MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//GNU General Public License for more details.
//
//You should have received a copy of the GNU General Public License
//along with this program; if not, write to the Free Software
//Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
//USA.
//
//Contact Information: 
// Cleversafe, 10 W. 35th Street, 16th Floor #84,
// Chicago IL 60616
// email: licensing@cleversafe.org
//
//END-OF-HEADER
//-----------------------
//@author: John Quigley <jquigley@cleversafe.com>
//@date: January 1, 2008
//---------------------

package org.jscsi.scsi.protocol.inquiry.vpd.deviceidentification;

import java.io.ByteArrayOutputStream;
import java.io.DataOutputStream;
import java.io.IOException;

public class IdentificationDescriptor
{
   private int protocolIdentifier;
   private CodeSet codeSet;
   private boolean PIV;
   private Association association;
   DeviceIdentifier identifier;

   public IdentificationDescriptor()
   {
   }

   public IdentificationDescriptor(
         int protocolIdentifier,
         CodeSet codeSet,
         boolean PIV,
         Association association,
         DeviceIdentifier identifier)
   {
      this.protocolIdentifier = protocolIdentifier;
      this.codeSet = codeSet;
      this.PIV = PIV;
      this.association = association;
      this.identifier = identifier;
   }

   public byte[] encode() throws IOException
   {
      ByteArrayOutputStream baos = new ByteArrayOutputStream(this.getIdentifierLength() + 4);
      DataOutputStream out = new DataOutputStream(baos);

      try
      {
         // byte 0
    	 // SPC-3 sec. 7.6.3.1 says that if PIV is false then the protocol identifier contents
    	 // are reserved (hence they should be zero by convention).
    	 if(this.isPIV() && this.pivIsValid())
    	 {
    		 out.writeByte((this.getProtocolIdentifier() << 4) | this.getCodeSet().value());
    	 }
    	 else
    	 {
    		 out.writeByte(this.getCodeSet().value());
    	 }

         // byte 1
    	 int b1 = 0;
         if(this.pivIsValid())
         {
        	 b1 = (this.isPIV() ? 1 : 0) << 7;
         }
         
         b1 |= this.getAssociation().value() << 4;
         b1 |= this.getIdentifier().getType().value();
         out.writeByte(b1);

         // byte 2
         out.writeByte(0);

         // byte 3
         out.writeByte(this.getIdentifierLength());

         // identifier
         out.write(this.identifier.encode());

         return baos.toByteArray();
      }
      catch (IOException e)
      {
         throw new RuntimeException("Unable to encode CDB.");
      }
   }
   
   // Helper function to examine if the PIV bit in the identification descriptor is valid
   // or should be reserved.
   // See SPC-3 section 7.6.3.1 (PDF page 324).
   private boolean pivIsValid()
   {
	   if((this.getAssociation() == Association.TARGET_DEVICE) ||
		  (this.getAssociation() == Association.TARGET_PORT))
	   {
		   return true;
	   }
	   else return false;
   }

   /////////////////////////////////////////////////////////////////////////////
   // getters/setters

   public int getProtocolIdentifier()
   {
      return protocolIdentifier;
   }

   public void setProtocolIdentifier(int protocolIdentifier)
   {
      this.protocolIdentifier = protocolIdentifier;
   }

   public CodeSet getCodeSet()
   {
      return codeSet;
   }

   public void setCodeSet(CodeSet codeSet)
   {
      this.codeSet = codeSet;
   }

   public boolean isPIV()
   {
      return PIV;
   }

   public void setPIV(boolean piv)
   {
      PIV = piv;
   }

   public Association getAssociation()
   {
      return association;
   }

   public void setAssociation(Association association)
   {
      this.association = association;
   }

   public DeviceIdentifier getIdentifier()
   {
      return identifier;
   }

   public void setIdentifier(DeviceIdentifier identifier)
   {
      this.identifier = identifier;
   }

   public int getIdentifierLength() throws IOException
   {
      return this.identifier.encode().length;
   }
}
