//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER

//
//Cleversafe open-source code header - Version 1.1 - December 1, 2006
//
//Cleversafe Dispersed Storage(TM) is software for secure, private and
//reliable storage of the world's data using information dispersal.
//
//Copyright (C) 2005-2007 Cleversafe, Inc.
//
//This program is free software; you can redistribute it and/or
//modify it under the terms of the GNU General Public License
//as published by the Free Software Foundation; either version 2
//of the License, or (at your option) any later version.
//
//This program is distributed in the hope that it will be useful,
//but WITHOUT ANY WARRANTY; without even the implied warranty of
//MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//GNU General Public License for more details.
//
//You should have received a copy of the GNU General Public License
//along with this program; if not, write to the Free Software
//Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
//USA.
//
//Contact Information: Cleversafe, 10 W. 35th Street, 16th Floor #84,
//Chicago IL 60616
//email licensing@cleversafe.org
//
//END-OF-HEADER
//-----------------------
//Author: Srini Palthepu (spalthepu@cleversafe.com)
//
//Date: Date: Nov 6, 2007
//---------------------

package org.cleversafe.layer.fileobj.jlanproto;

import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.atomic.AtomicInteger;

import org.alfresco.jlan.server.filesys.loader.FileSegment;
import org.alfresco.jlan.util.NameValueList;
import org.apache.log4j.Logger;
import org.cleversafe.config.BindingsProvider;
import org.cleversafe.config.ConfigurationFactory;
import org.cleversafe.config.exceptions.ConfigurationException;
import org.cleversafe.layer.block.exceptions.BlockIOException;
import org.cleversafe.layer.grid.CorruptedSliceObserver;
import org.cleversafe.layer.grid.DataSource;

import org.cleversafe.layer.grid.DeleteController;
import org.cleversafe.layer.grid.GridController;
import org.cleversafe.layer.grid.ReadController;
import org.cleversafe.layer.grid.ReadControllerFactory;
import org.cleversafe.layer.grid.SourceName;
import org.cleversafe.layer.grid.WriteController;
import org.cleversafe.layer.grid.WriteControllerFactory;
import org.cleversafe.layer.grid.exceptions.ControllerDataTransformationException;
import org.cleversafe.layer.grid.exceptions.ControllerException;
import org.cleversafe.layer.grid.exceptions.ControllerGridStateUnknownException;
import org.cleversafe.layer.grid.exceptions.ControllerIOException;
import org.cleversafe.layer.grid.exceptions.ControllerIllegalSourceNameException;
import org.cleversafe.layer.grid.exceptions.ControllerInformationDispersalException;
import org.cleversafe.layer.grid.exceptions.ControllerStoresNotFoundException;
import org.cleversafe.layer.grid.exceptions.ControllerTransactionException;

/**
* Handles block requests to the grid
*/

public class FileDeviceController
{
   private static Logger _logger = Logger.getLogger(FileDeviceController.class);

   // Number of times to retry a transaction in case of error
   public static int NUM_TRANSACTION_RETRIES = 5;

   // Vault containing device data
   protected FileDeviceVault vault;

   // Prototype controller -- a clone is made for each transaction
   protected GridController gridController;

   protected List<CorruptedSliceObserver> corruptedSliceObservers =
         new ArrayList<CorruptedSliceObserver>();

   // Internal statistics
   protected AtomicInteger statsNumReads = new AtomicInteger();
   protected AtomicInteger statsNumWrites = new AtomicInteger();

   protected boolean started = false;
   protected boolean requestedStarted = false;

   public FileDeviceController(FileDeviceVault vault) throws ConfigurationException
   {
      BindingsProvider bindingsProvider =
         ConfigurationFactory.getBindingsProvider(ConfigurationFactory.XML_CONFIG_TYPE);
      
      ReadControllerFactory rfc =
            bindingsProvider.getDefaultImplementation(ReadControllerFactory.class);
      WriteControllerFactory wfc =
            bindingsProvider.getDefaultImplementation(WriteControllerFactory.class);

      this.vault = vault;
      this.gridController = new GridController(vault, wfc, rfc);
   }

   public void addCorruptedSliceObserver(CorruptedSliceObserver observer)
   {
      this.corruptedSliceObservers.add(observer);
   }

   /**
    * Construct a controller without an actual system device
    * 
    * @param vault
    *           Vault that represents this device
    * @param controller
    *           Controller responsible for slice storage to the grid
    */
   public FileDeviceController(FileDeviceVault vault, GridController controller)
   {
      this.vault = vault;
      this.gridController = controller;
   }

   /**
    * Initialize the controller
    */
   public void startup()
   {
      this.requestedStarted = true;
   }

   protected synchronized void startupImpl()
   {
      assert this.requestedStarted == true;

      if (!this.started)
      {
         this.vault.startSessions();
         this.started = true;
      }
   }

   /**
    * Cleanup controller
    */
   public void shutdown()
   {
      if (this.started)
      {
         this.vault.endSessions();
         this.started = false;
      }
      this.requestedStarted = false;
   }

   /**
    * Read one or more sequential blocks from the grid
    * 
    * @param firstBlock
    *           Index of first block to read
    * @param numBlocks
    *           Number of blocks to read starting from firstBlock
    * @return data Block data of size blockSize * numBlocks
    * @throws ControllerException
    * @throws Exception
    */
   public byte[] readFileObject(int fileId, String objectId, int fragNo, FileSegment fileSeg)
         throws FileIOException
   {
      // Lazy initialization
      if (!this.started)
      {
         startupImpl();
      }

      long startTime = System.nanoTime();
      ReadController readController;
      try
      {
         readController = gridController.getReadController();
      }
      catch (final Exception ex)
      {
         throw new RuntimeException("Unable to instantiate read controller", ex);
      }

      // Add corrupted slice observers
      // TODO: Consider configuring observers from the vault definition or client config
      for (CorruptedSliceObserver observer : corruptedSliceObservers)
      {
         readController.addCorruptedSliceObserver(observer);
      }

      // Perform multi-block read

      List<SourceName> dataSources = new ArrayList<SourceName>();

      dataSources.add(new SourceName(objectId));

      byte[] data = null;

      try
      {
         List<DataSource> results = readController.read(dataSources);
         // we have only one read request.
         data = results.get(0).getData();

         // Update stats
         int numReads = this.statsNumReads.incrementAndGet();
         if (_logger.isDebugEnabled())
            _logger.debug(String.format(
                  "loadFileObject: numReads=%d, fileId=%d, objectId=%s (%.3fms)", numReads, fileId,
                  objectId, (System.nanoTime() - startTime) / 1.e6));

         System.out.println(String.format(
               "loadFileObject: numReads=%d, fileId=%d, objectId=%s (%.3fms)", numReads, fileId,
               objectId, (System.nanoTime() - startTime) / 1.e6));

         return data;
      }
      catch (ControllerDataTransformationException ex)
      {
         throw new FileIOException("Corrupt data for block request", ex);
      }
      catch (ControllerIOException ex)
      {
         throw new FileIOException("IO error reading from grid", ex);
      }
      catch (ControllerGridStateUnknownException ex)
      {
         throw new FileIOException("Indeterminate grid error", ex);
      }
      catch (ControllerInformationDispersalException ex)
      {
         // The IDA should not throw an error on read unless there is a configuration problem
         throw new RuntimeException("IDA error on read, likely a configuration problem", ex);
      }
      catch (ControllerIllegalSourceNameException ex)
      {
         // We create the source name locally, so this is an internal error
         throw new RuntimeException("Internal error: illegal source name", ex);
      }
      catch (ControllerStoresNotFoundException e)
      {
         throw new RuntimeException("Caught exception: " + e, e);
      }
   }

   /**
    * Write one or more sequential blocks to the grid
    * 
    * @param firstBlock
    *           Index of first block to write
    * @param numBlocks
    *           Number of blocks to write starting from firstBlock
    * @param data
    *           Block data of size blockSize * numBlocks
    * @throws BlockIOException
    */
   public String writeFileObject(
         int fileId,
         int streamId,
         int fragId,
         InputStream inStream,
         int fragSize,
         NameValueList attrs) throws FileIOException
   {
      // Lazy initialization
      if (!this.started)
      {
         startupImpl();
      }

      long startTime = System.nanoTime();
      WriteController writeController;

      // Retry a transaction some number of times
      for (int txTries = 0; txTries < NUM_TRANSACTION_RETRIES; ++txTries)
      {
         try
         {
            // Create a new controller w/transaction
            writeController = gridController.getWriteController();

            for (CorruptedSliceObserver observer : corruptedSliceObservers)
            {
               writeController.addCorruptedSliceObserver(observer);
            }
         }
         catch (final Exception ex)
         {
            throw new RuntimeException("Unable to instantiate write controller");
         }
         /*
         int available = 0;
         try {
           available = inStream.available();
         } catch (IOException ex){
           ex.printStackTrace();
         }
         */
         byte[] data = new byte[fragSize];
         // Build multi-block write request

         //    System.out.println("=========numbytes available=" + available);
         try
         {
            inStream.read(data, 0, fragSize);
         }
         catch (IOException ex)
         {
            ex.printStackTrace();
         }

         List<DataSource> request = new ArrayList<DataSource>();

         String objId = Integer.toString(fileId) + ":" + Integer.toString(fragId);
         request.add(new DataSource(new SourceName(objId), writeController.getTransactionId(), data));

         // Send the result to the device
         try
         {
            // Write & commit data
            writeController.write(request);
            writeController.commit();

            // Update stats
            int numWrites = this.statsNumWrites.incrementAndGet();
            if (_logger.isDebugEnabled())
               _logger.debug(String.format(
                     "saveFileObject: numWrites=%d, fileId=%d, streamId=%d, fragId=%d (%.3fms)",
                     numWrites, fileId, streamId, fragId, (System.nanoTime() - startTime) / 1.e6));

            return objId; // Success
         }
         catch (final ControllerTransactionException ex)
         {
            // Give it another try
            _logger.error("Grid transaction error:" + ex);
         }
         catch (final ControllerIllegalSourceNameException ex)
         {
            // Programmer error, since we determine the name here
            throw new RuntimeException("Invalid source name", ex);
         }
         catch (final ControllerDataTransformationException ex)
         {
            // Likely a configuration error
            throw new RuntimeException("Cannot encode data, likely a configuration error", ex);
         }
         catch (final ControllerInformationDispersalException ex)
         {
            // Likely a configuration error
            throw new RuntimeException("Cannot disperse data, likely a configuration error", ex);
         }
         catch (final ControllerIOException ex)
         {
            throw new FileIOException("Write error", ex);
         }
         catch (final ControllerGridStateUnknownException ex)
         {
            throw new FileIOException("Indeterminate grid state", ex);
         }
         catch (ControllerStoresNotFoundException e)
         {
            throw new RuntimeException("Caught exception: " + e, e);
         }
         finally
         {
            // Rolls back if necessary
            try
            {
               writeController.shutdown();
            }
            catch (final ControllerException ex)
            {
               _logger.error("Unable to rollback transaction", ex);
            }
         }
      }

      throw new FileIOException("Unable to complete transaction in " + NUM_TRANSACTION_RETRIES
            + " attempts");
   }

   public void deleteFileObject(int fileId, String objectId, int fragNo) throws FileIOException
   {
      System.out.println("Delete operation fileId=" + fileId + ", ObjectId=" + objectId
            + ", FragNo=" + fragNo);
      // Lazy initialization
      if (!this.started)
      {
         startupImpl();
      }

      DeleteController deleteController;
      try
      {
         deleteController = gridController.getDeleteController();
      }
      catch (final Exception ex)
      {
         throw new RuntimeException("Unable to instantiate read controller", ex);
      }

      List<SourceName> dataSources = new ArrayList<SourceName>();

      dataSources.add(new SourceName(objectId));

      try
      {
         deleteController.delete(dataSources);
      }
      catch (ControllerIOException ex)
      {
         throw new FileIOException("IO error reading from grid", ex);
      }
      catch (ControllerIllegalSourceNameException ex)
      {
         // We create the source name locally, so this is an internal error
         throw new RuntimeException("Internal error: illegal source name", ex);
      }
      catch (ControllerStoresNotFoundException e)
      {
         throw new RuntimeException("Caught exception: " + e, e);
      }
   }

   /**
    * Returns the size of the device, in bytes
    * 
    * @return Size of the device, in bytes
    */
   public long getDeviceSize()
   {
      return this.vault.getMaxDeviceSize();
   }

   public int getStatsNumReads()
   {
      return this.statsNumReads.get();
   }

   public int getStatsNumWrites()
   {
      return this.statsNumWrites.get();
   }
}
