//
// Cleversafe open-source code header - Version 1.1 - December 1, 2006
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2007 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 10 W. 35th Street, 16th Floor #84,
// Chicago IL 60616
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// Author: Zachary Mark
//
// Date: Jul 6, 2007
//---------------------

package org.cleversafe.vault.storage.asn1;

import static org.junit.Assert.*;

import java.security.GeneralSecurityException;
import java.security.KeyPair;
import java.security.KeyPairGenerator;
import java.security.NoSuchAlgorithmException;
import java.security.PublicKey;
import java.util.HashMap;
import java.util.Map;
import java.util.UUID;
import java.util.Map.Entry;

import org.apache.log4j.Logger;
import org.cleversafe.vault.storage.asn1.EncryptedKeyInfo;
import org.cleversafe.vault.storage.asn1.KeyInfo;
import org.cleversafe.vault.storage.asn1.KeyUsage;
import org.cleversafe.vault.storage.asn1.PlainKeyInfo;
import org.cleversafe.vault.storage.asn1.VaultACLEntry;
import org.cleversafe.vault.storage.asn1.VaultKeys;
import org.cleversafe.vault.storage.asn1.VaultPermissionEntry;
import org.cleversafe.vault.storage.asn1.KeyInfo.KeyInfoPair;
import org.junit.BeforeClass;
import org.junit.Test;


/**
 * Test of the VaultACLEntry class.
 *
 */
public class VaultACLEntryTest
{
   private static Logger _logger = Logger.getLogger(VaultACLEntryTest.class);

   private static final String DEFAULT_ENCRYPTION_ALGORITHM = "AES";
   private static final String DEFAULT_VERIFICATION_ALGORITHM = "RSA";

   private static final int DEFAULT_ENCRYPTION_KEY_SIZE = 128;
   private static final int DEFAULT_VERIFICATION_KEY_SIZE = 512;

   private static final String DEFAULT_WRAP_TRANSFORMATION = "RSA/ECB/PKCS1Padding";

   //TODO:  These are copied of the VaultACL class temporarily.  These will be obsolete as
   //VaultACL changes later on.
   protected static final int VAULT_SECRET_KEY = 0;
   protected static final int DATASOURCE_VERIFICATION_PRIVATE_KEY = 1;
   protected static final int SLICE_VERIFICATION_PRIVATE_KEY = 2;
   protected static final int DATASOURCE_VERIFICATION_PUBLIC_KEY = 3;
   protected static final int SLICE_VERIFICATION_PUBLIC_KEY = 4;

   Map<Integer, EncryptedKeyInfo> entryKeys = null;

   @BeforeClass
   public static void setUpBeforeClass() throws Exception
   {

   }

   /**
    * Helper method to simulate the VaultACLFactory's incredibly long and verbose key generation process.
    *
    */
   private void generateKeys()
   {
      KeyPairGenerator keygen;
      KeyPair accountKeyPair = null;
      try
      {
         keygen = KeyPairGenerator.getInstance(DEFAULT_VERIFICATION_ALGORITHM);
         keygen.initialize(DEFAULT_VERIFICATION_KEY_SIZE);
         accountKeyPair = keygen.generateKeyPair();

      }
      catch (NoSuchAlgorithmException e)
      {
         fail("Failed to generate account keypair");
      }

      KeyInfo secretKey = null;
      try
      {
         secretKey =
               PlainKeyInfo.generateKey(DEFAULT_ENCRYPTION_ALGORITHM, DEFAULT_ENCRYPTION_KEY_SIZE,
                     KeyUsage.DATASOURCE_ENCRYPTION);
      }
      catch (NoSuchAlgorithmException e1)
      {
         fail("Failed to generate secret key");
      }

      KeyInfoPair datasourceVerificationKeyPair = null;
      try
      {
         datasourceVerificationKeyPair =
               PlainKeyInfo.generateKeyPair(DEFAULT_VERIFICATION_ALGORITHM,
                     DEFAULT_VERIFICATION_KEY_SIZE, KeyUsage.DATASOURCE_VERIFICATION);
      }
      catch (NoSuchAlgorithmException e)
      {
         fail("Failed to generate datasourceVerificationKeypair");
      }

      KeyInfoPair sliceVerificationKeyPair = null;
      try
      {
         sliceVerificationKeyPair =
               PlainKeyInfo.generateKeyPair(DEFAULT_VERIFICATION_ALGORITHM,
                     DEFAULT_VERIFICATION_KEY_SIZE, KeyUsage.SLICE_VERIFICATION);
      }
      catch (NoSuchAlgorithmException e)
      {
         fail("Failed to generate sliceVerificationKeypair");
      }

      PublicKey publicKey = accountKeyPair.getPublic();

      EncryptedKeyInfo encryptedSecretKey = null;
      EncryptedKeyInfo encryptedDatasourceVerificationKey = null;
      EncryptedKeyInfo encryptedSliceVerificationKey = null;
      try
      {
         encryptedSecretKey =
               EncryptedKeyInfo.wrap((PlainKeyInfo) secretKey, publicKey,
                     DEFAULT_WRAP_TRANSFORMATION);

         encryptedDatasourceVerificationKey =
               EncryptedKeyInfo.wrap((PlainKeyInfo) datasourceVerificationKeyPair.getPrivate(),
                     publicKey, DEFAULT_WRAP_TRANSFORMATION);

         encryptedSliceVerificationKey =
               EncryptedKeyInfo.wrap((PlainKeyInfo) sliceVerificationKeyPair.getPrivate(),
                     publicKey, DEFAULT_WRAP_TRANSFORMATION);

         entryKeys = new HashMap<Integer, EncryptedKeyInfo>();
         entryKeys.put(VAULT_SECRET_KEY, encryptedSecretKey);
         entryKeys.put(DATASOURCE_VERIFICATION_PRIVATE_KEY, encryptedDatasourceVerificationKey);
         entryKeys.put(SLICE_VERIFICATION_PRIVATE_KEY, encryptedSliceVerificationKey);
      }
      catch (GeneralSecurityException ex)
      {
         fail("Could not wrap keys properly");
      }

   }

   /**
    * Tests the main constructor of the VaultACLEntry object.
    *
    */
   @Test
   public void testConstructor()
   {
      _logger.debug("Test case: VaultACLEntry constructor");

      UUID accountID = UUID.randomUUID();
      VaultPermissionEntry entry = VaultPermissionEntry.getFullPermission();

      this.generateKeys();

      VaultKeys keys = new VaultKeys();
      for (Entry<Integer, EncryptedKeyInfo> mapEntry : this.entryKeys.entrySet())
      {
         keys.put(mapEntry.getKey(), mapEntry.getValue());
      }

      VaultACLEntry aclEntry = new VaultACLEntry(accountID, entry, keys);

      assertEquals("UUIDs retrieved not equal", accountID, aclEntry.getAccountIdentifier());
      assertEquals("VaultPermissionEntry retrieved not equal", entry, aclEntry.getPermission());
      assertEquals("VaultKeys retrieved not equal", keys, aclEntry.getVaultKeys());
   }

}
