//
// Cleversafe open-source code header - Version 1.1 - December 1, 2006
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2007 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 10 W. 35th Street, 16th Floor #84,
// Chicago IL 60616
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// Author: wleggette
//
// Date: Dec 5, 2007
//---------------------

package org.cleversafe.vault;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;

import java.io.File;
import java.io.IOException;
import java.security.KeyPair;
import java.security.KeyPairGenerator;
import java.security.NoSuchAlgorithmException;
import java.util.List;
import java.util.Properties;
import java.util.UUID;

import org.cleversafe.TestDataLiterals;
import org.cleversafe.UnitTests;
import org.cleversafe.authentication.Credentials;
import org.cleversafe.authentication.CredentialsManager;
import org.cleversafe.authentication.credentials.PasswordCredentials;
import org.cleversafe.authentication.exceptions.CredentialsException;
import org.cleversafe.server.exceptions.ServerConfigurationLoadException;
import org.cleversafe.storage.ss.SliceServerConfiguration;
import org.cleversafe.storage.ss.SliceServerDaemon;
import org.cleversafe.storage.ss.configuration.ConfigurationLoader;
import org.cleversafe.storage.ss.configuration.XMLConfigurationLoader;
import org.cleversafe.test.BaseTest;
import org.cleversafe.test.TestException;
import org.cleversafe.util.FileSystemUtils;
import org.cleversafe.util.Tuple3;
import org.cleversafe.vault.exceptions.VaultException;
import org.cleversafe.vault.exceptions.VaultIOException;
import org.junit.After;
import org.junit.AfterClass;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;

// TODO: Describe class or interface
public abstract class VaultManagerTest extends BaseTest
{
   private static String DESCRIPTOR_PATH = "conf/dev/block-device/local-bdb-vault-descriptor.xml";
   private static String REMOTE_LOCAL_DESCRIPTOR_PATH = 
      "conf/dev/block-device/remote-local-crc-vault-descriptor.xml";
   
   private static String SERVER_CONFIGURATION_FILE = "server-config.xml";

   private static final String VAULT_NAME_A = "Vault A";
   private static final String VAULT_NAME_B = "Vault B";
   private static final String VAULT_NAME_C = "Vault C";

   public static final String VAULT_DESCRIPTOR_STRING;
   public static final String REMOTE_LOCAL_VAULT_DESCRIPTOR_STRING;

   private static final Credentials CREDENTIALS;
   private static final KeyPair KEY_PAIR;
   private static final CredentialsManager CREDENTIALS_MANAGER;

   private static String VAULT_TYPE = "block";
   private static int VAULT_WIDTH = 8;
   private static int VAULT_IDA_WIDTH = 8;
   private static int VAULT_IDA_THRESHOLD = 6;
   private static int VAULT_CHUNK_SIZE = 4096;
   private static long VAULT_MAX_SLICE_SIZE = 690;
   private static long VAULT_SIZE = 262144 * 4096;

   static String originalConfigurationProperty;
   
   // Slice servers
   private static SliceServerDaemon[] ssSet = null;
   private static Thread[] daemonThreads = null;

   static
   {
      PasswordCredentials credentials = new PasswordCredentials();
      credentials.setUsername("test");
      credentials.setPassword("test");
      CREDENTIALS = credentials;

      try
      {
         KeyPairGenerator keygen = KeyPairGenerator.getInstance("RSA");
         keygen.initialize(512);
         KEY_PAIR = keygen.generateKeyPair();
      }
      catch (NoSuchAlgorithmException e)
      {
         throw new ExceptionInInitializerError(e);
      }

      CREDENTIALS_MANAGER = new TestCredentialsManager(CREDENTIALS, KEY_PAIR);

      try
      {
         VAULT_DESCRIPTOR_STRING = FileSystemUtils.readFileAsString(new File(DESCRIPTOR_PATH));
         REMOTE_LOCAL_VAULT_DESCRIPTOR_STRING =
            FileSystemUtils.readFileAsString(new File(REMOTE_LOCAL_DESCRIPTOR_PATH));
      }
      catch (IOException e)
      {
         throw new ExceptionInInitializerError(e);
      }

   }

   public static class TestCredentialsManager implements CredentialsManager
   {
      private final Credentials credentials;
      private final KeyPair keyPair;

      public TestCredentialsManager(Credentials credentials, KeyPair keypair)
      {
         this.credentials = credentials;
         this.keyPair = keypair;
      }

      public UUID getAccountIdentifier(String username) throws CredentialsException
      {
         return this.credentials.getAccountIdentifier();
      }

      public KeyPair getAccountKeyPair(UUID accountUUID) throws CredentialsException
      {
         return this.keyPair;
      }

      public Credentials getCredentials(UUID accountUUID) throws CredentialsException
      {
         return this.credentials;
      }

   }


   private static void startSliceServers(String serverConfiguration, int daemonCount)
   {

      // Launch daemon
      try
      {
         // Create daemons with generated configurations
         ssSet = new SliceServerDaemon[daemonCount];
         for (int daemonID = 1; daemonID <= ssSet.length; daemonID++)
         {
            Properties properties = new Properties();
            properties.setProperty("DAEMON.ID", Integer.toString(daemonID));

            ConfigurationLoader configLoader = new XMLConfigurationLoader(serverConfiguration);
            SliceServerConfiguration configuration = configLoader.getConfiguration(properties);

            ssSet[daemonID - 1] = new SliceServerDaemon(configuration);
         }

         // Create threads for non-primary daemons
         daemonThreads = new Thread[ssSet.length];
         for (int a = 0; a < ssSet.length; a++)
         {
            daemonThreads[a] = new Thread(ssSet[a], "daemon-" + (a + 1));
            daemonThreads[a].start();
         }

         // Wait until servers have started
         for (SliceServerDaemon daemon : ssSet)
         {
            try
            {
               daemon.awaitStart();
            }
            catch (InterruptedException ignored)
            {
            }
         }
      }
      catch (ServerConfigurationLoadException ex)
      {
         fail(ex.getMessage());
      }
   }
   
   private static void stopSliceServer(int index) throws InterruptedException
   {
      ssSet[index].shutdown();
      daemonThreads[index].join();
      daemonThreads[index] = null;
   }
   
   private static void stopSliceServers() throws InterruptedException
   {
      for (SliceServerDaemon daemon : ssSet)
      {
         daemon.shutdown();
      }
      for (Thread thread : daemonThreads)
      {
         if (thread != null)
            thread.join();
      }
   }
   
   private static void restartSliceServer(int index)
   {
      daemonThreads[index] = new Thread(ssSet[index], "daemon-" + (index + 1));
      daemonThreads[index].start();
   }
   
   @BeforeClass
   public static void setUpBeforeClass() throws Exception
   {
      UnitTests.resetConfiguration();
      TestDataLiterals.setFixedCredentialsProperties();
   }

   @AfterClass
   public static void tearDownAfterClass() throws Exception
   {
   }

   @Before
   public void setUp() throws Exception
   {
   }

   @After
   public void tearDown() throws Exception
   {
   }

   public File getInputDirectory(String testName) throws TestException
   {
      return new File(super.getInputDirectory() + File.separator + testName);
   }

   public File getOutputDirectory(String testName) throws TestException
   {
      return new File(super.getOutputDirectory() + File.separator + testName);
   }

   public String getVaultDescriptor(String vaultName, String testName) throws TestException
   {
      String sliceOutputPath = this.getOutputDirectory(testName).toURI().getPath() + "/slice-data";
      return VAULT_DESCRIPTOR_STRING.replaceAll("00000000-0000-0000-0000-000000000000",
            vaultName.replaceAll(" ", "")).replaceAll("output/slice-data", sliceOutputPath);
   }

   public abstract VaultManager getVaultManagerInstance();
   
   
   @Test
   public void testLoadVault() throws Exception
   {
      Tuple3<VaultManager, UUID, UUID> tuple = this.testCreateVault("testLoadVault");

      VaultManager manager = tuple.getFirst();
      UUID vaultIdentifierA = tuple.getSecond();

      org.cleversafe.vault.Vault vault =
            manager.loadVault(vaultIdentifierA, CREDENTIALS.getAccountIdentifier(),
                  CREDENTIALS_MANAGER);

      assertEquals("VAULT_TYPE incorrect", VAULT_TYPE, vault.getType());
      assertEquals("VAULT_WIDTH incorrect", VAULT_WIDTH, vault.getWidth());
      assertEquals("VAULT_IDA_WIDTH incorrect", VAULT_IDA_WIDTH,
            vault.getInformationDispersalCodec().getNumSlices());
      assertEquals("VAULT_IDA_THRESHOLD incorrect", VAULT_IDA_THRESHOLD,
            vault.getInformationDispersalCodec().getThreshold());
      assertEquals("VAULT_CHUNK_SIZE incorrect", VAULT_CHUNK_SIZE,
            vault.getInformationDispersalCodec().getChunkSize());
      assertEquals("VAULT_MAX_SLICE_SIZE incorrect", VAULT_MAX_SLICE_SIZE, vault.getMaxSliceSize());
      assertEquals("VAULT_SIZE incorrect", VAULT_SIZE, vault.getSize());

      assertEquals("Number of slice stores incorrect", VAULT_WIDTH, vault.getSliceStores().size());
   }
   
   @Test
   public void testGetVaultDescriptor() throws Exception
   {
      Tuple3<VaultManager, UUID, UUID> tuple = this.testCreateVault("testGetVaultDescriptor");
      
      // This is the vault name used in testCreateVault()
      String vaultNameA = "testGetVaultDescriptor_" + VAULT_NAME_A;
      
      VaultManager manager = tuple.getFirst();
      UUID vaultIdentifierA = tuple.getSecond();
      
      String vaultDescriptor = getVaultDescriptor(vaultNameA, "testGetVaultDescriptor");
      
      assertEquals("Retrieved vault descriptor not as expected",
            vaultDescriptor, manager.getVaultDescriptor(vaultIdentifierA));
   }

   @Test
   public void testGetVaultACL() throws Exception
   {
      Tuple3<VaultManager, UUID, UUID> tuple = this.testCreateVault("testGetVaultACL");

      VaultManager manager = tuple.getFirst();
      UUID vaultIdentifierA = tuple.getSecond();
      UUID vaultIdentifierB = tuple.getThird();

      VaultACL acl = manager.getVaultACL(vaultIdentifierA);

      assertEquals("ACL has incorrect vault identifier", vaultIdentifierA, acl.getVaultIdentifier());
      assertTrue("ACL does not have expected account permissions",
            acl.getPermission(CREDENTIALS.getAccountIdentifier()) != null);
      assertEquals("ACL does not have correct account permissions",
            VaultPermission.getFullPermissions(vaultIdentifierA),
            acl.getPermission(CREDENTIALS.getAccountIdentifier()));
      try
      {
         acl.getPermission(UUID.randomUUID());
         fail("ACL has permissions for unexpected account");
      }
      catch (SecurityException e)
      {
      }

      acl = manager.getVaultACL(vaultIdentifierB);

      assertEquals("ACL has incorrect vault identifier", vaultIdentifierB, acl.getVaultIdentifier());
      assertTrue("ACL does not have expected account permissions",
            acl.getPermission(CREDENTIALS.getAccountIdentifier()) != null);
      assertEquals("ACL does not have correct account permissions",
            VaultPermission.getFullPermissions(vaultIdentifierB),
            acl.getPermission(CREDENTIALS.getAccountIdentifier()));
      try
      {
         acl.getPermission(UUID.randomUUID());
         fail("ACL has permissions for unexpected account");
      }
      catch (SecurityException e)
      {
      }

   }

   private Tuple3<VaultManager, UUID, UUID> testCreateVault(String testName) throws Exception
   {
      File output = this.getOutputDirectory(testName + File.separator + "directory");

      // clear old content from test output directory
      // (we leave files at end of test for human verification)
      FileSystemUtils.deleteDir(this.getOutputDirectory(testName));

      String vaultNameA = testName + "_" + VAULT_NAME_A;
      String vaultNameB = testName + "_" + VAULT_NAME_B;

      VaultManager manager = getVaultManagerInstance();
      System.out.println("test output: " + output);
      manager.load(output.toURI());

      List<UUID> ids = manager.getVaults();
      assertEquals("queried vault list not correct", 0, ids.size());

      UUID vaultIdentifier =
            manager.createVault(vaultNameA, getVaultDescriptor(vaultNameA, testName),
                  CREDENTIALS.getAccountIdentifier(), CREDENTIALS_MANAGER);

      assertEquals("queried vault identifier not correct", vaultIdentifier,
            manager.getVaultIdentifier(vaultNameA));

      assertEquals("queried vault name not correct", vaultNameA,
            manager.getVaultName(vaultIdentifier));

      ids = manager.getVaults();
      assertEquals("queried vault list not correct", 1, ids.size());
      assertTrue("vault list does not contain expected entry", ids.contains(vaultIdentifier));

      try
      {
         manager.createVault(vaultNameA, getVaultDescriptor(vaultNameA, testName),
               CREDENTIALS.getAccountIdentifier(), CREDENTIALS_MANAGER);
         fail("vault with duplicate name created, should have failed");
      }
      catch (VaultException e)
      {
      }

      UUID vaultIdentifierB =
            manager.createVault(vaultNameB, getVaultDescriptor(vaultNameB, testName),
                  CREDENTIALS.getAccountIdentifier(), CREDENTIALS_MANAGER);

      assertTrue("new vault created with duplicate vault identifier",
            !vaultIdentifier.equals(vaultIdentifierB));

      assertEquals("queried vault identifier not correct", vaultIdentifierB,
            manager.getVaultIdentifier(vaultNameB));

      assertEquals("queried vault name not correct", vaultNameB,
            manager.getVaultName(vaultIdentifierB));

      ids = manager.getVaults();
      assertEquals("queried vault list not correct", 2, ids.size());
      assertTrue("vault list does not contain expected entry", ids.contains(vaultIdentifier));
      assertTrue("vault list does not contain expected entry", ids.contains(vaultIdentifierB));

      // re-query vault A

      assertEquals("queried vault identifier not correct", vaultIdentifier,
            manager.getVaultIdentifier(vaultNameA));

      assertEquals("queried vault name not correct", vaultNameA,
            manager.getVaultName(vaultIdentifier));

      return new Tuple3<VaultManager, UUID, UUID>(manager, vaultIdentifier, vaultIdentifierB);
   }

   @Test
   public void testCreateVault() throws Exception
   {
      this.testCreateVault("testCreateVault");

   }

   @Test
   public void testRenameVault() throws Exception
   {
      Tuple3<VaultManager, UUID, UUID> tuple = this.testCreateVault("testRenameVault");

      VaultManager manager = tuple.getFirst();
      UUID vaultIdentifierA = tuple.getSecond();
      UUID vaultIdentifierB = tuple.getThird();

      manager.renameVault(vaultIdentifierA, VAULT_NAME_C, CREDENTIALS.getAccountIdentifier(),
            CREDENTIALS_MANAGER);

      assertEquals("queried vault identifier not correct", vaultIdentifierA,
            manager.getVaultIdentifier(VAULT_NAME_C));

      assertEquals("queried vault name not correct", VAULT_NAME_C,
            manager.getVaultName(vaultIdentifierA));

      List<UUID> ids = manager.getVaults();
      assertEquals("queried vault list not correct", 2, ids.size());
      assertTrue("vault list does not contain expected entry", ids.contains(vaultIdentifierA));
      assertTrue("vault list does not contain expected entry", ids.contains(vaultIdentifierB));

   }

   @Test
   public void testDeleteVault() throws Exception
   {
      Tuple3<VaultManager, UUID, UUID> tuple = this.testCreateVault("testDeleteVault");

      VaultManager manager = tuple.getFirst();
      UUID vaultIdentifierA = tuple.getSecond();
      UUID vaultIdentifierB = tuple.getThird();

      manager.deleteVault(vaultIdentifierA, CREDENTIALS.getAccountIdentifier(), CREDENTIALS_MANAGER);

      try
      {
         manager.getVaultName(vaultIdentifierA);
         fail("vault name lookup succeeded unexpectedly, vault should have been deleted");
      }
      catch (VaultIOException e)
      {
      }

      List<UUID> ids = manager.getVaults();
      assertEquals("queried vault list not correct", 1, ids.size());
      assertTrue("vault list does not contain expected entry", ids.contains(vaultIdentifierB));

      manager.deleteVault(vaultIdentifierB, CREDENTIALS.getAccountIdentifier(), CREDENTIALS_MANAGER);

      try
      {
         manager.getVaultName(vaultIdentifierB);
         fail("vault name lookup succeeded unexpectedly, vault should have been deleted");
      }
      catch (VaultIOException e)
      {
      }

      ids = manager.getVaults();
      assertEquals("queried vault list not correct", 0, ids.size());

   }

   @Test
   public void testHandleDeleteVaultFailures() throws Exception
   {
      final String serverConfiguration =
         this.getInputDirectory().toString() + File.separator + SERVER_CONFIGURATION_FILE;
      final String serverOutput =
         this.getOutputDirectory() + File.separator + "slice-server";
      
      System.out.println("server configuration: " + serverConfiguration);
      
      final File output = this.getOutputDirectory("testHandleDeleteVaultFailures/directory");
      FileSystemUtils.deleteDir(output.getParentFile());
      FileSystemUtils.deleteDir(new File(serverOutput));
      
      startSliceServers(serverConfiguration, 8);
      
      VaultManager manager = getVaultManagerInstance();
      System.out.println("test output: " + output);
      manager.load(output.toURI());
      
      UUID vaultIdentifier = manager.createVault(
            VAULT_NAME_A, 
            REMOTE_LOCAL_VAULT_DESCRIPTOR_STRING, 
            CREDENTIALS.getAccountIdentifier(),
            CREDENTIALS_MANAGER);
      
      stopSliceServer(2);
      
      try
      {
         manager.deleteVault(
               vaultIdentifier, 
               CREDENTIALS.getAccountIdentifier(), 
               CREDENTIALS_MANAGER);
         fail("Succeeded unexpectedly while deleting vault");
      }
      catch (VaultIOException e)
      {
         e.printStackTrace();
      }
      
      stopSliceServers();

      startSliceServers(serverConfiguration, 8);
      
      try
      {
         manager.deleteVault(
               vaultIdentifier, 
               CREDENTIALS.getAccountIdentifier(), 
               CREDENTIALS_MANAGER);
      }
      catch (VaultIOException e)
      {
         e.printStackTrace();
         fail("Failed to delete vault after restarting slice servers");
      }
      
   }

}
