//
// Cleversafe open-source code header - Version 1.1 - December 1, 2006
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2007 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 10 W. 35th Street, 16th Floor #84,
// Chicago IL 60616
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// Author: mmotwani
//
// Date: Jun 21, 2007
//---------------------

package org.cleversafe.vault;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.security.Key;
import java.security.KeyPair;
import java.security.KeyPairGenerator;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.util.ArrayList;
import java.util.List;
import java.util.UUID;

import javax.crypto.SecretKey;

import org.cleversafe.authentication.credentials.PasswordCredentials;
import org.cleversafe.test.BaseTest;
import org.cleversafe.vault.VaultACL.Entry;
import org.cleversafe.vault.storage.VaultKeyInfo;
import org.cleversafe.vault.storage.asn1.KeyUsage;
import org.junit.BeforeClass;
import org.junit.Test;

public class VaultACLTest
{
   static final File _outputdir =
         new File(System.getProperty(BaseTest.TEST_OUTPUT_PROPERTY, ".") + "/org/cleversafe/vault");

   @BeforeClass
   public static void setUpBeforeClass() throws Exception
   {
      _outputdir.mkdirs();
   }

   @Test
   public void testGetPermission() throws Exception
   {
      UUID vaultIdentifier = UUID.randomUUID();

      String vaultFileName =
            System.getProperty(BaseTest.TEST_OUTPUT_PROPERTY, ".") + "/org/cleversafe/vault/"
                  + vaultIdentifier.toString() + ".der";
      FileOutputStream out = new FileOutputStream(vaultFileName);

      PasswordCredentials credentials = new PasswordCredentials();
      credentials.setUsername("vaultacltest");
      credentials.setPassword("password");

      System.out.println("Creating Vault ACL for account: "
            + credentials.getAccountIdentifier().toString());

      KeyPairGenerator keygen = KeyPairGenerator.getInstance("RSA");
      keygen.initialize(512);
      KeyPair keypair = keygen.generateKeyPair();

      List<VaultKeyInfo> keyInfoList = new ArrayList<VaultKeyInfo>();

      VaultKeyInfo sliceverify = new VaultKeyInfo();
      sliceverify.setTransformation("RSA/ECB/PKCS1Padding");
      sliceverify.setSize(1024);
      sliceverify.setStorageIndex(0);
      sliceverify.setUsage(KeyUsage.SLICE_VERIFICATION);

      keyInfoList.add(sliceverify);

      VaultKeyInfo dataverify = new VaultKeyInfo();
      dataverify.setTransformation("RSA/ECB/PKCS1Padding");
      dataverify.setSize(1024);
      dataverify.setStorageIndex(2);
      dataverify.setUsage(KeyUsage.DATASOURCE_VERIFICATION);

      keyInfoList.add(dataverify);

      VaultKeyInfo dataenc = new VaultKeyInfo();
      dataenc.setAlgorithm("AES");
      dataenc.setSize(256);
      dataenc.setStorageIndex(1);
      dataenc.setUsage(KeyUsage.DATASOURCE_ENCRYPTION);

      keyInfoList.add(dataenc);

      VaultACLFactory fact = new VaultACLFactory();
      fact.create(out, vaultIdentifier, credentials, keypair.getPublic(), keyInfoList);
      out.close();

      FileInputStream in = new FileInputStream(vaultFileName);

      VaultACL vaultACL = fact.getInstance(in);
      in.close();

      assertEquals(credentials.getAccountIdentifier(), vaultACL.getOwner());

      VaultPermission permission = vaultACL.getPermission(vaultACL.getOwner());
      assertNotNull(permission);

      try
      {
         vaultACL.getPermission(UUID.randomUUID());
         fail("Should have thrown a SecurityException!");
      }
      catch (SecurityException se)
      {
         // Expected
      }

      Entry entry = vaultACL.getEntry(credentials.getAccountIdentifier(), keypair.getPrivate());

      Key key = entry.getKey(0);
      assertTrue(key instanceof PrivateKey);

      key = entry.getPublicKey(0);
      assertTrue(key instanceof PublicKey);

      key = entry.getKey(1);
      assertTrue(key instanceof SecretKey);

      key = entry.getKey(2);
      assertTrue(key instanceof PrivateKey);

      key = entry.getPublicKey(2);
      assertTrue(key instanceof PublicKey);
   }
}
