//
// Cleversafe open-source code header - Version 1.1 - December 1, 2006
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2007 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 10 W. 35th Street, 16th Floor #84,
// Chicago IL 60616
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// Author: wleggette
//
// Date: Jun 20, 2007
//---------------------

package org.cleversafe.vault;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;

import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.security.KeyPair;
import java.security.KeyPairGenerator;
import java.security.Permission;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.util.ArrayList;
import java.util.List;
import java.util.UUID;

import javax.crypto.SecretKey;

import org.cleversafe.UnitTests;
import org.cleversafe.authentication.credentials.PasswordCredentials;
import org.cleversafe.vault.exceptions.VaultKeyGenerationException;
import org.cleversafe.vault.exceptions.VaultKeyLookupException;
import org.cleversafe.vault.storage.VaultKeyInfo;
import org.cleversafe.vault.storage.asn1.KeyUsage;
import org.junit.After;
import org.junit.AfterClass;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;


// TODO: Describe class or interface
public class VaultACLFactoryTest
{
   
   @BeforeClass
   public static void setUpBeforeClass() throws Exception
   {
      UnitTests.resetConfiguration();
   }

   @AfterClass
   public static void tearDownAfterClass() throws Exception
   {
   }

   @Before
   public void setUp() throws Exception
   {
   }

   @After
   public void tearDown() throws Exception
   {
   }

   @Test
   public void testCreate() throws Exception
   {
      FileOutputStream out = new FileOutputStream(
            System.getProperty("org.cleversafe.test-input",".") + "/org/cleversafe/vault/VaultACLFactory.Create01.der" );
      
      UUID vaultIdentifier = UUID.randomUUID();
      PasswordCredentials credentials = new PasswordCredentials();
      credentials.setUsername("test");
      credentials.setPassword("test");
      
      System.out.println(
            "Creating Vault ACL for account: " + credentials.getAccountIdentifier().toString() );
      
      KeyPairGenerator keygen = KeyPairGenerator.getInstance("RSA");
      keygen.initialize( 2048 );
      KeyPair keypair = keygen.generateKeyPair();
      
      List<VaultKeyInfo> keyInfoList = new ArrayList<VaultKeyInfo>();
      
      VaultKeyInfo sliceverify = new VaultKeyInfo();
      sliceverify.setTransformation("RSA/ECB/PKCS1Padding");
      sliceverify.setSize(1024);
      sliceverify.setStorageIndex(0);
      sliceverify.setUsage(KeyUsage.SLICE_VERIFICATION);
      
      keyInfoList.add(sliceverify);
      
      VaultKeyInfo dataenc = new VaultKeyInfo();
      dataenc.setAlgorithm("AES/CBC/PKCS5Padding");
      dataenc.setSize(256);
      dataenc.setStorageIndex(1);
      dataenc.setUsage(KeyUsage.DATASOURCE_ENCRYPTION);
      
      keyInfoList.add(dataenc);
      
      VaultKeyInfo dataverify = new VaultKeyInfo();
      dataverify.setAlgorithm("SHA1withDSA");
      dataverify.setSize(1024);
      dataverify.setStorageIndex(2);
      dataverify.setUsage(KeyUsage.DATASOURCE_VERIFICATION);
      
      keyInfoList.add(dataverify);
      
      
      VaultACLFactory fact = new VaultACLFactory();
      fact.create( out, vaultIdentifier, credentials, keypair.getPublic(), keyInfoList );
      out.close();
      
      ////////
      
      FileInputStream in = new FileInputStream(
            System.getProperty("org.cleversafe.test-input",".") + "/org/cleversafe/vault/VaultACLFactory.Create01.der" );
      
      VaultACL recovered = fact.getInstance(in);
      
      assertEquals(
            "Incorrect vault owner", credentials.getAccountIdentifier(), recovered.getOwner() );
      
      VaultACL.Entry entry = recovered.getEntry(
                                 credentials.getAccountIdentifier(),
                                 keypair.getPrivate() );
      
      assertEquals( "Incorrect key algorithm", "RSA", entry.getKey(0).getAlgorithm() );
      assertTrue( "Incorrect key type", entry.getKey(0) instanceof PrivateKey );
      assertEquals( "Incorrect key algorithm", "RSA", entry.getPublicKey(0).getAlgorithm() );
      assertTrue( "Incorrect key type", entry.getPublicKey(0) instanceof PublicKey );
      assertEquals( "Incorrect key algorithm", "AES", entry.getKey(1).getAlgorithm() );
      assertTrue( "Incorrect key type", entry.getKey(1) instanceof SecretKey );
      assertEquals( "Incorrect key algorithm", "DSA", entry.getKey(2).getAlgorithm() );
      assertTrue( "Incorrect key type", entry.getKey(2) instanceof PrivateKey );
      assertEquals( "Incorrect key algorithm", "DSA", entry.getPublicKey(2).getAlgorithm() );
      assertTrue( "Incorrect key type", entry.getPublicKey(2) instanceof PublicKey );
      
      try
      {
         entry.getPublicKey(1);
         fail("Expected vault key lookup exception did not occur");
      }
      catch ( VaultKeyLookupException e) {}
      catch ( Exception e )
      {
         fail("Unexpected exception occured during bad vault key lookup");
      }
      
      Permission vaultPermission = entry.getPermission();
      assertEquals( "incorrect vault ID on vault permission", 
                    vaultIdentifier.toString(), 
                    vaultPermission.getName() );
   }
  
   
   @Test
   public void testBadAsymmetricKeyUsageCreate() throws Exception
   {
      FileOutputStream out = new FileOutputStream(
            System.getProperty("org.cleversafe.test-input",".") + "/org/cleversafe/vault/VaultACLFactory.Create02.der" );

      UUID vaultIdentifier = UUID.randomUUID();
      PasswordCredentials credentials = new PasswordCredentials();
      credentials.setUsername("test");
      credentials.setPassword("test");
      
      System.out.println(
            "Creating Vault ACL for account: " + credentials.getAccountIdentifier().toString() );
      
      KeyPairGenerator keygen = KeyPairGenerator.getInstance("RSA");
      keygen.initialize( 2048 );
      KeyPair keypair = keygen.generateKeyPair();
      
      List<VaultKeyInfo> keyInfoList = new ArrayList<VaultKeyInfo>();
      
      VaultKeyInfo sliceverify = new VaultKeyInfo();
      sliceverify.setTransformation("AES");
      sliceverify.setSize(1024);
      sliceverify.setStorageIndex(0);
      sliceverify.setUsage(KeyUsage.SLICE_VERIFICATION);
      
      keyInfoList.add(sliceverify);
      
      
      try
      {
         VaultACLFactory fact = new VaultACLFactory();
         fact.create( out, vaultIdentifier, credentials, keypair.getPublic(), keyInfoList );
         fail("ACL generation with faulty asymmetric key succeeded; " +
         		"expected key generation exception" );
      }
      catch (VaultKeyGenerationException e) {}
      catch (Exception e)
      {
         fail("unexpected exception with faulty key algorithm");
      }
   }
   
   @Test
   public void testBadSymmetricKeyUsageCreate() throws Exception
   {
      FileOutputStream out = new FileOutputStream(
            System.getProperty("org.cleversafe.test-input",".") + "/org/cleversafe/vault/VaultACLFactory.Create03.der" );

      UUID vaultIdentifier = UUID.randomUUID();
      PasswordCredentials credentials = new PasswordCredentials();
      credentials.setUsername("test");
      credentials.setPassword("test");
      
      System.out.println(
            "Creating Vault ACL for account: " + credentials.getAccountIdentifier().toString() );
      
      KeyPairGenerator keygen = KeyPairGenerator.getInstance("RSA");
      keygen.initialize( 2048 );
      KeyPair keypair = keygen.generateKeyPair();
      
      List<VaultKeyInfo> keyInfoList = new ArrayList<VaultKeyInfo>();
      
      VaultKeyInfo dataenc = new VaultKeyInfo();
      dataenc.setAlgorithm("RSA");
      dataenc.setSize(256);
      dataenc.setStorageIndex(1);
      dataenc.setUsage(KeyUsage.DATASOURCE_ENCRYPTION);
      
      keyInfoList.add(dataenc);
      
      
      try
      {
         VaultACLFactory fact = new VaultACLFactory();
         fact.create( out, vaultIdentifier, credentials, keypair.getPublic(), keyInfoList );
         fail("ACL generation with faulty symmetric key succeeded; " +
               "expected key generation exception" );
      }
      catch (VaultKeyGenerationException e) {}
      catch (Exception e)
      {
         fail("unexpected exception with faulty key algorithm");
      }
      
   }
}

