//
// Cleversafe open-source code header - Version 1.1 - December 1, 2006
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2007 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 10 W. 35th Street, 16th Floor #84,
// Chicago IL 60616
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// Author: Wesley Leggette
//
// Date: May 3, 2007
//---------------------

package org.cleversafe.vault;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.fail;

import java.security.InvalidKeyException;
import java.security.KeyPair;
import java.security.cert.CertificateExpiredException;
import java.security.cert.CertificateNotYetValidException;
import java.security.cert.X509Certificate;
import java.util.UUID;

import org.junit.After;
import org.junit.AfterClass;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;

// TODO: Describe class or interface
public class ModelTestUtilTest
{

   @BeforeClass
   public static void setUpBeforeClass() throws Exception
   {
   }

   @AfterClass
   public static void tearDownAfterClass() throws Exception
   {
   }

   @Before
   public void setUp() throws Exception
   {
   }

   @After
   public void tearDown() throws Exception
   {
   }









   @Test
   public void testGenerateFixedKeyPair()
   {
      KeyPair a1 = ModelTestUtil.generateFixedKeyPair();
      KeyPair a2 = ModelTestUtil.generateFixedKeyPair();
      KeyPair a3 = ModelTestUtil.generateFixedKeyPair();
      
      assertEquals( a1.getPublic(), a2.getPublic() );
      assertEquals( a1.getPrivate(), a2.getPrivate() );
      assertEquals( a2.getPublic(), a3.getPublic() );
      assertEquals( a2.getPrivate(), a3.getPrivate() );
      assertEquals( a3.getPublic(), a1.getPublic() );
      assertEquals( a3.getPrivate(), a1.getPrivate() );
   }

   @Test
   public void testGenerateRandomKeyPair()
   {
      KeyPair a1 = ModelTestUtil.generateRandomKeyPair();
      KeyPair a2 = ModelTestUtil.generateRandomKeyPair();
      KeyPair a3 = ModelTestUtil.generateRandomKeyPair();
      
      assertFalse( a1.getPublic().equals(a2.getPublic()) );
      assertFalse( a1.getPrivate().equals(a2.getPrivate()) );
      assertFalse( a2.getPublic().equals(a3.getPublic()) );
      assertFalse( a2.getPrivate().equals(a3.getPrivate()) );
      assertFalse( a3.getPublic().equals(a1.getPublic()) );
      assertFalse( a3.getPrivate().equals(a1.getPrivate()) );
   }

   @Test
   public void testGenerateFixedValidCertificate()
   {
      X509Certificate ca = ModelTestUtil.generateFixedCaCertificate();
      X509Certificate a1 = ModelTestUtil.generateFixedValidCertificate();
      X509Certificate a2 = ModelTestUtil.generateFixedValidCertificate();
      
      assertEquals( a1.getIssuerDN(), a2.getIssuerDN() );
      assertEquals( a1.getSubjectDN(), a2.getSubjectDN() );
      assertEquals( a1.getPublicKey(), a2.getPublicKey() );
      
      assertEquals( a1.getIssuerDN(), ca.getSubjectDN() );
      
      try
      {
         a1.checkValidity();
         a2.checkValidity();
      }
      catch (Exception e)
      {
         fail("Certificate is not valid: " + e);
      }
      
      try
      {
         a1.verify(ca.getPublicKey());
         a2.verify(ca.getPublicKey());
      }
      catch (Exception e)
      {
         fail("Certificate verification failed: " + e);
      }
   }

   @Test
   public void testGenerateFixedExpiredCertificate()
   {
      X509Certificate ca = ModelTestUtil.generateFixedCaCertificate();
      X509Certificate a1 = ModelTestUtil.generateFixedExpiredCertificate();
      X509Certificate a2 = ModelTestUtil.generateFixedExpiredCertificate();
      
      assertEquals( a1.getIssuerDN(), a2.getIssuerDN() );
      assertEquals( a1.getSubjectDN(), a2.getSubjectDN() );
      assertEquals( a1.getPublicKey(), a2.getPublicKey() );
      
      assertEquals( a1.getIssuerDN(), ca.getSubjectDN() );
      
      try
      {
         a1.checkValidity();
         fail("Certificate not expired; expected to be");
      }
      catch (CertificateNotYetValidException e)
      {
         fail("Certificate not yet valid; expected to be: " + e);
      }
      catch (CertificateExpiredException e) {}
      
      try
      {
         a1.verify(ca.getPublicKey());
         a2.verify(ca.getPublicKey());
      }
      catch (Exception e)
      {
         fail("Certificate verification failed: " + e);
      }
   }

   @Test
   public void testGenerateFixedBadIssuerCertificate()
   {
      X509Certificate ca = ModelTestUtil.generateFixedCaCertificate();
      X509Certificate a1 = ModelTestUtil.generateFixedBadIssuerCertificate();
      X509Certificate a2 = ModelTestUtil.generateFixedBadIssuerCertificate();
      
      assertEquals( a1.getIssuerDN(), a2.getIssuerDN() );
      assertEquals( a1.getSubjectDN(), a2.getSubjectDN() );
      assertEquals( a1.getPublicKey(), a2.getPublicKey() );
      
      assertFalse( a1.getIssuerDN().equals(ca.getSubjectDN()) );
      
      try
      {
         a1.checkValidity();
         a2.checkValidity();
      }
      catch (Exception e)
      {
         fail("Certificate is not valid: " + e);
      }
      
      try
      {
         a1.verify( ca.getPublicKey() );
         fail("Certificate verification successful; should have failed due to incorrect issuer");
      }
      catch (InvalidKeyException e) {}
      catch (Exception e)
      {
         fail("Unexpected exception: " + e);
      }
   }

   @Test
   public void testGenerateFixedCaCertificate()
   {
      X509Certificate a1 = ModelTestUtil.generateFixedCaCertificate();
      X509Certificate a2 = ModelTestUtil.generateFixedCaCertificate();
      
      assertEquals( a1.getIssuerDN(), a2.getIssuerDN() );
      assertEquals( a1.getSubjectDN(), a2.getSubjectDN() );
      assertEquals( a1.getPublicKey(), a2.getPublicKey() );
      
      // ensure certificate is self-signed
      assertEquals( a1.getIssuerDN(), a1.getSubjectDN() );
      
      try
      {
         a1.checkValidity();
         a2.checkValidity();
      }
      catch (Exception e)
      {
         fail("Certificate is not valid: " + e);
      }
      
      try
      {
         a1.verify(a1.getPublicKey());
         a2.verify(a2.getPublicKey());
      }
      catch (Exception e)
      {
         fail("Certificate verification failed: " + e);
      }
   }

   @Test
   public void testGenerateFixedRaCertificate()
   {
      X509Certificate ca = ModelTestUtil.generateFixedCaCertificate();
      X509Certificate a1 = ModelTestUtil.generateFixedRaCertificate();
      X509Certificate a2 = ModelTestUtil.generateFixedRaCertificate();
      
      assertEquals( a1.getIssuerDN(), a2.getIssuerDN() );
      assertEquals( a1.getSubjectDN(), a2.getSubjectDN() );
      assertEquals( a1.getPublicKey(), a2.getPublicKey() );
      
      assertEquals( a1.getIssuerDN(), ca.getSubjectDN() );
      
      try
      {
         a1.checkValidity();
         a2.checkValidity();
      }
      catch (Exception e)
      {
         fail("Certificate is not valid: " + e);
      }
      
      try
      {
         a1.verify(ca.getPublicKey());
         a2.verify(ca.getPublicKey());
      }
      catch (Exception e)
      {
         fail("Certificate verification failed: " + e);
      }
   }

   @Test
   public void testGenerateRandomValidCertificate()
   {
      UUID account = UUID.randomUUID();
      
      X509Certificate ca = ModelTestUtil.generateFixedCaCertificate();
      X509Certificate a1 = ModelTestUtil.generateRandomValidCertificate( account );
      X509Certificate a2 = ModelTestUtil.generateRandomValidCertificate( account );
      
      assertFalse( a1.getIssuerDN() == a2.getIssuerDN() );
      assertFalse( a1.getSubjectDN() == a2.getSubjectDN() );
      assertFalse( a1.getPublicKey() == a2.getPublicKey() );
      
      assertEquals( a1.getIssuerDN(), ca.getSubjectDN() );
      assertEquals( a2.getIssuerDN(), ca.getSubjectDN() );
      
      try
      {
         a1.checkValidity();
         a2.checkValidity();
      }
      catch (Exception e)
      {
         fail("Certificate is not valid: " + e);
      }
      
      try
      {
         a1.verify(ca.getPublicKey());
         a2.verify(ca.getPublicKey());
      }
      catch (Exception e)
      {
         fail("Certificate verification failed: " + e);
      }
   }

   @Test
   public void testGenerateRandomExpiredCertificate()
   {
      UUID account = UUID.randomUUID();
      
      X509Certificate ca = ModelTestUtil.generateFixedCaCertificate();
      X509Certificate a1 = ModelTestUtil.generateRandomExpiredCertificate( account );
      X509Certificate a2 = ModelTestUtil.generateRandomExpiredCertificate( account );
      
      assertFalse( a1.getIssuerDN() == a2.getIssuerDN() );
      assertFalse( a1.getSubjectDN() == a2.getSubjectDN() );
      assertFalse( a1.getPublicKey() == a2.getPublicKey() );
      
      assertEquals( a1.getIssuerDN(), ca.getSubjectDN() );
      assertEquals( a2.getIssuerDN(), ca.getSubjectDN() );
      
      try
      {
         a1.checkValidity();
         fail("Certificate not expired; expected to be");
      }
      catch (CertificateNotYetValidException e)
      {
         fail("Certificate not yet valid; expected to be: " + e);
      }
      catch (CertificateExpiredException e) {}
      
      try
      {
         a1.verify(ca.getPublicKey());
         a2.verify(ca.getPublicKey());
      }
      catch (Exception e)
      {
         fail("Certificate verification failed: " + e);
      }
   }

   @Test
   public void testGenerateRandomBadIssuerCertificate()
   {
      UUID account = UUID.randomUUID();
      
      X509Certificate ca = ModelTestUtil.generateFixedCaCertificate();
      X509Certificate a1 = ModelTestUtil.generateRandomBadIssuerCertificate( account );
      X509Certificate a2 = ModelTestUtil.generateRandomBadIssuerCertificate( account );
      
      assertFalse( a1.getIssuerDN() == a2.getIssuerDN() );
      assertFalse( a1.getSubjectDN() == a2.getSubjectDN() );
      assertFalse( a1.getPublicKey() == a2.getPublicKey() );
      
      try
      {
         a1.checkValidity();
         a2.checkValidity();
      }
      catch (Exception e)
      {
         fail("Certificate is not valid: " + e);
      }
      
      try
      {
         a1.verify( ca.getPublicKey() );
         fail("Certificate verification successful; should have failed due to incorrect issuer");
      }
      catch (InvalidKeyException e) {}
      catch (Exception e)
      {
         fail("Unexpected exception: " + e);
      }
   }

   @Test
   public void testGenerateFixedUUID()
   {
      UUID a1 = ModelTestUtil.generateFixedUUID();
      UUID a2 = ModelTestUtil.generateFixedUUID();
      UUID a3 = ModelTestUtil.generateFixedUUID();
      
      assertEquals( a1, a2 );
      assertEquals( a2, a3 );
      assertEquals( a3, a1 );
   }





}


