
package org.cleversafe.vault;

import java.util.ArrayList;
import java.util.List;

import org.apache.xmlbeans.XmlOptions;
import org.cleversafe.config.evaluator.Argument;
import org.cleversafe.config.evaluator.Literal;
import org.cleversafe.config.evaluator.Method;
import org.cleversafe.config.evaluator.Parameter;
import org.cleversafe.config.evaluator.Reference;
import org.cleversafe.config.evaluator.SupportedTypes;
import org.cleversafe.test.BaseTest;
import org.junit.Test;
public class CreateXmlVaultDescriptor extends BaseTest
{
   public enum SliceSignature
   {
      NONE, CRC, DSA
   }
   public enum Encryption
   {
      NONE, AES, DES3
   }

   public void addLiteralParam(
         Parameter parameter,
         String name,
         SupportedTypes.Enum type,
         String value)
   {
      parameter.setName(name);

      Literal literal = parameter.addNewLiteral();
      literal.setType(type);
      literal.setValue(value);
   }

   public void addOneArgMethod(
         Parameter parameter,
         String paramName,
         String object,
         String name,
         int value)
   {
      parameter.setName(paramName);
      Method method = parameter.addNewMethod();
      method.setObject(object);
      method.setName(name);
      Argument arg = method.addNewArg();
      Literal literal = arg.addNewLiteral();
      literal.setType(SupportedTypes.INT);
      literal.setValue(String.valueOf(value));
   }

   public void addIDA(VaultDocument.Vault vault, int width, int threshold)
   {
      Reference ida = vault.addNewIda();
      ida.setKind("IDA");
      ida.setReferral("optimizedcauchy");
      addLiteralParam(ida.addNewParam(), "num-slices", SupportedTypes.INT, String.valueOf(width));
      addLiteralParam(ida.addNewParam(), "threshold", SupportedTypes.INT, String.valueOf(threshold));
   }

   public void addSize(VaultDocument.Vault vault, long size)
   {
      long blockSize = 4096;
      long numBlocks = size / blockSize;
      addLiteralParam(vault.addNewParam(), "block-size", SupportedTypes.INT,
            String.valueOf(blockSize));
      addLiteralParam(vault.addNewParam(), "num-blocks", SupportedTypes.LONG,
            String.valueOf(numBlocks));

   }

   public void addSliceServer(VaultDocument.Vault vault, String ip)
   {
      Reference sliceStore = vault.addNewSliceStore();
      sliceStore.setKind("SliceStore");
      sliceStore.setReferral("remote");
      Parameter connection = sliceStore.addNewParam();
      connection.setName("connection");
      Reference conReference = connection.addNewReference();
      conReference.setKind("Connection");

      addLiteralParam(conReference.addNewParam(), "host", SupportedTypes.STRING, ip);
      addLiteralParam(conReference.addNewParam(), "port", SupportedTypes.INT, "5000");
      addLiteralParam(conReference.addNewParam(), "enable-SSL", SupportedTypes.BOOLEAN, "false");

      Parameter viParam = sliceStore.addNewParam();
      viParam.setName("vault-identifier");
      Method viMethod = viParam.addNewMethod();
      viMethod.setObject(VaultDescriptor.ACL_CTX_STRING);
      viMethod.setName("getVaultIdentifier");
      Parameter credential = sliceStore.addNewParam();
      credential.setName("credentials");
      Reference credRef = credential.addNewReference();
      credRef.setObject(VaultDescriptor.CREDENTIALS_CTX_STRING);
   }

   public void addSliceSignature(VaultDocument.Vault vault, boolean isPost, SliceSignature preSlice)
   {
      if (preSlice != SliceSignature.NONE)
      {
         Reference sliceCodec;
         if (isPost)
         {
            sliceCodec = vault.addNewSliceCodec();
         }
         else
         {
            sliceCodec = vault.addNewCodec();
         }

         sliceCodec.setKind("Codec.Integrity");

         switch (preSlice)
         {
            case CRC :
               sliceCodec.setReferral("crc");
               break;
            case DSA :
               sliceCodec.setReferral("signature");
               addLiteralParam(sliceCodec.addNewParam(), "algorithm", SupportedTypes.STRING,
                     "SHA1withDSA");
               addOneArgMethod(sliceCodec.addNewParam(), "private-key",
                     VaultDescriptor.ACL_ENTRY_CTX_STRING, "getKey", 1);
               addOneArgMethod(sliceCodec.addNewParam(), "public-key",
                     VaultDescriptor.ACL_ENTRY_CTX_STRING, "getPublicKey", 1);
         }
      }
   }

   public void addTransposition(VaultDocument.Vault vault, boolean transposition, int threshold)
   {
      if (transposition)
      {
         Reference transpose = vault.addNewCodec();
         transpose.setKind("Codec");
         transpose.setReferral("transposition");

         addLiteralParam(transpose.addNewParam(), "block-size", SupportedTypes.INT, "4112");
         addLiteralParam(transpose.addNewParam(), "threshold", SupportedTypes.INT,
               String.valueOf(threshold));

      }
   }

   public void addEncryption(VaultDocument.Vault vault, Encryption encryption)
   {
      if (encryption != Encryption.NONE)
      {
         Reference codec = vault.addNewCodec();
         codec.setKind("Codec.Encryption");
         codec.setReferral("cipher-codec");

         addOneArgMethod(codec.addNewParam(), "key", VaultDescriptor.ACL_ENTRY_CTX_STRING,
               "getKey", 0);
         switch (encryption)
         {
            case AES :
               addLiteralParam(codec.addNewParam(), "transformation", SupportedTypes.STRING,
                     "AES/CBC/PKCS5Padding");
               break;
            case DES3 :
               addLiteralParam(codec.addNewParam(), "transformation", SupportedTypes.STRING,
                     "3DES/CBC/PKCS5Padding");
               break;

         }
         addLiteralParam(codec.addNewParam(), "use-essiv", SupportedTypes.BOOLEAN, "true");
      }
   }

   @Test
   public void testCreateBlockDescriptor()
   {
      List<String> ips = new ArrayList<String>();
      ips.add("192.168.128.34");
      ips.add("192.168.128.35");
      createBlockDescriptor(1L * 1024 * 1024 * 1024, 1, SliceSignature.CRC, SliceSignature.DSA,
            Encryption.AES, true, ips);
   }

   public void createBlockDescriptor(
         long size,
         int threshold,
         SliceSignature preSlice,
         SliceSignature postSlice,
         Encryption encryption,
         boolean transposition,
         List<String> ips)
   {
      VaultDocument doc = VaultDocument.Factory.newInstance();
      XmlOptions options = new XmlOptions();
      options.setSavePrettyPrint();
      options.setSavePrettyPrintIndent(4);
      options.setSaveAggressiveNamespaces();

      VaultDocument.Vault vault = doc.addNewVault();
      vault.setType("block");
      Parameter viParam = vault.addNewParam();
      viParam.setName("vault-identifier");
      Method viMethod = viParam.addNewMethod();
      viMethod.setObject(VaultDescriptor.ACL_CTX_STRING);
      viMethod.setName("getVaultIdentifier");

      addSize(vault, size);
      addIDA(vault, ips.size(), threshold);

      addSliceSignature(vault, false, preSlice);
      addEncryption(vault, encryption);
      addSliceSignature(vault, true, postSlice);
      addTransposition(vault, transposition, threshold);

      for (String ip : ips)
      {
         addSliceServer(vault, ip);
      }

      // numSlicesLiteral.setValue(XmlAnySimpleType.Factory.newValue(8));

      // doc.dump();
      System.out.println(doc.xmlText(options));
      // XmlAnySimpleType value = numSlicesLiteral.addNewValue();
      // value.type = Integer.TYPE;

   }
}
