//
// Cleversafe open-source code header - Version 1.1 - December 1, 2006
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2007 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 10 W. 35th Street, 16th Floor #84,
// Chicago IL 60616
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// @author: gdhuse
//
// Date: Oct 4, 2007
//---------------------

package org.cleversafe.vault;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Random;
import java.util.UUID;
import java.util.concurrent.atomic.AtomicInteger;

import org.cleversafe.codec.Codec;
import org.cleversafe.codec.integrity.CRCIntegrityCodec;
import org.cleversafe.ida.InformationDispersalCodec;
import org.cleversafe.ida.InformationDispersalCodecBase;
import org.cleversafe.ida.InformationDispersalDecoder;
import org.cleversafe.ida.InformationDispersalEncoder;
import org.cleversafe.ida.exceptions.IDADecodeException;
import org.cleversafe.ida.exceptions.IDAEncodeException;
import org.cleversafe.ida.exceptions.IDAInvalidParametersException;
import org.cleversafe.ida.exceptions.IDANotInitializedException;
import org.cleversafe.ida.optimizedcauchy.CauchyInformationDispersalCodec;
import org.cleversafe.layer.grid.DataSlice;
import org.cleversafe.layer.grid.DataSource;
import org.cleversafe.layer.grid.GridTransaction;
import org.cleversafe.layer.grid.OperationScorecard;
import org.cleversafe.layer.grid.PlacedDataSlice;
import org.cleversafe.layer.grid.PlacedSliceName;
import org.cleversafe.layer.grid.PlacedSourceSliceSet;
import org.cleversafe.layer.grid.SliceName;
import org.cleversafe.layer.grid.SourceName;
import org.cleversafe.layer.grid.SourceSliceSet;
import org.cleversafe.layer.slicestore.NotificationHandler;
import org.cleversafe.layer.slicestore.SliceInfo;
import org.cleversafe.layer.slicestore.SliceStore;
import org.cleversafe.layer.slicestore.SliceStoreTransaction;
import org.cleversafe.layer.slicestore.exceptions.IllegalSourceNameException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreExistsException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreIOException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreQuotaException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreTransactionException;
import org.cleversafe.util.Tuple2;
import org.cleversafe.vault.exceptions.DataTransformationException;
import org.cleversafe.vault.exceptions.InformationDispersalException;
import org.cleversafe.vault.exceptions.VaultDescriptorException;
import org.cleversafe.vault.exceptions.VaultIOException;
import org.cleversafe.vault.exceptions.VaultSecurityException;
import org.junit.After;
import org.junit.Before;
import org.junit.Test;

public class BaseVaultTest
{
   // private static Logger _logger = Logger.getLogger(BaseVaultTest.class);

   private static String VAULT_TYPE = "test-vault-type";
   private static int VAULT_WIDTH = 8;
   private static int VAULT_IDA_WIDTH = 8;
   private static int VAULT_IDA_THRESHOLD = 6;
   private static int VAULT_CHUNK_SIZE = 4096;
   private static long VAULT_MAX_SLICE_SIZE = 500;
   private static long VAULT_SIZE = 1024 * 1024; 

   private static int MAX_CODEC_STACK_DEPTH = 10;

   private UUID vaultUUID;
   private InformationDispersalCodec idaCodec;
   private List<Codec> dsCodecs;
   private List<Codec> sliceCodecs;

   /**
    * Set up a random coding stack
    */
   @Before
   public void setUp() throws Exception
   {
      Random r = new Random();
      this.vaultUUID = UUID.randomUUID();

      this.idaCodec = new CauchyInformationDispersalCodec(VAULT_IDA_WIDTH, VAULT_IDA_THRESHOLD, VAULT_CHUNK_SIZE);

      this.dsCodecs = new LinkedList<Codec>();
      for (int i = r.nextInt(MAX_CODEC_STACK_DEPTH); i >= 0; --i)
      {
         this.dsCodecs.add(new CRCIntegrityCodec());
      }

      this.sliceCodecs = new LinkedList<Codec>();
      for (int i = r.nextInt(MAX_CODEC_STACK_DEPTH); i >= 0; --i)
      {
         this.sliceCodecs.add(new CRCIntegrityCodec());
      }
   }

   @After
   public void tearDown()
   {
      this.vaultUUID = null;
      this.dsCodecs = null;
      this.sliceCodecs = null;
   }

   /**
    * Test construction, getters, and setters
    */
   @Test
   public void testParameters()
   {
      MyVault v;
      List<SliceStore> sliceStores = this.buildSliceStoreList(new MySliceStoreBase());

      // Config constructor
      v = new MyVault();
      v.setVaultIdentifier(this.vaultUUID);
      v.setInformationDispersalCodec(this.idaCodec);
      v.setDatasourceCodecs(this.dsCodecs);
      v.setSliceCodecs(this.sliceCodecs);
      v.setSliceStores(sliceStores);

      assertEquals(v.getType(), VAULT_TYPE);
      assertEquals(v.getThreshold(), VAULT_IDA_THRESHOLD);
      assertEquals(v.getWidth(), VAULT_WIDTH);
      assertEquals(v.getVaultIdentifier(), this.vaultUUID);
      assertTrue(v.getDatasourceCodecs().equals(this.dsCodecs));
      assertTrue(v.getSliceCodecs().equals(this.sliceCodecs));
      assertTrue(v.getSliceStores().equals(sliceStores));
      assertEquals(v.getInformationDispersalCodec(), idaCodec);
      assertNotNull(v.toString());

      // Full constructor
      v = new MyVault(this.idaCodec, this.dsCodecs, this.sliceCodecs, sliceStores);
      v.setVaultIdentifier(this.vaultUUID);

      assertEquals(v.getType(), VAULT_TYPE);
      assertEquals(v.getThreshold(), VAULT_IDA_THRESHOLD);
      assertEquals(v.getWidth(), VAULT_WIDTH);
      assertEquals(v.getVaultIdentifier(), this.vaultUUID);
      assertTrue(v.getDatasourceCodecs().equals(this.dsCodecs));
      assertTrue(v.getSliceCodecs().equals(this.sliceCodecs));
      assertTrue(v.getSliceStores().equals(sliceStores));
      assertEquals(v.getInformationDispersalCodec(), this.idaCodec);
      assertNotNull(v.toString());

      // No codecs
      v = new MyVault();
      v.setDatasourceCodecs(null);
      v.setSliceCodecs(null);

      assertTrue(v.getDatasourceCodecs().isEmpty());
      assertTrue(v.getSliceCodecs().isEmpty());
   }

   @Test
   public void testSliceSize()
   {
      class MyCodec extends CRCIntegrityCodec
      {
         int blowupFactor;

         public MyCodec(int blowupFactor)
         {
            this.blowupFactor = blowupFactor;
         }

         public long getEncodedSize(long inputSize)
         {
            return inputSize * this.blowupFactor;
         }
      }

      // Init codec stacks with known properties
      List<Codec> dsCodecs = new ArrayList<Codec>(2);
      List<Codec> sliceCodecs = new ArrayList<Codec>(3);

      dsCodecs.add(new MyCodec(2));
      dsCodecs.add(new MyCodec(3));

      sliceCodecs.add(new MyCodec(5));
      sliceCodecs.add(new MyCodec(7));
      sliceCodecs.add(new MyCodec(11));

      MyVault v =
            new MyVault(this.idaCodec, dsCodecs, sliceCodecs,
                  this.buildSliceStoreList(new MySliceStoreBase()));

      Random r = new Random();
      for (int i = 0; i < 100; ++i)
      {
         long dataSize = r.nextLong();

         long expectedSize = dataSize * 2 * 3; // Datasource blowup
         expectedSize = this.idaCodec.getDispersedSize(expectedSize) / VAULT_IDA_WIDTH;
         expectedSize *= 5 * 7 * 11; // Slice blowup

         assertEquals(expectedSize, v.getSliceSize(dataSize));
      }
   }

   /**
    * Test session management functions
    * 
    * @throws VaultIOException
    */
   @Test
   public void testSessions() throws VaultIOException
   {
      class MySliceStore extends MySliceStoreBase
      {
         public AtomicInteger openSessions = new AtomicInteger();

         public void startSession() throws SliceStoreIOException
         {
            this.openSessions.incrementAndGet();
         }

         public void endSession() throws SliceStoreIOException
         {
            this.openSessions.decrementAndGet();
         }
      }

      MySliceStore ss = new MySliceStore();
      MyVault v =
            new MyVault(this.idaCodec, this.dsCodecs, this.sliceCodecs,
                  this.buildSliceStoreList(ss));

      assertEquals(0, ss.openSessions.get());
      v.startSessions();
      assertEquals(VAULT_WIDTH, ss.openSessions.get());
      v.endSessions();
      assertEquals(0, ss.openSessions.get());

      class StartIOErrorSliceStore extends MySliceStore
      {
         public void startSession() throws SliceStoreIOException
         {
            throw new SliceStoreIOException();
         }
      }
      v =
            new MyVault(this.idaCodec, this.dsCodecs, this.sliceCodecs,
                  this.buildSliceStoreList(new StartIOErrorSliceStore()));
      v.startSessions();

      class EndIOErrorSliceStore extends MySliceStore
      {
         public void endSession() throws SliceStoreIOException
         {
            throw new SliceStoreIOException();
         }
      }
      v =
            new MyVault(this.idaCodec, this.dsCodecs, this.sliceCodecs,
                  this.buildSliceStoreList(new EndIOErrorSliceStore()));
      v.startSessions();
      v.endSessions();
   }

   /**
    * Test coding stack and IDA
    * 
    * @throws DataTransformationException
    * @throws InformationDispersalException
    */
   @Test
   public void testCodingStack() throws DataTransformationException, InformationDispersalException
   {
      // Positive case
      MyVault v =
            new MyVault(this.idaCodec, this.dsCodecs, this.sliceCodecs,
                  this.buildSliceStoreList(new MySliceStoreBase()));

      this.helperUDC(v, new NoCorruptor());
   }

   /**
    * Test restore of corrupted slices
    * 
    * @throws DataTransformationException
    * @throws InformationDispersalException
    */
   @Test
   public void testCorruptedRestore() throws InformationDispersalException,
         DataTransformationException
   {
      MyVault v =
            new MyVault(this.idaCodec, this.dsCodecs, this.sliceCodecs,
                  this.buildSliceStoreList(new MySliceStoreBase()));

      // Positive case: Corrupt up to VAULT_IDA_WIDTH - VAULT_IDA_THRESHOLD slices
      int toCorrupt = VAULT_IDA_WIDTH - VAULT_IDA_THRESHOLD;
      this.helperUDC(v, new ByteCorruptor(toCorrupt));

      // Negative case: Corrupt VAULT_IDA_WIDTH - VAULT_IDA_THRESHOLD + 1 slices
      toCorrupt = VAULT_IDA_WIDTH - VAULT_IDA_THRESHOLD + 1;
      try
      {
         this.helperUDC(v, new ByteCorruptor(toCorrupt));
         fail("Failed to throw an exception");
      }
      catch (DataTransformationException ex)
      {
         // Good
      }
   }

   /**
    * Test restore stack in the case of missing data
    * 
    * @throws DataTransformationException
    * @throws InformationDispersalException
    */
   @Test
   public void testMissingRestore() throws InformationDispersalException,
         DataTransformationException
   {
      MyVault v =
            new MyVault(this.idaCodec, this.dsCodecs, this.sliceCodecs,
                  this.buildSliceStoreList(new MySliceStoreBase()));

      // Positive case: Remove up to VAULT_IDA_WIDTH - VAULT_IDA_THRESHOLD slices
      int numToRemove = VAULT_IDA_WIDTH - VAULT_IDA_THRESHOLD;
      this.helperUDC(v, new RemoveCorruptor(numToRemove));

      // Negative case: Remove VAULT_IDA_WIDTH - VAULT_IDA_THRESHOLD + 1slices
      numToRemove = VAULT_IDA_WIDTH - VAULT_IDA_THRESHOLD + 1;
      try
      {
         this.helperUDC(v, new RemoveCorruptor(numToRemove));
         fail("Did not throw expected exception");
      }
      catch (InformationDispersalException ex)
      {
         // Good
      }
   }

   /**
    * Test vault handling and propagation of dispersal errors
    */
   @Test
   public void testDispersalErrors() throws Exception
   {
      DataSource source = new DataSource(new SourceName("test"), 1, new byte[4096]);

      // Disersal error
      InformationDispersalCodec ida1 =
            new ErrorIDACodec(new ErrorIDAEncoder(), this.idaCodec.getDecoder());
      MyVault v1 =
            new MyVault(ida1, this.dsCodecs, this.sliceCodecs,
                  this.buildSliceStoreList(new MySliceStoreBase()));
      try
      {
         v1.disperseSource(source);
         fail("Failed to throw InformationDispersalException");
      }
      catch (InformationDispersalException ex)
      {
         // Good
      }

      // Assembly error
      InformationDispersalCodec ida2 =
            new ErrorIDACodec(this.idaCodec.getEncoder(), new ErrorIDADecoder());
      MyVault v2 =
            new MyVault(ida2, this.dsCodecs, this.sliceCodecs,
                  this.buildSliceStoreList(new MySliceStoreBase()));

      SourceSliceSet slices = v2.disperseSource(source);
      try
      {
         v2.assembleSource(slices);
         fail("Failed to throw InformationDispersalException");
      }
      catch (InformationDispersalException ex)
      {
         // Good
      }
   }

   /**
    * Test the simple DataSlice => SliceStore mapping that BaseVault provides
    */
   @Test
   public void testStoreMapping()
   {
      SourceName source = new SourceName("Test");
      SourceSliceSet slices = new SourceSliceSet(source);
      for (int i = 0; i < VAULT_IDA_WIDTH; ++i)
      {
         slices.addSlice(new DataSlice(new SliceName(source, i), -1, new byte[1024]));
      }

      List<SliceStore> sliceStores = this.buildSliceStoreList(new MySliceStoreBase());
      MyVault v = new MyVault(this.idaCodec, this.dsCodecs, this.sliceCodecs, sliceStores);
      PlacedSourceSliceSet placedSliceSet = v.mapSliceSetToStore(slices);

      List<PlacedDataSlice> placedSlices = placedSliceSet.getSlices();
      for (int i = 0; i < VAULT_IDA_WIDTH; ++i)
      {
         assertEquals(placedSlices.get(i).getStore(), sliceStores.get(i));
      }
   }

   /**
    * Tests the vault's mapping from SourceName => List<PlacedSliceName>
    */
   @Test
   public void testSliceMapping()
   {
      List<SliceStore> sliceStores = this.buildSliceStoreList(new MySliceStoreBase());
      MyVault v = new MyVault(this.idaCodec, this.dsCodecs, this.sliceCodecs, sliceStores);

      SourceName source = new SourceName("Test");
      List<PlacedSliceName> slices = v.mapSourceNameToSliceNames(source);
      for (int i = 0; i < VAULT_IDA_WIDTH; ++i)
      {
         PlacedSliceName slice = slices.get(i);
         assertEquals(slice.getStore(), sliceStores.get(i));
         assertEquals(slice.getSliceIndex(), i);
         assertEquals(slice.getSourceName(), source);
      }
   }

   /**
    * Tests the getIncompleteSources() function
    */
   @Test
   public void testGetIncompleteSources()
   {
      List<SliceStore> sliceStores = this.buildSliceStoreList(new MySliceStoreBase());
      MyVault v = new MyVault(this.idaCodec, this.dsCodecs, this.sliceCodecs, sliceStores);

      List<SourceName> sources = new ArrayList<SourceName>(7);
      for (int i = 0; i < 7; ++i)
      {
         sources.add(new SourceName("Source " + i));
      }

      // Construct a matrix with known properties
      int canLose = VAULT_IDA_WIDTH - VAULT_IDA_THRESHOLD;
      OperationScorecard  matrix = new OperationScorecard (sources, sliceStores);
      Map<SourceName, Integer> sourceStatus = new HashMap<SourceName, Integer>();

      // Source 0 - All slices present
      this.helperFillMatrixRow(matrix, sources.get(0), OperationScorecard.Status.EXISTING,
            1234, VAULT_IDA_WIDTH);

      // Source 1 - Minimum slices present
      this.helperFillMatrixRow(matrix, sources.get(1), OperationScorecard.Status.EXISTING,
            1234, VAULT_IDA_THRESHOLD);
      this.helperFillMatrixRow(matrix, sources.get(1), OperationScorecard.Status.ERROR, 1234,
            canLose);

      // Source 2 - Not enough slices - Pending
      this.helperFillMatrixRow(matrix, sources.get(2), OperationScorecard.Status.EXISTING,
            1234, VAULT_IDA_THRESHOLD - 2);
      this.helperFillMatrixRow(matrix, sources.get(2), OperationScorecard.Status.PENDING,
            1234, canLose + 2);
      sourceStatus.put(sources.get(2), 2);

      // Source 3 - Not enough slices - Error
      this.helperFillMatrixRow(matrix, sources.get(3), OperationScorecard.Status.EXISTING,
            1234, VAULT_IDA_THRESHOLD - 1);
      this.helperFillMatrixRow(matrix, sources.get(3), OperationScorecard.Status.ERROR, 1234,
            canLose + 1);
      sourceStatus.put(sources.get(3), 1);

      // Source 4 - Not enough slices - Ignored
      this.helperFillMatrixRow(matrix, sources.get(4), OperationScorecard.Status.EXISTING,
            1234, VAULT_IDA_THRESHOLD - 2);
      this.helperFillMatrixRow(matrix, sources.get(4), OperationScorecard.Status.IGNORED,
            1234, canLose + 2);
      sourceStatus.put(sources.get(4), 2);

      // Source 5 - Not enough slices - Expected
      this.helperFillMatrixRow(matrix, sources.get(5), OperationScorecard.Status.EXISTING,
            1234, VAULT_IDA_THRESHOLD - 1);
      this.helperFillMatrixRow(matrix, sources.get(5), OperationScorecard.Status.EXPECTED,
            1234, canLose + 1);
      sourceStatus.put(sources.get(5), 1);

      // Source 6 - Not enough slices - Two different transactions
      this.helperFillMatrixRow(matrix, sources.get(6), OperationScorecard.Status.EXISTING,
            1234, VAULT_IDA_THRESHOLD - 1);
      this.helperFillMatrixRow(matrix, sources.get(6), OperationScorecard.Status.EXISTING,
            4321, canLose + 1);
      sourceStatus.put(sources.get(6), 1);

      List<Tuple2<SourceName, Integer>> incompleteSources = v.getIncompleteSources(matrix);
      for (Tuple2<SourceName, Integer> tuple : incompleteSources)
      {
         SourceName source = tuple.getFirst();
         int numNeeded = tuple.getSecond();

         Integer expected = sourceStatus.remove(source);
         assertNotNull(expected);
         assertTrue(numNeeded == expected);
      }
      assertTrue(sourceStatus.isEmpty());
   }

   /**
    * Tests the base rebuildSources() implementation
    * 
    * @throws DataTransformationException
    * @throws InformationDispersalException
    */
   @Test
   public void testRebuildSources() throws InformationDispersalException,
         DataTransformationException
   {
      Random r = new Random();
      List<SliceStore> sliceStores = this.buildSliceStoreList(new MySliceStoreBase());
      MyVault v = new MyVault(this.idaCodec, this.dsCodecs, this.sliceCodecs, sliceStores);

      SourceName goodSourceName = new SourceName("Good source");
      SourceName otherSourceName = new SourceName("Other source");
      List<SourceName> sources = new ArrayList<SourceName>(2);
      sources.add(goodSourceName);
      sources.add(otherSourceName);

      long otherTxid = 4321;
      byte[] tmpData = new byte[4096];
      r.nextBytes(tmpData);
      DataSource goodSource = new DataSource(goodSourceName, 1234, tmpData);
      tmpData = new byte[4096];
      r.nextBytes(tmpData);
      DataSource otherSource = new DataSource(otherSourceName, otherTxid, tmpData);

      SourceSliceSet goodSourceSlices = v.applyDispersalStack(goodSource);
      PlacedSourceSliceSet placedGoodSourceSlices = v.mapSliceSetToStore(goodSourceSlices);
      SourceSliceSet otherSourceSlices = v.applyDispersalStack(otherSource);
      PlacedSourceSliceSet placedOtherSourceSlices = v.mapSliceSetToStore(otherSourceSlices);

      // Construct a matrix with known properties
      int canLose = VAULT_IDA_WIDTH - VAULT_IDA_THRESHOLD;
      OperationScorecard matrix;

      // Positive case
      matrix = new OperationScorecard(sources, sliceStores);
      matrix.addSourceSliceSet(placedGoodSourceSlices, OperationScorecard.Status.EXISTING);
      matrix.addSourceSliceSet(placedOtherSourceSlices, OperationScorecard.Status.EXISTING);
      try
      {
         v.rebuildSources(matrix);
         // Good
      }
      catch (Exception ex)
      {
         fail("Unexpected exception");
      }

      // Positive case - not found
      matrix = new OperationScorecard(sources, sliceStores);
      matrix.addSourceSliceSet(placedGoodSourceSlices, OperationScorecard.Status.EXISTING);
      matrix.addSourceSliceSet(placedOtherSourceSlices, OperationScorecard.Status.EXISTING);
      this.helperSetRowStatus(matrix, otherSourceName, OperationScorecard.Status.EXISTING,
            OperationScorecard.Status.EXISTING, GridTransaction.NOT_FOUND_TRANSACTION_ID,
            VAULT_IDA_THRESHOLD);
      try
      {
         v.rebuildSources(matrix);
         // Good
      }
      catch (Exception ex)
      {
         fail("Unexpected exception");
      }

      // Negative case - expected
      matrix = new OperationScorecard(sources, sliceStores);
      matrix.addSourceSliceSet(placedGoodSourceSlices, OperationScorecard.Status.EXISTING);
      matrix.addSourceSliceSet(placedOtherSourceSlices, OperationScorecard.Status.EXISTING);
      this.helperSetRowStatus(matrix, otherSourceName, OperationScorecard.Status.EXISTING,
            OperationScorecard.Status.EXPECTED, otherTxid, canLose + 1);
      try
      {
         v.rebuildSources(matrix);
         fail("Expected exception");
      }
      catch (Exception ex)
      {
         // Good
      }

      // Negative case - ignored
      matrix = new OperationScorecard(sources, sliceStores);
      matrix.addSourceSliceSet(placedGoodSourceSlices, OperationScorecard.Status.EXISTING);
      matrix.addSourceSliceSet(placedOtherSourceSlices, OperationScorecard.Status.EXISTING);
      this.helperSetRowStatus(matrix, otherSourceName, OperationScorecard.Status.EXISTING,
            OperationScorecard.Status.IGNORED, otherTxid, canLose + 1);
      try
      {
         v.rebuildSources(matrix);
         fail("Expected exception");
      }
      catch (Exception ex)
      {
         // Good
      }

      // Negative case - error
      matrix = new OperationScorecard(sources, sliceStores);
      matrix.addSourceSliceSet(placedGoodSourceSlices, OperationScorecard.Status.EXISTING);
      matrix.addSourceSliceSet(placedOtherSourceSlices, OperationScorecard.Status.EXISTING);
      this.helperSetRowStatus(matrix, otherSourceName, OperationScorecard.Status.EXISTING,
            OperationScorecard.Status.ERROR, otherTxid, canLose + 1);
      try
      {
         v.rebuildSources(matrix);
         fail("Expected exception");
      }
      catch (Exception ex)
      {
         // Good
      }

      // Negative case - pending
      matrix = new OperationScorecard(sources, sliceStores);
      matrix.addSourceSliceSet(placedGoodSourceSlices, OperationScorecard.Status.EXISTING);
      matrix.addSourceSliceSet(placedOtherSourceSlices, OperationScorecard.Status.EXISTING);
      this.helperSetRowStatus(matrix, otherSourceName, OperationScorecard.Status.EXISTING,
            OperationScorecard.Status.PENDING, otherTxid, canLose + 1);
      try
      {
         v.rebuildSources(matrix);
         fail("Expected exception");
      }
      catch (Exception ex)
      {
         // Good
      }

      // Negative case - transaction id
      matrix = new OperationScorecard(sources, sliceStores);
      matrix.addSourceSliceSet(placedGoodSourceSlices, OperationScorecard.Status.EXISTING);
      matrix.addSourceSliceSet(placedOtherSourceSlices, OperationScorecard.Status.EXISTING);
      this.helperSetRowStatus(matrix, otherSourceName, OperationScorecard.Status.EXISTING,
            OperationScorecard.Status.EXISTING, otherTxid + 1, canLose + 1);
      try
      {
         v.rebuildSources(matrix);
         fail("Expected exception");
      }
      catch (Exception ex)
      {
         // Good
      }
   }

   /**
    * Change the status and/or transactionId of entries in the row of the given matrix defined by
    * source from originalStatus to newStatus
    * 
    * @param matrix
    * @param source
    * @param originalStatus
    * @param newStatus
    * @param transactionId
    * @param count
    */
   private void helperSetRowStatus(
         OperationScorecard matrix,
         SourceName source,
         OperationScorecard.Status originalStatus,
         OperationScorecard.Status newStatus,
         long transactionId,
         int count)
   {
      int done = 0;
      for (SliceStore store : matrix.getStores())
      {
         OperationScorecard.Entry entry = matrix.getEntry(source, store);
         if (entry.getStatus() == originalStatus)
         {
            OperationScorecard.Entry newEntry =
                  new OperationScorecard.Entry(newStatus, entry.getSliceIndex(),
                        transactionId, entry.getData());
            matrix.addEntry(source, store, newEntry);

            if (++done >= count)
            {
               return;
            }
         }
      }
      assert done == count : "Not enough entries to convert";
   }

   /**
    * Create count new entries in the row defined by source with the provided status. These will be
    * created in random unoccupied columns.
    * 
    * @param matrix
    * @param source
    * @param status
    * @param count
    */
   private void helperFillMatrixRow(
         OperationScorecard matrix,
         SourceName source,
         OperationScorecard.Status status,
         long transactionId,
         int count)
   {
      // Make a list of null columns
      List<SliceStore> nullColumns = new LinkedList<SliceStore>();
      for (SliceStore store : matrix.getStores())
      {
         if (matrix.getEntry(source, store) == null)
         {
            nullColumns.add(store);
         }
      }
      assert nullColumns.size() >= count : "Not enough null entries to fill";

      Random r = new Random();
      for (int i = 0; i < count; ++i)
      {
         SliceStore store = nullColumns.remove(r.nextInt(nullColumns.size()));
         OperationScorecard.Entry entry =
               new OperationScorecard.Entry(status, i + 1, transactionId, new byte[1024]);
         matrix.addEntry(source, store, entry);
      }
   }

   /**
    * Helper function to perform a coding stack "udc" with various permutations
    */
   private void helperUDC(MyVault v, UDCCorruptor corrupter) throws DataTransformationException,
         InformationDispersalException
   {
      int numIterations = 100;
      Random r = new Random();

      for (int i = 0; i < numIterations; ++i)
      {
         byte[] data = new byte[4096];
         r.nextBytes(data);
         DataSource source = new DataSource(new SourceName("Test"), 1, data);

         SourceSliceSet dispersedSource = v.applyDispersalStack(source);
         assertEquals(dispersedSource.size(), VAULT_IDA_WIDTH);

         // Corrupt slices
         corrupter.corrupt(dispersedSource);

         DataSource newSource = v.applyAssemblyStack(dispersedSource);
         byte[] newData = newSource.getData();
         assertEquals(newData.length, data.length);

         for (int j = 0; j < data.length; ++j)
         {
            assertEquals(newData[j], data[j]);
         }
      }
   }

   /**
    * Corruptors for the helperUDC function
    */
   private interface UDCCorruptor
   {
      public void corrupt(SourceSliceSet slices);
   }
   private static class NoCorruptor implements UDCCorruptor
   {
      public void corrupt(SourceSliceSet slices)
      {
         // No corruption
      }
   }
   class ByteCorruptor implements UDCCorruptor
   {
      private int numToCorrupt;

      public ByteCorruptor(int numToCorrupt)
      {
         this.numToCorrupt = numToCorrupt;
      }

      public void corrupt(SourceSliceSet slices)
      {
         Random r = new Random();
         List<DataSlice> dataSlices = slices.getSlices();
         for (int j = 0; j < this.numToCorrupt; ++j)
         {
            DataSlice slice = dataSlices.get(j);
            byte[] sliceData = slice.getData();
            sliceData[r.nextInt(sliceData.length)] ^= 0xff;
         }
      }
   }
   class RemoveCorruptor implements UDCCorruptor
   {
      private int numToRemove;

      public RemoveCorruptor(int numToRemove)
      {
         this.numToRemove = numToRemove;
      }

      public void corrupt(SourceSliceSet slices)
      {
         List<DataSlice> dataSlices = slices.getSlices();
         for (int j = 0; j < this.numToRemove; ++j)
         {
            dataSlices.remove(dataSlices.size() - 1);
         }
      }
   }

   /**
    * Helper method to generate a list of stores from a prototype
    * 
    * @param prototype
    * @return
    */
   private List<SliceStore> buildSliceStoreList(MySliceStoreBase prototype)
   {
      List<SliceStore> sliceStores = new ArrayList<SliceStore>(VAULT_WIDTH);

      for (int i = 0; i < VAULT_WIDTH; ++i)
      {
         try
         {
            sliceStores.add((MySliceStoreBase) prototype.clone());
         }
         catch (CloneNotSupportedException ex)
         {
            fail("Internal test error");
         }
      }

      return sliceStores;
   }

   /**
    * Concrete vault to test
    */
   private class MyVault extends BaseVault
   {
      protected MyVault()
      {
         super(VAULT_TYPE);
      }

      public MyVault(
            InformationDispersalCodec idaCodec,
            List<Codec> codecs,
            List<Codec> sliceCodecs,
            List<SliceStore> sliceStores)
      {
         super(VAULT_TYPE, idaCodec, codecs, sliceCodecs, sliceStores);
      }

      public long getMaxSliceSize()
      {
         return VAULT_MAX_SLICE_SIZE;
      }

      public long getSize()
      {
         return VAULT_SIZE;
      }

      @Override
      public void setInformationDispersalCodec(InformationDispersalCodec idaCodec)
      {
         super.setInformationDispersalCodec(idaCodec);
      }

      @Override
      public void setSliceCodecs(List<Codec> sliceCodecs)
      {
         super.setSliceCodecs(sliceCodecs);
      }

      @Override
      public void setDatasourceCodecs(List<Codec> datasourceCodecs)
      {
         super.setDatasourceCodecs(datasourceCodecs);
      }

      @Override
      public void setSliceStores(List<SliceStore> sliceStores)
      {
         super.setSliceStores(sliceStores);
      }

      public void optimizeVault()
      {
         // Does nothing
      }
   }

   private static class MySliceStoreBase implements SliceStore, Cloneable
   {
      public void beginTransaction(SliceStoreTransaction transaction)
            throws SliceStoreTransactionException, SliceStoreIOException
      {
         throw new RuntimeException("Not implemented");
      }

      public void commitTransaction(SliceStoreTransaction transaction)
            throws SliceStoreTransactionException, SliceStoreIOException
      {
         throw new RuntimeException("Not implemented");
      }

      public void createStore(
            String vaultType,
            long maxSliceSize,
            long sliceStoreSize,
            VaultACL accessControlList,
            byte[] vaultDescriptorBytes,
            Map<String, String> options) throws SliceStoreExistsException, SliceStoreIOException
      {
         throw new RuntimeException("Not implemented");
      }

      public void createStore(
            String vaultType,
            long maxSliceSize,
            long sliceStoreSize,
            VaultACL accessControlList,
            byte[] vaultDescriptorBytes) throws SliceStoreExistsException, SliceStoreIOException
      {
         throw new RuntimeException("Not implemented");
      }

      public SliceStoreTransaction createTransaction(long transactionId)
            throws SliceStoreTransactionException, SliceStoreIOException
      {
         throw new RuntimeException("Not implemented");
      }

      public void deleteStore() throws SliceStoreIOException
      {
         throw new RuntimeException("Not implemented");
      }

      public void updateStore(
            String vaultType,
            long maxSliceStize,
            long sliceStoreSize,
            VaultACL accessControlList,
            byte[] vaultDescriptorBytes) throws SliceStoreIOException
      {
         throw new RuntimeException("Not implemented");
      }

      public void endSession() throws SliceStoreIOException
      {
         throw new RuntimeException("Not implemented");
      }

      public boolean exists(SliceName name) throws SliceStoreIOException,
            IllegalSourceNameException
      {
         throw new RuntimeException("Not implemented");
      }

      public String getIdentification()
      {
         throw new RuntimeException("Not implemented");
      }

      public SliceInfo getSliceInfo(SliceName name) throws SliceStoreIOException,
            IllegalSourceNameException
      {
         throw new RuntimeException("Not implemented");
      }

      public SliceStoreTransaction getTransaction(long transactionId)
            throws SliceStoreTransactionException, SliceStoreIOException
      {
         throw new RuntimeException("Not implemented");
      }

      public VaultDescriptor getVaultDescriptor() throws SliceStoreIOException, VaultIOException,
            VaultSecurityException, VaultDescriptorException
      {
         throw new RuntimeException("Not implemented");
      }

      public boolean isActiveTransaction(SliceStoreTransaction transaction)
      {
         throw new RuntimeException("Not implemented");
      }

      public boolean isOperational()
      {
         throw new RuntimeException("Not implemented");
      }

      public void listBegin() throws SliceStoreIOException
      {
         throw new RuntimeException("Not implemented");
      }

      public void listBegin(SliceName name) throws SliceStoreIOException,
            IllegalSourceNameException
      {
         throw new RuntimeException("Not implemented");
      }

      public List<SliceInfo> listContinue() throws SliceStoreIOException
      {
         throw new RuntimeException("Not implemented");
      }

      public boolean listInProgress()
      {
         throw new RuntimeException("Not implemented");
      }

      public void listStop() throws SliceStoreIOException
      {
         throw new RuntimeException("Not implemented");
      }

      public DataSlice read(SliceName name) throws SliceStoreIOException,
            IllegalSourceNameException
      {
         throw new RuntimeException("Not implemented");
      }

      public List<DataSlice> read(List<SliceName> names) throws SliceStoreIOException,
            IllegalSourceNameException
      {
         throw new RuntimeException("Not implemented");
      }

      public void registerForNorNotification(NotificationHandler handler)
            throws SliceStoreIOException
      {
         throw new RuntimeException("Not implemented");
      }

      public boolean remove(SliceName name) throws SliceStoreIOException,
            IllegalSourceNameException
      {
         throw new RuntimeException("Not implemented");
      }

      public boolean[] remove(List<SliceName> names) throws SliceStoreIOException,
            IllegalSourceNameException
      {
         throw new RuntimeException("Not implemented");
      }

      public void rollbackTransaction(SliceStoreTransaction transaction)
            throws SliceStoreTransactionException, SliceStoreIOException
      {
         throw new RuntimeException("Not implemented");
      }

      public void startSession() throws SliceStoreIOException
      {
         throw new RuntimeException("Not implemented");
      }

      public List<SliceInfo> verifyIntegrity() throws VaultDescriptorException,
            SliceStoreIOException, IllegalSourceNameException, VaultIOException,
            VaultSecurityException
      {
         throw new RuntimeException("Not implemented");
      }

      public void write(DataSlice data) throws SliceStoreIOException, IllegalSourceNameException,
            SliceStoreTransactionException, SliceStoreQuotaException
      {
         throw new RuntimeException("Not implemented");
      }

      public void write(DataSlice data, boolean allowOverwriteNewer) throws SliceStoreIOException,
            IllegalSourceNameException, SliceStoreTransactionException, SliceStoreQuotaException
      {
         throw new RuntimeException("Not implemented");
      }

      public void write(List<DataSlice> dataSlices) throws SliceStoreIOException,
            IllegalSourceNameException, SliceStoreTransactionException, SliceStoreQuotaException
      {
         throw new RuntimeException("Not implemented");
      }

      public void write(List<DataSlice> dataSlices, boolean allowOverwriteNewer)
            throws SliceStoreIOException, IllegalSourceNameException,
            SliceStoreTransactionException, SliceStoreQuotaException
      {
         throw new RuntimeException("Not implemented");
      }

      public Object clone() throws CloneNotSupportedException
      {
         return super.clone();
      }

      public List<SliceInfo> verifyIntegrity(List<SliceInfo> slices)
            throws VaultDescriptorException, SliceStoreIOException, IllegalSourceNameException,
            VaultIOException, VaultSecurityException
      {
         throw new RuntimeException("Not implemented");
      }
   }

   /**
    * IDA Encoder that throws an IDAEncodeException
    */
   class ErrorIDAEncoder implements InformationDispersalEncoder
   {
      public List<byte[]> finish(byte[] buffer) throws IDAEncodeException,
            IDANotInitializedException
      {
         throw new IDAEncodeException("Test")
         {
            private static final long serialVersionUID = -445970325479807272L;
         };
      }

      public int getNumSlices()
      {
         return VAULT_IDA_WIDTH;
      }

      public int getThreshold()
      {
         return VAULT_IDA_THRESHOLD;
      }

      public int getChunkSize()
      {
         throw new RuntimeException("Test - not supported");
      }
      
      public void initialize() throws IDAInvalidParametersException
      {
      }

      public void setNumSlices(int numSlices)
      {
         throw new RuntimeException("Test - not supported");
      }

      public void setThreshold(int threshold)
      {
         throw new RuntimeException("Test - not supported");
      }
      


      public void setChunkSize(int chunkSize)
      {
         throw new RuntimeException("Test - not supported");
      }

      public List<byte[]> update(byte[] buffer) throws IDAEncodeException,
            IDANotInitializedException
      {
         throw new IDAEncodeException("Test")
         {
            private static final long serialVersionUID = -4488926601269463281L;
         };
      }

   }

   /**
    * IDA decoder that throws IDADecodeExceptions
    */
   class ErrorIDADecoder implements InformationDispersalDecoder
   {
      public byte[] finish(List<byte[]> encodedBuffers) throws IDADecodeException,
            IDANotInitializedException
      {
         throw new IDADecodeException("Test")
         {
            private static final long serialVersionUID = -8581093386940536832L;
         };
      }

      public int getNumSlices()
      {
         return VAULT_IDA_WIDTH;
      }

      public int getThreshold()
      {
         return VAULT_IDA_THRESHOLD;
      }
      
      public int getChunkSize()
      {
         throw new RuntimeException("Test - not supported");
      }

      public void initialize() throws IDAInvalidParametersException
      {
      }

      public void setNumSlices(int numSlices)
      {
         throw new RuntimeException("Test - not supported");
      }

      public void setThreshold(int threshold)
      {
         throw new RuntimeException("Test - not supported");
      }

      public void setChunkSize(int chunkSize)
      {
         throw new RuntimeException("Test - not supported");
      }
      
      public byte[] update(List<byte[]> encodedBuffers) throws IDADecodeException,
            IDANotInitializedException
      {
         throw new IDADecodeException("Test")
         {
            private static final long serialVersionUID = -8581093386940536832L;
         };
      }

   }

   class ErrorIDACodec extends InformationDispersalCodecBase
   {
      InformationDispersalEncoder encoder;
      InformationDispersalDecoder decoder;
      
      public ErrorIDACodec(InformationDispersalEncoder encoder, InformationDispersalDecoder decoder)
      {
         this.encoder = encoder;
         this.decoder = decoder;
      }

      public float getBlowup()
      {
         throw new RuntimeException("Test - not supported");
      }

      public long getDispersedSize(long inputSize)
      {
         throw new RuntimeException("Test - not supported");
      }

      public String getName()
      {
         return "Test - Custom";
      }

      @Override
      public InformationDispersalDecoder getDecoder()
      {
         return this.decoder;
      }

      @Override
      public InformationDispersalEncoder getEncoder()
      {
         return this.encoder;
      }
      
      @Override
      protected InformationDispersalDecoder getNewDecoder()
      {
         return this.decoder;
      }

      @Override
      protected InformationDispersalEncoder getNewEncoder()
      {
         return this.encoder;
      }
   }
}
