//
// Cleversafe open-source code header - Version 1.1 - December 1, 2006
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2007 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 10 W. 35th Street, 16th Floor #84,
// Chicago IL 60616
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// Author: Vance Thornton
//
//---------------------

package org.cleversafe.util.math;

//--------------------------------------
//Systematically generate combinations.
//--------------------------------------

import java.math.BigInteger;

public class CombinationGenerator
{

   private int[] a;

   private int n;

   private int r;

   private BigInteger numLeft;

   private BigInteger total;

   // ------------
   // Constructor
   // ------------

   public CombinationGenerator(int n, int r)
   {
      if (r > n)
      {
         throw new IllegalArgumentException();
      }
      if (n < 1)
      {
         throw new IllegalArgumentException();
      }
      this.n = n;
      this.r = r;
      a = new int[r];
      BigInteger nFact = getFactorial(n);
      BigInteger rFact = getFactorial(r);
      BigInteger nminusrFact = getFactorial(n - r);
      total = nFact.divide(rFact.multiply(nminusrFact));
      reset();
   }

   // ------
   // Reset
   // ------

   public void reset()
   {
      for (int i = 0; i < a.length; i++)
      {
         a[i] = i;
      }
      numLeft = new BigInteger(total.toString());
   }

   // ------------------------------------------------
   // Return number of combinations not yet generated
   // ------------------------------------------------

   public BigInteger getNumLeft()
   {
      return numLeft;
   }

   // -----------------------------
   // Are there more combinations?
   // -----------------------------

   public boolean hasMore()
   {
      return numLeft.compareTo(BigInteger.ZERO) == 1;
   }

   // ------------------------------------
   // Return total number of combinations
   // ------------------------------------

   public BigInteger getTotal()
   {
      return total;
   }

   // ------------------
   // Compute factorial
   // ------------------

   private static BigInteger getFactorial(int n)
   {
      BigInteger fact = BigInteger.ONE;
      for (int i = n; i > 1; i--)
      {
         fact = fact.multiply(new BigInteger(Integer.toString(i)));
      }
      return fact;
   }

   // --------------------------------------------------------
   // Generate next combination (algorithm from Rosen p. 286)
   // --------------------------------------------------------

   public int[] getNext()
   {

      if (numLeft.equals(total))
      {
         numLeft = numLeft.subtract(BigInteger.ONE);
         // Shallow copy (.clone()) works because array is a one dimensional
         // primitive type. See http://www.javapractices.com/Topic3.cjp
         // - MM
         return a.clone();
      }

      int i = r - 1;
      while (a[i] == n - r + i)
      {
         i--;
      }
      a[i] = a[i] + 1;
      for (int j = i + 1; j < r; j++)
      {
         a[j] = a[i] + j - i;
      }

      numLeft = numLeft.subtract(BigInteger.ONE);
      return a.clone();
   }
}
