//
// Cleversafe open-source code header - Version 1.1 - December 1, 2006
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2007 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 10 W. 35th Street, 16th Floor #84,
// Chicago IL 60616
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// Author: mmotwani
//
// Date: May 1, 2007
//---------------------

package org.cleversafe.serialization;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;

import org.apache.log4j.Logger;
import org.cleversafe.layer.grid.DataSlice;
import org.cleversafe.layer.grid.SliceName;
import org.cleversafe.layer.protocol.BeginSessionRequest;
import org.cleversafe.layer.protocol.BeginSessionResponse;
import org.cleversafe.layer.protocol.BeginTransactionRequest;
import org.cleversafe.layer.protocol.BeginTransactionResponse;
import org.cleversafe.layer.protocol.CommitTransactionRequest;
import org.cleversafe.layer.protocol.CommitTransactionResponse;
import org.cleversafe.layer.protocol.EndSessionRequest;
import org.cleversafe.layer.protocol.EndSessionResponse;
import org.cleversafe.layer.protocol.ErrorResponse;
import org.cleversafe.layer.protocol.ErrorUnsolicited;
import org.cleversafe.layer.protocol.GridProtocolOperation;
import org.cleversafe.layer.protocol.MultipleReadRequest;
import org.cleversafe.layer.protocol.MultipleReadResponse;
import org.cleversafe.layer.protocol.MultipleRemoveRequest;
import org.cleversafe.layer.protocol.MultipleRemoveResponse;
import org.cleversafe.layer.protocol.MultipleWriteRequest;
import org.cleversafe.layer.protocol.MultipleWriteResponse;
import org.cleversafe.layer.protocol.NoopRequest;
import org.cleversafe.layer.protocol.NoopResponse;
import org.cleversafe.layer.protocol.ProtocolMessage;
import org.cleversafe.layer.protocol.RegisterForNotificationsRequest;
import org.cleversafe.layer.protocol.RegisterForNotificationsResponse;
import org.cleversafe.layer.protocol.Request;
import org.cleversafe.layer.protocol.Response;
import org.cleversafe.layer.protocol.RollbackTransactionRequest;
import org.cleversafe.layer.protocol.RollbackTransactionResponse;
import org.cleversafe.layer.protocol.SequencedProtocolMessage;
import org.cleversafe.layer.protocol.UnsolicitedMessage;
import org.cleversafe.layer.protocol.WriteNotifyUnsolicited;
import org.cleversafe.layer.protocol.exceptions.OperationAlreadyRegisteredException;
import org.cleversafe.layer.protocol.exceptions.OperationNotRegisteredException;
import org.cleversafe.layer.protocol.exceptions.ProtocolDeserializationException;
import org.cleversafe.layer.protocol.exceptions.ProtocolSerializationException;
import org.cleversafe.serialization.asn1.ASN1GridProtocolMessageFactory;
import org.junit.Test;

/**
 * This class unit tests the ProtocolMessageFactory class.
 * 
 * @author Manish Motwani
 * 
 */
public abstract class ProtocolMessageFactoryTestBase
{
   private static Logger _logger = Logger.getLogger(ProtocolMessageFactoryTestBase.class);

   private static ProtocolMessageFactory messageFactory = null;

   private int sequenceNumber = 0;
   private static int SLICE_BLOCK_SIZE = 353;

   public ProtocolMessageFactoryTestBase()
   {
      messageFactory = getProtocolMessageFactory();
   }

   protected abstract ProtocolMessageFactory getProtocolMessageFactory();

   private void genericProtocolMessageTest(SequencedProtocolMessage message)
   {
      int oldSequenceNumber = message.getSequenceNumber();
      message.setSequenceNumber(this.sequenceNumber);
      assertEquals(message.getSequenceNumber(), this.sequenceNumber);
      this.sequenceNumber++;

      if (message instanceof Request)
      {
         assertEquals(message.isRequest(), true);
         assertEquals(message.isResponse(), false);
         assertEquals(message.isUnsolicited(), false);
      }
      else if (message instanceof Response)
      {
         assertEquals(message.isRequest(), false);
         assertEquals(message.isResponse(), true);
         assertEquals(message.isUnsolicited(), false);

         Response response = (Response) message;
         Exception exception = new Exception("Test Exception");

         assertEquals(response.getExceptionFlag(), false);
         assertEquals(response.getException(), null);
         response.setException(exception);
         assertEquals(response.getExceptionFlag(), true);
         assertEquals(response.getException(), exception);
      }
      else if (message instanceof UnsolicitedMessage)
      {
         assertEquals(message.isRequest(), false);
         assertEquals(message.isResponse(), false);
         assertEquals(message.isUnsolicited(), true);
      }

      message.setSequenceNumber(oldSequenceNumber);
   }

   /**
    * This method tests the serialize() and deserialize() methods using a BeginSessionRequest
    * operation type.
    * 
    */
   @Test
   public void testBeginSessionRequestSerializeDeserialize()
   {
      BeginSessionRequest beginSessionRequest = new BeginSessionRequest();
      genericProtocolMessageTest(beginSessionRequest);

      try
      {
         byte[] serializedBeginSessionRequest = serialize(beginSessionRequest);
         BeginSessionRequest testedBeginSessionRequest =
               (BeginSessionRequest) deserialize(
                     beginSessionRequest.getOperation().getOperationCode(),
                     serializedBeginSessionRequest);
         assertEquals(beginSessionRequest.toString(), testedBeginSessionRequest.toString());
      }
      catch (Exception e)
      {
         fail(e.getMessage());
      }
   }

   /**
    * This method tests the serialize() and deserialize() methods using a BeginSessionResponse
    * operation type.
    * 
    */
   @Test
   public void testBeginSessionResponseSerializeDeserialize()
   {
      int sessionID = 10234;
      BeginSessionResponse beginSessionResponse = new BeginSessionResponse(sessionID);
      genericProtocolMessageTest(beginSessionResponse);

      assertEquals(beginSessionResponse.getSessionID(), sessionID);
      beginSessionResponse.setSessionID(sessionID + 1);
      assertEquals(beginSessionResponse.getSessionID(), sessionID + 1);

      try
      {
         byte[] serializedBeginSessionResponse = serialize(beginSessionResponse);

         BeginSessionResponse testedBeginSessionResponse =
               (BeginSessionResponse) deserialize(
                     beginSessionResponse.getOperation().getOperationCode(),
                     serializedBeginSessionResponse);
         assertEquals(beginSessionResponse.toString(), testedBeginSessionResponse.toString());
      }
      catch (Exception e)
      {
         System.out.println(e.getMessage());
         fail(e.getMessage());
      }
   }

   /**
    * This method tests the serialize() and deserialize() methods using a BeginTransactionRequest
    * operation type.
    * 
    */
   @Test
   public void testBeginTransactionRequestSerializeDeserialize()
   {
      long txid = 10419;
      BeginTransactionRequest beginTransactionRequest = new BeginTransactionRequest(txid);
      genericProtocolMessageTest(beginTransactionRequest);

      assertEquals(beginTransactionRequest.getTransactionID(), txid);
      beginTransactionRequest.setTransactionID(txid + 1);
      assertEquals(beginTransactionRequest.getTransactionID(), (txid + 1));

      try
      {
         byte[] serializedBeginTransactionRequest = serialize(beginTransactionRequest);
         BeginTransactionRequest testedBeginTransactionRequest =
               (BeginTransactionRequest) deserialize(
                     beginTransactionRequest.getOperation().getOperationCode(),
                     serializedBeginTransactionRequest);
         assertEquals(beginTransactionRequest.toString(), testedBeginTransactionRequest.toString());
      }
      catch (Exception e)
      {
         System.out.println(e.getMessage());
         fail(e.getMessage());
      }
   }

   /**
    * This method tests the serialize() and deserialize() methods using a BeginTransactionResponse
    * operation type.
    * 
    */
   @Test
   public void testBeginTransactionResponseSerializeDeserialize()
   {
      BeginTransactionResponse beginTransactionResponse = new BeginTransactionResponse();
      genericProtocolMessageTest(beginTransactionResponse);

      try
      {
         byte[] serializedBeginTransactionResponse = serialize(beginTransactionResponse);
         BeginTransactionResponse testedBeginTransactionResponse =
               (BeginTransactionResponse) deserialize(
                     beginTransactionResponse.getOperation().getOperationCode(),
                     serializedBeginTransactionResponse);
         assertEquals(beginTransactionResponse.toString(),
               testedBeginTransactionResponse.toString());
      }
      catch (Exception e)
      {
         System.out.println(e.getMessage());
         fail(e.getMessage());
      }
   }

   /**
    * This method tests the serialize() and deserialize() methods using a CommitTransactionRequest
    * operation type.
    * 
    */
   @Test
   public void testCommitTransactionRequestSerializeDeserialize()
   {
      long txid = 10340;
      CommitTransactionRequest commitTransactionRequest = new CommitTransactionRequest(txid);
      genericProtocolMessageTest(commitTransactionRequest);

      assertEquals(commitTransactionRequest.getTransactionID(), txid);
      commitTransactionRequest.setTransactionID(txid + 1);
      assertEquals(commitTransactionRequest.getTransactionID(), txid + 1);

      try
      {
         byte[] serializedCommitTransactionRequest = serialize(commitTransactionRequest);
         CommitTransactionRequest testedCommitTransactionRequest =
               (CommitTransactionRequest) deserialize(
                     commitTransactionRequest.getOperation().getOperationCode(),
                     serializedCommitTransactionRequest);
         assertEquals(commitTransactionRequest.toString(),
               testedCommitTransactionRequest.toString());
      }
      catch (Exception e)
      {
         System.out.println(e.getMessage());
         fail(e.getMessage());
      }
   }

   /**
    * This method tests the serialize() and deserialize() methods using a CommitTransactionResponse
    * operation type.
    * 
    */
   @Test
   public void testCommitTransactionResponseSerializeDeserialize()
   {
      CommitTransactionResponse commitTransactionResponse = new CommitTransactionResponse();
      genericProtocolMessageTest(commitTransactionResponse);

      try
      {
         byte[] serializedCommitTransactionResponse = serialize(commitTransactionResponse);
         CommitTransactionResponse testedCommitTransactionResponse =
               (CommitTransactionResponse) deserialize(
                     commitTransactionResponse.getOperation().getOperationCode(),
                     serializedCommitTransactionResponse);
         assertEquals(commitTransactionResponse.toString(),
               testedCommitTransactionResponse.toString());
      }
      catch (Exception e)
      {
         System.out.println(e.getMessage());
         fail(e.getMessage());
      }
   }

   /**
    * This method tests the serialize() and deserialize() methods using a EndSessionRequest
    * operation type.
    * 
    */
   @Test
   public void testEndSessionRequestSerializeDeserialize()
   {
      int sessionID = 13441;
      EndSessionRequest endSessionRequest = new EndSessionRequest(sessionID);
      genericProtocolMessageTest(endSessionRequest);

      assertEquals(endSessionRequest.getSessionID(), sessionID);
      endSessionRequest.setSessionID(sessionID + 1);
      assertEquals(endSessionRequest.getSessionID(), sessionID + 1);

      try
      {
         byte[] serializedEndSessionRequest = serialize(endSessionRequest);
         EndSessionRequest testedEndSessionRequest =
               (EndSessionRequest) deserialize(endSessionRequest.getOperation().getOperationCode(),
                     serializedEndSessionRequest);
         assertEquals(endSessionRequest.toString(), testedEndSessionRequest.toString());
      }
      catch (Exception e)
      {
         System.out.println(e.getMessage());
         fail(e.getMessage());
      }
   }

   /**
    * This method tests the serialize() and deserialize() methods using a EndSessionResponse
    * operation type.
    * 
    */
   @Test
   public void testEndSessionResponseSerializeDeserialize()
   {
      EndSessionResponse endSessionResponse = new EndSessionResponse();
      genericProtocolMessageTest(endSessionResponse);

      try
      {
         byte[] serializedEndSessionResponse = serialize(endSessionResponse);
         EndSessionResponse testedEndSessionResponse =
               (EndSessionResponse) deserialize(
                     endSessionResponse.getOperation().getOperationCode(),
                     serializedEndSessionResponse);
         assertEquals(endSessionResponse.toString(), testedEndSessionResponse.toString());
      }
      catch (Exception e)
      {
         System.out.println(e.getMessage());
         fail(e.getMessage());
      }
   }

   /**
    * This method tests the serialize() and deserialize() methods using a ErrorResponse operation
    * type.
    * 
    */
   @Test
   public void testErrorResponseSerializeDeserialize()
   {
      ErrorResponse errorResponse = new ErrorResponse();
      genericProtocolMessageTest(errorResponse);

      errorResponse.setException(new Exception("This is an error!"));

      try
      {
         byte[] serializedErrorResponse = serialize(errorResponse);
         ErrorResponse testedErrorResponse =
               (ErrorResponse) deserialize(errorResponse.getOperation().getOperationCode(),
                     serializedErrorResponse);
         _logger.info("Input: " + errorResponse.toString());
         _logger.info("Output: " + testedErrorResponse.toString());
         assertEquals(errorResponse.toString(), testedErrorResponse.toString());
      }
      catch (Exception e)
      {
         System.out.println(e.getMessage());
         fail(e.getMessage());
      }
   }

   /**
    * This method tests the serialize() and deserialize() methods using a NoopRequest operation
    * type.
    * 
    */
   @Test
   public void testNoopRequestSerializeDeserialize()
   {
      NoopRequest noopRequest = new NoopRequest();
      genericProtocolMessageTest(noopRequest);

      try
      {
         byte[] serializedNoopRequest = serialize(noopRequest);
         NoopRequest testedNoopRequest =
               (NoopRequest) deserialize(noopRequest.getOperation().getOperationCode(),
                     serializedNoopRequest);
         assertEquals(noopRequest.toString(), testedNoopRequest.toString());
      }
      catch (Exception e)
      {
         System.out.println(e.getMessage());
         fail(e.getMessage());
      }
   }

   /**
    * This method tests the serialize() and deserialize() methods using a NoopResponse operation
    * type.
    * 
    */
   @Test
   public void testNoopResponseSerializeDeserialize()
   {
      NoopResponse noopResponse = new NoopResponse();
      genericProtocolMessageTest(noopResponse);

      try
      {
         byte[] serializedNoopResponse = serialize(noopResponse);
         NoopResponse testedNoopResponse =
               (NoopResponse) deserialize(noopResponse.getOperation().getOperationCode(),
                     serializedNoopResponse);
         assertEquals(noopResponse.toString(), testedNoopResponse.toString());
      }
      catch (Exception e)
      {
         System.out.println(e.getMessage());
         fail(e.getMessage());
      }
   }

   // /**
   // * This method tests the serialize() and deserialize() methods using a ReadRequest operation
   // * type.
   // *
   // */
   // @Test
   // public void testReadRequestSerializeDeserialize()
   // {
   // String sliceID = "10341-AFAIK";
   // ReadRequest readRequest = new ReadRequest(new SliceName(sliceID, 3));
   // genericProtocolMessageTest(readRequest);
   //
   // try
   // {
   // byte[] serializedReadRequest = serialize(readRequest);
   // ReadRequest testedReadRequest =
   // (ReadRequest) messageFactory.deserialize(
   // readRequest.getOperation().getOperationCode(), serializedReadRequest);
   // assertEquals(readRequest.toString(), testedReadRequest.toString());
   // assertEquals(readRequest.getSliceName(), testedReadRequest.getSliceName());
   // }
   // catch (Exception e)
   // {
   // System.out.println(e.getMessage());
   // fail(e.getMessage());
   // }
   // }

   // /**
   // * This method tests the serialize() and deserialize() methods using a ReadResponse operation
   // * type.
   // *
   // */
   // @Test
   // public void testReadResponseSerializeDeserialize()
   // {
   // int txid = 1044;
   // byte[] buf = new byte[4096];
   // randomizeBuffer(buf);
   // String sliceID = "10341-AFAIK";
   // ReadResponse readResponse =
   // new ReadResponse(new DataSlice(new SliceName(sliceID, 3), txid, buf));
   // genericProtocolMessageTest(readResponse);
   //
   // try
   // {
   // byte[] serializedReadResponse = serialize(readResponse);
   // ReadResponse testedReadResponse =
   // (ReadResponse) messageFactory.deserialize(
   // readResponse.getOperation().getOperationCode(), serializedReadResponse);
   // assertEquals(readResponse.toString(), testedReadResponse.toString());
   // }
   // catch (Exception e)
   // {
   // System.out.println(e.getMessage());
   // fail(e.getMessage());
   // }
   //
   // // Try a second test with an empty ReadResponse
   // ReadResponse readResponse2 = new ReadResponse();
   // genericProtocolMessageTest(readResponse2);
   //
   // try
   // {
   // byte[] serializedReadResponse = serialize(readResponse2);
   // ReadResponse testedReadResponse =
   // (ReadResponse) messageFactory.deserialize(
   // readResponse2.getOperation().getOperationCode(), serializedReadResponse);
   //
   // assertEquals(readResponse2.toString(), testedReadResponse.toString());
   // }
   // catch (Exception e)
   // {
   // System.out.println(e.getMessage());
   // fail(e.getMessage());
   // }
   // }

   /**
    * This method tests the serialize() and deserialize() methods using a
    * RegisterForNotificationsRequest operation type.
    * 
    */
   @Test
   public void testRegisterForNotificationsRequestSerializeDeserialize()
   {
      // True
      RegisterForNotificationsRequest registerForNotificationsRequest =
            new RegisterForNotificationsRequest(true);
      genericProtocolMessageTest(registerForNotificationsRequest);

      try
      {
         byte[] serializedRegisterForNotificationsRequest =
               serialize(registerForNotificationsRequest);
         RegisterForNotificationsRequest testedRegisterForNotificationsRequest =
               (RegisterForNotificationsRequest) deserialize(
                     registerForNotificationsRequest.getOperation().getOperationCode(),
                     serializedRegisterForNotificationsRequest);
         assertEquals(registerForNotificationsRequest.toString(),
               testedRegisterForNotificationsRequest.toString());
         _logger.info(registerForNotificationsRequest.toString());
      }
      catch (Exception e)
      {
         System.out.println(e.getMessage());
         fail(e.getMessage());
      }

      // False
      RegisterForNotificationsRequest registerForNotificationsRequest2 =
            new RegisterForNotificationsRequest(false);
      genericProtocolMessageTest(registerForNotificationsRequest2);

      try
      {
         byte[] serializedRegisterForNotificationsRequest =
               serialize(registerForNotificationsRequest2);
         RegisterForNotificationsRequest testedRegisterForNotificationsRequest =
               (RegisterForNotificationsRequest) deserialize(
                     registerForNotificationsRequest2.getOperation().getOperationCode(),
                     serializedRegisterForNotificationsRequest);
         assertEquals(registerForNotificationsRequest2.toString(),
               testedRegisterForNotificationsRequest.toString());
         _logger.info(registerForNotificationsRequest2.toString());
      }
      catch (Exception e)
      {
         System.out.println(e.getMessage());
         fail(e.getMessage());
      }
   }

   /**
    * This method tests the serialize() and deserialize() methods using a
    * RegisterForNotificationsResponse operation type.
    * 
    */
   @Test
   public void testRegisterForNotificationsResponseSerializeDeserialize()
   {
      RegisterForNotificationsResponse registerForNotificationsResponse =
            new RegisterForNotificationsResponse();
      genericProtocolMessageTest(registerForNotificationsResponse);

      try
      {
         byte[] serializedRegisterForNotificationsResponse =
               serialize(registerForNotificationsResponse);
         RegisterForNotificationsResponse testedRegisterForNotificationsResponse =
               (RegisterForNotificationsResponse) deserialize(
                     registerForNotificationsResponse.getOperation().getOperationCode(),
                     serializedRegisterForNotificationsResponse);
         assertEquals(registerForNotificationsResponse.toString(),
               testedRegisterForNotificationsResponse.toString());
      }
      catch (Exception e)
      {
         System.out.println(e.getMessage());
         fail(e.getMessage());
      }
   }

   // /**
   // * This method tests the serialize() and deserialize() methods using a RemoveRequest operation
   // * type.
   // *
   // */
   // @Test
   // public void testRemoveRequestSerializeDeserialize()
   // {
   // String sliceID = "AS3F-A23UMB";
   // RemoveRequest removeRequest = new RemoveRequest(new SliceName(sliceID, 5));
   // genericProtocolMessageTest(removeRequest);
   //
   // try
   // {
   // byte[] serializedRemoveRequest = serialize(removeRequest);
   // RemoveRequest testedRemoveRequest =
   // (RemoveRequest) messageFactory.deserialize(
   // removeRequest.getOperation().getOperationCode(), serializedRemoveRequest);
   // assertEquals(removeRequest.toString(), testedRemoveRequest.toString());
   // assertEquals(removeRequest.getSliceName(), testedRemoveRequest.getSliceName());
   // }
   // catch (Exception e)
   // {
   // System.out.println(e.getMessage());
   // fail(e.getMessage());
   // }
   // }

   // /**
   // * This method tests the serialize() and deserialize() methods using a RemoveResponse operation
   // * type.
   // *
   // */
   // @Test
   // public void testRemoveResponseSerializeDeserialize()
   // {
   // RemoveResponse removeResponse = new RemoveResponse(true);
   // genericProtocolMessageTest(removeResponse);
   //
   // try
   // {
   // byte[] serializedRemoveResponse = serialize(removeResponse);
   // RemoveResponse testedRemoveResponse =
   // (RemoveResponse) messageFactory.deserialize(
   // removeResponse.getOperation().getOperationCode(), serializedRemoveResponse);
   // assertEquals(removeResponse.toString(), testedRemoveResponse.toString());
   // }
   // catch (Exception e)
   // {
   // System.out.println(e.getMessage());
   // fail(e.getMessage());
   // }
   // }

   /**
    * This method tests the serialize() and deserialize() methods using a RollbackTransactionRequest
    * operation type.
    * 
    */
   @Test
   public void testRollbackTransactionRequestSerializeDeserialize()
   {
      long txid = 1034471;
      RollbackTransactionRequest rollbackTransactionRequest = new RollbackTransactionRequest(txid);
      genericProtocolMessageTest(rollbackTransactionRequest);

      assertEquals(rollbackTransactionRequest.getTransactionID(), txid);
      rollbackTransactionRequest.setTransactionID(txid + 1);
      assertEquals(rollbackTransactionRequest.getTransactionID(), txid + 1);

      try
      {
         byte[] serializedRollbackTransactionRequest = serialize(rollbackTransactionRequest);
         RollbackTransactionRequest testedRollbackTransactionRequest =
               (RollbackTransactionRequest) deserialize(
                     rollbackTransactionRequest.getOperation().getOperationCode(),
                     serializedRollbackTransactionRequest);
         assertEquals(rollbackTransactionRequest.toString(),
               testedRollbackTransactionRequest.toString());
      }
      catch (Exception e)
      {
         System.out.println(e.getMessage());
         fail(e.getMessage());
      }
   }

   /**
    * This method tests the serialize() and deserialize() methods using a
    * RollbackTransactionResponse operation type.
    * 
    */
   @Test
   public void testRollbackTransactionResponseSerializeDeserialize()
   {
      RollbackTransactionResponse rollbackTransactionResponse = new RollbackTransactionResponse();
      genericProtocolMessageTest(rollbackTransactionResponse);

      try
      {
         byte[] serializedRollbackTransactionResponse = serialize(rollbackTransactionResponse);
         RollbackTransactionResponse testedRollbackTransactionResponse =
               (RollbackTransactionResponse) deserialize(
                     rollbackTransactionResponse.getOperation().getOperationCode(),
                     serializedRollbackTransactionResponse);
         assertEquals(rollbackTransactionResponse.toString(),
               testedRollbackTransactionResponse.toString());
      }
      catch (Exception e)
      {
         System.out.println(e.getMessage());
         fail(e.getMessage());
      }
   }

   /**
    * This method tests the serialize() and deserialize() methods using a UnsolicitedError operation
    * type.
    * 
    */
   @Test
   public void testErrorUnsolicitedSerializeDeserialize()
   {
      // Exception
      ErrorUnsolicited unsolicitedError = new ErrorUnsolicited();
      assertEquals(unsolicitedError.getException(), null);
      Exception exception = new Exception("My new unsolicited error!");
      unsolicitedError.setException(exception);
      assertEquals(unsolicitedError.getException(), exception);
      genericProtocolMessageTest(unsolicitedError);

      try
      {
         byte[] serializedUnsolicitedError = serialize(unsolicitedError);
         ErrorUnsolicited testedUnsolicitedError =
               (ErrorUnsolicited) deserialize(unsolicitedError.getOperation().getOperationCode(),
                     serializedUnsolicitedError);
         assertEquals(unsolicitedError.toString(), testedUnsolicitedError.toString());
      }
      catch (Exception e)
      {
         System.out.println(e.getMessage());
         fail(e.getMessage());
      }

      // // No exception
      ErrorUnsolicited unsolicitedError2 = new ErrorUnsolicited();
      assertEquals(unsolicitedError2.getException(), null);
      genericProtocolMessageTest(unsolicitedError2);
      try
      {
         byte[] serializedUnsolicitedError = serialize(unsolicitedError2);
         ErrorUnsolicited testedUnsolicitedError =
               (ErrorUnsolicited) deserialize(unsolicitedError2.getOperation().getOperationCode(),
                     serializedUnsolicitedError);
         assertEquals(unsolicitedError2.toString(), testedUnsolicitedError.toString());
      }
      catch (Exception e)
      {
         System.out.println(e.getMessage());
         fail(e.getMessage());
      }

      // Exception in constructor
      Exception exception2 = new Exception("My third new unsolicited error!");
      ErrorUnsolicited unsolicitedError3 = new ErrorUnsolicited(exception2);
      assertEquals(unsolicitedError3.getException(), exception2);
      unsolicitedError3.setException(new Exception("My new exception for my old error!"));
      genericProtocolMessageTest(unsolicitedError3);

      try
      {
         byte[] serializedUnsolicitedError = serialize(unsolicitedError3);
         ErrorUnsolicited testedUnsolicitedError =
               (ErrorUnsolicited) deserialize(unsolicitedError3.getOperation().getOperationCode(),
                     serializedUnsolicitedError);
         assertEquals(unsolicitedError3.toString(), testedUnsolicitedError.toString());
      }
      catch (Exception e)
      {
         System.out.println(e.getMessage());
         fail(e.getMessage());
      }
   }

   /**
    * This method tests the serialize() and deserialize() methods using a WriteNotify operation
    * type.
    * 
    */
   @Test
   public void testWriteNotifySerializeDeserialize()
   {
      SliceName[] sliceNames = getRandomSliceNames(10);

      // Write notify msg with slice names
      WriteNotifyUnsolicited writeNotify = new WriteNotifyUnsolicited(sliceNames);
      genericProtocolMessageTest(writeNotify);

      try
      {
         byte[] serializedWriteNotify = serialize(writeNotify);
         WriteNotifyUnsolicited testedWriteNotify =
               (WriteNotifyUnsolicited) deserialize(writeNotify.getOperation().getOperationCode(),
                     serializedWriteNotify);
         assertEquals(writeNotify.toString(), testedWriteNotify.toString());
      }
      catch (Exception e)
      {
         e.printStackTrace();
         fail(e.getMessage());
      }

      // // Empty write notify msg
      // WriteNotifyUnsolicited writeNotify2 = new WriteNotifyUnsolicited();
      // genericProtocolMessageTest(writeNotify2);
      //
      // try
      // {
      // byte[] serializedWriteNotify = serialize(writeNotify2);
      // WriteNotifyUnsolicited testedWriteNotify =
      // (WriteNotifyUnsolicited) messageFactory.deserialize(
      // writeNotify2.getOperation().getOperationCode(), serializedWriteNotify);
      // assertEquals(writeNotify2.toString(), testedWriteNotify.toString());
      // }
      // catch (Exception e)
      // {
      // System.out.println(e.getMessage());
      // fail(e.getMessage());
      // }
   }

   // /**
   // * This method tests the serialize() and deserialize() methods using a WriteRequest operation
   // * type.
   // *
   // */
   // @Test
   // public void testWriteRequestSerializeDeserialize()
   // {
   // int txid = 13903;
   // String sliceID = "1040A-21";
   // // 2 MB byte array
   // byte[] buf = new byte[2097152];
   // randomizeBuffer(buf);
   // WriteRequest writeRequest =
   // new WriteRequest(new DataSlice(new SliceName(sliceID, 6), txid, buf));
   // genericProtocolMessageTest(writeRequest);
   //
   // try
   // {
   // byte[] serializedWriteRequest = serialize(writeRequest);
   // WriteRequest testedWriteRequest =
   // (WriteRequest) messageFactory.deserialize(
   // writeRequest.getOperation().getOperationCode(), serializedWriteRequest);
   // assertEquals(writeRequest.toString(), testedWriteRequest.toString());
   // assertEquals(writeRequest.getDataSlice(), testedWriteRequest.getDataSlice());
   // }
   // catch (Exception e)
   // {
   // System.out.println(e.getMessage());
   // fail(e.getMessage());
   // }
   //
   // // Default constructor test
   // WriteRequest writeRequest2 = new WriteRequest();
   // genericProtocolMessageTest(writeRequest2);
   //
   // try
   // {
   // byte[] serializedWriteRequest = serialize(writeRequest2);
   // WriteRequest testedWriteRequest =
   // (WriteRequest) messageFactory.deserialize(
   // writeRequest2.getOperation().getOperationCode(), serializedWriteRequest);
   // assertEquals(writeRequest2.toString(), testedWriteRequest.toString());
   // }
   // catch (Exception e)
   // {
   // System.out.println(e.getMessage());
   // fail(e.getMessage());
   // }
   // }

   // /**
   // * This method tests the serialize() and deserialize() methods using a WriteResponse operation
   // * type.
   // *
   // */
   // @Test
   // public void testWriteResponseSerializeDeserialize()
   // {
   // WriteResponse writeResponse = new WriteResponse();
   // genericProtocolMessageTest(writeResponse);
   //
   // try
   // {
   // byte[] serializedWriteResponse = serialize(writeResponse);
   // WriteResponse testedWriteResponse =
   // (WriteResponse) messageFactory.deserialize(
   // writeResponse.getOperation().getOperationCode(), serializedWriteResponse);
   // assertEquals(writeResponse.toString(), testedWriteResponse.toString());
   // }
   // catch (Exception e)
   // {
   // System.out.println(e.getMessage());
   // fail(e.getMessage());
   // }
   // }

   @Test
   public void testMultipleReadRequestSerializeDeserialize() throws Exception
   {
      // First test with source names
      SliceName[] sliceNames = getRandomSliceNames(10);
      MultipleReadRequest mrr = new MultipleReadRequest(sliceNames);
      genericProtocolMessageTest(mrr);

      try
      {
         byte[] serializedMRR = serialize(mrr);

         MultipleReadRequest testedMRR =
               (MultipleReadRequest) deserialize(mrr.getOperation().getOperationCode(),
                     serializedMRR);
         assertEquals(mrr.toString(), testedMRR.toString());
         assertTrue(Arrays.equals(mrr.getSliceNames(), testedMRR.getSliceNames()));
      }
      catch (Exception e)
      {
         System.out.println(e.getMessage());
         fail(e.getMessage());
      }

      // Second test without source names
      MultipleReadRequest mrr2 = new MultipleReadRequest();
      mrr2.setSliceNames(new ArrayList<SliceName>());
      genericProtocolMessageTest(mrr2);

      byte[] serializedMRR = serialize(mrr2);

      MultipleReadRequest testedMRR =
            (MultipleReadRequest) deserialize(mrr2.getOperation().getOperationCode(), serializedMRR);
      assertEquals(mrr2.toString(), testedMRR.toString());
      assertTrue(Arrays.equals(mrr2.getSliceNames(), testedMRR.getSliceNames()));

   }

   @Test
   public void testMultipleReadResponseSerializeDeserialize()
   {
      // First test with slice buffers and exception
      DataSlice[] dataSlices = getRandomDataSliceArray(10, SLICE_BLOCK_SIZE);
      MultipleReadResponse mrr = new MultipleReadResponse(dataSlices);
      genericProtocolMessageTest(mrr);

      try
      {
         byte[] serializedMRR = serialize(mrr);
         MultipleReadResponse testedMRR =
               (MultipleReadResponse) deserialize(mrr.getOperation().getOperationCode(),
                     serializedMRR);
         assertEquals(mrr.toString(), testedMRR.toString());
         assertTrue(Arrays.equals(mrr.getDataSlices(), testedMRR.getDataSlices()));
      }
      catch (Exception e)
      {
         System.out.println(e.getMessage());
         fail(e.getMessage());
      }

      // Second test without slice buffers
      MultipleReadResponse mrr2 = new MultipleReadResponse();
      mrr2.setDataSlices(new ArrayList<DataSlice>());
      try
      {
         byte[] serializedMRR = serialize(mrr2);
         MultipleReadResponse testedMRR =
               (MultipleReadResponse) deserialize(mrr2.getOperation().getOperationCode(),
                     serializedMRR);
         assertEquals(mrr2.toString(), testedMRR.toString());

         // should be null
         assertTrue(Arrays.equals(mrr2.getDataSlices(), testedMRR.getDataSlices()));
      }
      catch (Exception e)
      {
         System.out.println(e.getMessage());
         fail(e.getMessage());
      }
   }

   @Test
   public void testMultipleWriteRequestSerializeDeserialize()
   {
      // First test with source names and slice buffers
      DataSlice[] dataSlices = getRandomDataSliceArray(10, SLICE_BLOCK_SIZE);
      MultipleWriteRequest mwr = new MultipleWriteRequest(dataSlices);
      genericProtocolMessageTest(mwr);

      try
      {
         byte[] serializedMWR = serialize(mwr);
         MultipleWriteRequest testedMWR =
               (MultipleWriteRequest) deserialize(mwr.getOperation().getOperationCode(),
                     serializedMWR);
         assertEquals(mwr.toString(), testedMWR.toString());
         assertTrue(Arrays.equals(mwr.getDataSlices(), testedMWR.getDataSlices()));
      }
      catch (Exception e)
      {
         System.out.println(e.getMessage());
         fail(e.getMessage());
      }

      // Null fields won't work with raw serialization
      // // Second test without source names or slice buffers
      // MultipleWriteRequest mwr2 = new MultipleWriteRequest();
      //
      // try
      // {
      // byte[] serializedMWR = serialize(mwr2);
      // MultipleWriteRequest testedMWR =
      // (MultipleWriteRequest) messageFactory.deserialize(
      // mwr2.getOperation().getOperationCode(), serializedMWR);
      // assertEquals(mwr2.toString(), testedMWR.toString());
      //
      // // should be null
      // assertTrue(mwr2.getDataSlices() == testedMWR.getDataSlices());
      // }
      // catch (Exception e)
      // {
      // System.out.println(e.getMessage());
      // fail(e.getMessage());
      // }
   }

   @Test
   public void testMultipleWriteResponseSerializeDeserialize()
   {
      // First test with exception (it's set in the genericProtocolMessageTest method
      MultipleWriteResponse mwr = new MultipleWriteResponse();
      genericProtocolMessageTest(mwr);

      try
      {
         byte[] serializedMWR = serialize(mwr);
         MultipleWriteResponse testedMWR =
               (MultipleWriteResponse) deserialize(mwr.getOperation().getOperationCode(),
                     serializedMWR);
         assertEquals(mwr.toString(), testedMWR.toString());
      }
      catch (Exception e)
      {
         System.out.println(e.getMessage());
         fail(e.getMessage());
      }

      // 2nd test without exception
      MultipleWriteResponse mwr2 = new MultipleWriteResponse();
      genericProtocolMessageTest(mwr2);

      try
      {
         byte[] serializedMWR = serialize(mwr2);
         MultipleWriteResponse testedMWR =
               (MultipleWriteResponse) deserialize(mwr2.getOperation().getOperationCode(),
                     serializedMWR);
         assertEquals(mwr2.toString(), testedMWR.toString());
      }
      catch (Exception e)
      {
         System.out.println(e.getMessage());
         fail(e.getMessage());
      }
   }

   @Test
   public void testMultipleRemoveRequestSerializeDeserialize()
   {
      SliceName[] sliceNames = getRandomSliceNames(10);
      MultipleRemoveRequest mrr = new MultipleRemoveRequest(sliceNames);
      genericProtocolMessageTest(mrr);

      try
      {
         byte[] serializedMRR = serialize(mrr);
         MultipleRemoveRequest testedMRR =
               (MultipleRemoveRequest) deserialize(mrr.getOperation().getOperationCode(),
                     serializedMRR);
         assertEquals(mrr.toString(), testedMRR.toString());
         assertTrue(Arrays.equals(mrr.getSliceNames(), testedMRR.getSliceNames()));
      }
      catch (Exception e)
      {
         System.out.println(e.getMessage());
         fail(e.getMessage());
      }
   }

   @Test
   public void testMultipleRemoveResponseSerializeDeserialize()
   {
      boolean removed[] = new boolean[32];
      for (int i = 0; i < removed.length; ++i)
      {
         // alternating values
         removed[i] = (i % 2 == 0) ? true : false;
      }

      MultipleRemoveResponse mrr = new MultipleRemoveResponse(removed);
      genericProtocolMessageTest(mrr);

      try
      {
         byte[] serializedMRR = serialize(mrr);
         MultipleRemoveResponse testedMRR =
               (MultipleRemoveResponse) deserialize(mrr.getOperation().getOperationCode(),
                     serializedMRR);
         assertEquals(mrr.toString(), testedMRR.toString());
      }
      catch (Exception e)
      {
         System.out.println(e.getMessage());
         fail(e.getMessage());
      }
   }

   public static SliceName[] getRandomSliceNames(int arraySize)
   {
      SliceName[] sliceNames = new SliceName[arraySize];

      for (int i = 0; i < arraySize; i++)
      {
         sliceNames[i] = getRandomSliceName();
      }

      return sliceNames;
   }

   public static SliceName getRandomSliceName()
   {
      int sourceName = (int) (Math.random() * 100000);
      return new SliceName(Integer.toString(sourceName), (int) (Math.random() * 100));
   }

   public static DataSlice[] getRandomDataSliceArray(int arraySize, int dataSize)
   {
      DataSlice[] dataSlices = new DataSlice[arraySize];

      for (int i = 0; i < arraySize; i++)
      {
         int txid = (int) (Math.random() * 1000);
         dataSlices[i] = getRandomDataSlice(txid, dataSize);
      }

      return dataSlices;
   }

   public static DataSlice getRandomDataSlice(int txid, int dataSize)
   {
      byte[] buf = new byte[dataSize];
      randomizeBuffer(buf);
      int sourceName = (int) (Math.random() * 100000);

      return new DataSlice(
            new SliceName(Integer.toString(sourceName), (int) (Math.random() * 100)), txid, buf);
   }

   public static void randomizeBuffer(byte buffer[])
   {
      for (int idx = 0; idx < buffer.length; idx++)
      {
         byte value = (byte) (Math.random() * 256);
         buffer[idx] = value;
      }
   }

   public byte[] serialize(ProtocolMessage message) throws ProtocolSerializationException
   {
      ByteArrayOutputStream bytes = new ByteArrayOutputStream();
      DataOutputStream out = new DataOutputStream(bytes);
      messageFactory.serialize(out, message);
      return bytes.toByteArray();
   }

   public ProtocolMessage deserialize(int operationCode, byte[] payload)
         throws OperationNotRegisteredException, ProtocolDeserializationException
   {
      return messageFactory.deserialize(operationCode, new DataInputStream(
            new ByteArrayInputStream(payload)));
   }

   /**
    * Main function that serializes a protocol message and saves it to a file.
    * 
    * @param args
    * @throws Exception
    */
   public static void main(String[] args) throws Exception
   {
      ProtocolMessageFactory messageFactory = new ASN1GridProtocolMessageFactory();

      try
      {
         GridProtocolMessageFactory.register(GridProtocolOperation.NOOP_RESPONSE,
               NoopResponse.class);
      }
      catch (OperationAlreadyRegisteredException oare)
      {

      }

      byte[] serializedNoopResponse = null;
      NoopResponse noopResponse = new NoopResponse();

      try
      {
         ByteArrayOutputStream bytes = new ByteArrayOutputStream();
         DataOutputStream out = new DataOutputStream(bytes);
         messageFactory.serialize(out, noopResponse);
         serializedNoopResponse = bytes.toByteArray();

      }
      catch (Exception e)
      {
         fail(e.getMessage());
      }

      if (serializedNoopResponse == null)
      {
         System.out.println("Serialization failed!");
         System.exit(1);
      }

      if (args.length != 1)
      {
         System.out.println("Invalid number of arguments!");
         System.exit(1);
      }
      File file = null;
      file = new File(args[0]);

      // Now write the data array to the file.
      try
      {
         FileOutputStream file_output = new FileOutputStream(file);
         DataOutputStream data_out = new DataOutputStream(file_output);

         // Get rid of header
         byte[] payload = new byte[serializedNoopResponse.length - 16];
         System.arraycopy(serializedNoopResponse, 16, payload, 0, payload.length);

         data_out.write(payload);
         file_output.close();
      }
      catch (IOException e)
      {
         System.out.println("IO exception = " + e);
      }

      System.out.println("Completed!");
   }
}
