package org.cleversafe.layer.slicestore.remote;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;

import java.io.InputStream;
import java.io.OutputStream;
import java.net.URI;
import java.security.Key;
import java.security.PrivateKey;
import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Queue;
import java.util.UUID;

import org.cleversafe.authentication.Credentials;
import org.cleversafe.authentication.credentials.PasswordCredentials;
import org.cleversafe.layer.communication.Connector;
import org.cleversafe.layer.communication.exceptions.CommunicationConnectionException;
import org.cleversafe.layer.communication.exceptions.CommunicationIOException;
import org.cleversafe.layer.communication.exceptions.CommunicationInterruptedException;
import org.cleversafe.layer.communication.exceptions.CommunicationResponseException;
import org.cleversafe.layer.communication.exceptions.CommunicationTransmissionException;
import org.cleversafe.layer.communication.exceptions.NotConnectedException;
import org.cleversafe.layer.grid.DataSlice;
import org.cleversafe.layer.grid.SliceName;
import org.cleversafe.layer.grid.SourceName;
import org.cleversafe.layer.protocol.BeginSessionRequest;
import org.cleversafe.layer.protocol.BeginSessionResponse;
import org.cleversafe.layer.protocol.BeginTransactionRequest;
import org.cleversafe.layer.protocol.BeginTransactionResponse;
import org.cleversafe.layer.protocol.CommitTransactionRequest;
import org.cleversafe.layer.protocol.CommitTransactionResponse;
import org.cleversafe.layer.protocol.CreateStoreRequest;
import org.cleversafe.layer.protocol.CreateStoreResponse;
import org.cleversafe.layer.protocol.EndSessionRequest;
import org.cleversafe.layer.protocol.EndSessionResponse;
import org.cleversafe.layer.protocol.ExistsRequest;
import org.cleversafe.layer.protocol.ExistsResponse;
import org.cleversafe.layer.protocol.IntegrityVerificationRequest;
import org.cleversafe.layer.protocol.IntegrityVerificationResponse;
import org.cleversafe.layer.protocol.ListBeginRequest;
import org.cleversafe.layer.protocol.ListBeginResponse;
import org.cleversafe.layer.protocol.ListContinueRequest;
import org.cleversafe.layer.protocol.ListContinueResponse;
import org.cleversafe.layer.protocol.ListInProgressResponse;
import org.cleversafe.layer.protocol.ListStopRequest;
import org.cleversafe.layer.protocol.ListStopResponse;
import org.cleversafe.layer.protocol.MultipleReadResponse;
import org.cleversafe.layer.protocol.MultipleRemoveRequest;
import org.cleversafe.layer.protocol.MultipleRemoveResponse;
import org.cleversafe.layer.protocol.MultipleWriteRequest;
import org.cleversafe.layer.protocol.MultipleWriteResponse;
import org.cleversafe.layer.protocol.PasswordAuthenticationRequest;
import org.cleversafe.layer.protocol.PasswordAuthenticationResponse;
import org.cleversafe.layer.protocol.RemoveStoreRequest;
import org.cleversafe.layer.protocol.RemoveStoreResponse;
import org.cleversafe.layer.protocol.Request;
import org.cleversafe.layer.protocol.Response;
import org.cleversafe.layer.protocol.RollbackTransactionRequest;
import org.cleversafe.layer.protocol.RollbackTransactionResponse;
import org.cleversafe.layer.protocol.VaultBindRequest;
import org.cleversafe.layer.protocol.VaultBindResponse;
import org.cleversafe.layer.slicestore.SliceInfo;
import org.cleversafe.layer.slicestore.SliceStoreTransaction;
import org.cleversafe.layer.slicestore.block.BlockFileSliceStoreInfo;
import org.cleversafe.layer.slicestore.exceptions.IllegalSourceNameException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreExistsException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreIOException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreNotFoundException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreQuotaException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreTransactionException;
import org.cleversafe.serialization.ProtocolMessageFactory;
import org.cleversafe.serialization.raw.RawGridProtocolMessageFactory;
import org.cleversafe.vault.VaultACL;
import org.cleversafe.vault.exceptions.VaultACLException;
import org.cleversafe.vault.exceptions.VaultDescriptorException;
import org.cleversafe.vault.exceptions.VaultIOException;
import org.cleversafe.vault.exceptions.VaultKeyException;
import org.cleversafe.vault.exceptions.VaultKeyLookupException;
import org.cleversafe.vault.exceptions.VaultSecurityException;
import org.cleversafe.vault.storage.asn1.EncryptedKeyInfo;
import org.cleversafe.vault.storage.asn1.PlainKeyInfo;
import org.cleversafe.vault.storage.asn1.VaultACLEntry;
import org.cleversafe.vault.storage.asn1.VaultPermissionEntry;
import org.junit.Before;
import org.junit.Test;


public class RemoteSliceStoreTest
{
   private static final Exception NO_EXCEPTION = null;

   private static interface RemoteSliceStoreTestMethod
   {
      public void execute(Exception protocolException, Exception communicationException) throws Exception;      
   }
   
   private static class MockVaultACL extends VaultACL
   {
      public void flush() throws VaultIOException
      {
      }

      @Override
      public void flush(OutputStream out) throws VaultIOException
      {
      }

      @Override
      public void flush(URI vaultLocation) throws VaultIOException
      {
      }

      @Override
      protected VaultACLEntry getACLEntry(UUID account)
      {
         return null;
      }

      @Override
      public byte[] getEncoded()
      {
         return new byte[0];
      }

      @Override
      protected Key getKey(UUID account, int index, PrivateKey privateKey)
            throws VaultACLException, VaultKeyException, VaultKeyLookupException,
            VaultSecurityException
      {
         return null;
      }

      public UUID getOwner()
      {
         return null;
      }

      @Override
      protected Key getPublicKey(UUID account, int index, PrivateKey privateKey)
            throws VaultACLException, VaultKeyException, VaultKeyLookupException,
            VaultSecurityException
      {
         return null;
      }

      @Override
      public UUID getVaultIdentifier()
      {
         return null;
      }

      @Override
      protected void loadFromStream(InputStream in) throws VaultIOException
      {
      }

      @Override
      protected void loadFromURI(URI vaultLocation, UUID vaultIdentifier) throws VaultIOException
      {
      }

      @Override
      protected void sign(UUID adminAccount, PrivateKey adminPrivateEncryptionKey)
            throws VaultSecurityException
      {
      }

      @Override
      protected void storeACLEntry(
            UUID account,
            VaultPermissionEntry permission,
            Map<Integer, EncryptedKeyInfo> vaultKeys)
      {
      }

      @Override
      protected void storeOwner(UUID account)
      {
      }

      @Override
      protected void storePublicKeys(Map<Integer, PlainKeyInfo> publicKeys)
      {
      }

      @Override
      protected void storeVaultIdentifier(UUID vault)
      {
      }

      @Override
      public boolean verify()
      {
         return false;
      }      
   }
   
   private static class MockConnector implements Connector
   {
      private boolean connected = false;
      
      private Queue<Request> requestQueue = new LinkedList<Request>();
      private Queue<Response> responseQueue = new LinkedList<Response>();
      private Queue<Exception> protocolExceptionQueue = new LinkedList<Exception>();
      private Queue<Exception> communicationExceptionQueue = new LinkedList<Exception>();
      
      
      public void connect() throws CommunicationIOException, CommunicationConnectionException
      {
         connected = true;
      }

      public void disconnect() throws CommunicationIOException, CommunicationInterruptedException
      {
         connected = false;
      }

      public boolean ensureConnected() throws CommunicationIOException,
            CommunicationConnectionException
      {
         if (isConnected())
         {
            return false;
         }
         else
         {
            connect();
            return true;
         }
      }

      public Response exchange(Request request) throws CommunicationIOException,
            NotConnectedException, CommunicationInterruptedException,
            CommunicationResponseException, CommunicationTransmissionException
      {
         return exchange(request, 60);
      }

      public Response exchange(Request request, int timeout) throws CommunicationIOException,
            NotConnectedException, CommunicationInterruptedException,
            CommunicationResponseException, CommunicationTransmissionException
      {
         Exception ex = communicationExceptionQueue.poll();
         
         if(ex != NO_EXCEPTION)
         {
            try
            {
               throw ex;
            } catch(CommunicationIOException e)
            {
               throw e;
            } catch(NotConnectedException e)
            {
               throw e;
            } catch(CommunicationInterruptedException e)
            {
               throw e;
            } catch(CommunicationResponseException e)
            {
               throw e;
            } catch(CommunicationTransmissionException e)
            {
               throw e;
            } 
            catch (Exception e)
            {
               e.printStackTrace();
            }
         }
         
         requestQueue.add(request);
         Response response = responseQueue.poll();;
         
         ex = protocolExceptionQueue.poll();
         
         if(ex != NO_EXCEPTION)
         {
            response.setException(ex);
         }
         
         return response;
      }

      public String getIdentification()
      {
         return "mock";
      }

      public String getLocalAddress()
      {
         return "mock";
      }

      public int getNumOutstandingExchanges()
      {
         return 0;
      }

      public ProtocolMessageFactory getProtocolMessageFactory()
      {
         return new RawGridProtocolMessageFactory();
      }

      public String getRemoteAddress()
      {
         return "mock";
      }

      public void initialize()
      {
      }

      public boolean isConnected()
      {
         return connected;
      }

      public void sendPing() throws CommunicationInterruptedException,
            CommunicationResponseException, CommunicationTransmissionException,
            CommunicationIOException, NotConnectedException
      {
      }     
      
      public void enqueueResponse(Response response)
      {
         responseQueue.add(response);
      }
      
      public Request dequeueRequest()
      {
         return requestQueue.poll();
      }
      
      public void enqueueProtocolException(Exception ex)
      {
         protocolExceptionQueue.add(ex);
      }
      
      public void enqueueCommunicationException(Exception ex)
      {
         communicationExceptionQueue.add(ex);
      }

      public void reset()
      {
         requestQueue.clear();
         responseQueue.clear();
         protocolExceptionQueue.clear();
         communicationExceptionQueue.clear();
      }
   }

   private UUID vaultId;
   private RemoteSliceStore sliceStore;
   private MockConnector connector;

   private List<Exception> protocolExceptions = new ArrayList<Exception>();
   private List<Exception> communicationExceptions = new ArrayList<Exception>();
   
   public RemoteSliceStoreTest()
   {
      protocolExceptions.add(new SliceStoreTransactionException());      
      protocolExceptions.add(new IllegalSourceNameException());
      protocolExceptions.add(new SliceStoreExistsException());
      protocolExceptions.add(new SliceStoreQuotaException());
      protocolExceptions.add(new SliceStoreNotFoundException());
      protocolExceptions.add(new SliceStoreIOException());
      protocolExceptions.add(new IllegalSourceNameException());
      
      communicationExceptions.add(new NotConnectedException()); 
      communicationExceptions.add(new CommunicationIOException());
      communicationExceptions.add(new CommunicationInterruptedException()); 
      communicationExceptions.add(new CommunicationResponseException());
      communicationExceptions.add(new CommunicationTransmissionException());       
   }
   
   @Before
   public void setup()
   {
      vaultId = UUID.randomUUID();
      connector = new MockConnector();
      Credentials credentials = new PasswordCredentials("test-user", "test-pass");
      sliceStore = new RemoteSliceStore(vaultId, credentials, connector);
      sliceStore.setRemoteSliceStoreInfo(new BlockFileSliceStoreInfo());
      
      assertEquals(connector, sliceStore.getConnection());
      sliceStore.setConnection(connector);
      assertEquals(connector, sliceStore.getConnection());

      assertEquals(credentials, sliceStore.getCredentials());
      sliceStore.setCredentials(credentials);
      assertEquals(credentials, sliceStore.getCredentials());
     
      assertEquals(vaultId, sliceStore.getVaultIdentifier());
      sliceStore.setVaultIdentifier(vaultId);
      assertEquals(vaultId, sliceStore.getVaultIdentifier());
      
      sliceStore.initialize();
      
      assertTrue(sliceStore.isOperational());
   }

   @Test
   public void testCreateStore()
   {
      testMethod(new RemoteSliceStoreTestMethod() 
      {
         public void execute(Exception protocolException, Exception communicationException) throws Exception
         {
            testCreateStore(protocolException, communicationException);
         }         
      });
   }

   private void testCreateStore(Exception protocolException, Exception communicationException) 
      throws SliceStoreExistsException, SliceStoreIOException
   {      
      // Slice store will authenticate
      PasswordAuthenticationResponse authResponse = new PasswordAuthenticationResponse();
      authResponse.setAuthenticated(true);
      connector.enqueueResponse(authResponse);      
      connector.enqueueProtocolException(NO_EXCEPTION);
      connector.enqueueCommunicationException(NO_EXCEPTION);
      
      // Slice store will send a create store request
      connector.enqueueResponse(new CreateStoreResponse());
      connector.enqueueProtocolException(protocolException);
      connector.enqueueCommunicationException(communicationException);
      
      sliceStore.createStore(BlockFileSliceStoreInfo.TYPE, 500, 1000, new MockVaultACL(), "<vault/>".getBytes());
      
      PasswordAuthenticationRequest authRequest = (PasswordAuthenticationRequest) connector.dequeueRequest();
      assertEquals(authRequest.getUsername(), "test-user");
      assertEquals(authRequest.getPassword(), "test-pass");

      CreateStoreRequest createStoreRequest = (CreateStoreRequest) connector.dequeueRequest();
      assertEquals(BlockFileSliceStoreInfo.TYPE, createStoreRequest.getSliceStoreType());
      assertEquals(500L, createStoreRequest.getMaxSliceSize());
      assertEquals(1000L, createStoreRequest.getSliceStoreSize());
      assertEquals(vaultId, createStoreRequest.getVaultID());            
   }
   
   @Test 
   public void testDeleteStore() throws SliceStoreIOException, SliceStoreNotFoundException
   {
      testMethod(new RemoteSliceStoreTestMethod() 
      {
         public void execute(Exception protocolException, Exception communicationException) throws Exception
         {
            testDeleteStore(protocolException, communicationException, false);
         }         
      });
      
      testMethod(new RemoteSliceStoreTestMethod() 
      {
         public void execute(Exception protocolException, Exception communicationException) throws Exception
         {
            testDeleteStore(protocolException, communicationException, true);
         }         
      });

   }
   
   public void testDeleteStore(Exception protocolException, Exception communicationException, boolean connectException) throws SliceStoreIOException, SliceStoreNotFoundException, CommunicationIOException, CommunicationConnectionException, CommunicationInterruptedException
   {
      // Slice store will send remove store request

      if(connectException)
      {
         connector.reset();
         connector.disconnect();
         
         PasswordAuthenticationResponse authResponse = new PasswordAuthenticationResponse();
         authResponse.setAuthenticated(true);
         
         connector.enqueueResponse(authResponse);      
         connector.enqueueProtocolException(NO_EXCEPTION);
         connector.enqueueCommunicationException(NO_EXCEPTION);

         connector.enqueueResponse(new VaultBindResponse());
         connector.enqueueProtocolException(NO_EXCEPTION);
         connector.enqueueCommunicationException(NO_EXCEPTION);

         connector.enqueueResponse(new BeginSessionResponse());
         connector.enqueueProtocolException(protocolException);
         connector.enqueueCommunicationException(communicationException);
         connector.enqueueResponse(new EndSessionResponse());
         connector.enqueueResponse(new RemoveStoreResponse());
   
         sliceStore.deleteStore();
      } else
      {
         connector.ensureConnected();
         
         connector.enqueueResponse(new RemoveStoreResponse());
         connector.enqueueProtocolException(protocolException);
         connector.enqueueCommunicationException(communicationException);
         
         sliceStore.deleteStore();
               
         RemoveStoreRequest removeStoreRequest = (RemoveStoreRequest) connector.dequeueRequest();
         assertEquals(vaultId, removeStoreRequest.getVaultIdentifier());
      }               
   }
   
   @Test
   public void testStartSession()
   {
      testMethod(new RemoteSliceStoreTestMethod() 
      {
         public void execute(Exception protocolException, Exception communicationException) throws Exception
         {
            testStartSession(protocolException, communicationException, true, false, false);
         }         
      });

      testMethod(new RemoteSliceStoreTestMethod() 
      {
         public void execute(Exception protocolException, Exception communicationException) throws Exception
         {
            testStartSession(protocolException, communicationException, false, true, false);
         }         
      });

      testMethod(new RemoteSliceStoreTestMethod() 
      {
         public void execute(Exception protocolException, Exception communicationException) throws Exception
         {
            testStartSession(protocolException, communicationException, false, false, true);
         }         
      });      
   }
   
   private void testStartSession(Exception protocolException, Exception communicationException, 
         boolean authException, boolean bindException, boolean startSessionException) throws SliceStoreIOException, SliceStoreNotFoundException
   {      
      // Slice store will authenticate      
      // Slice store will then bind to the vault
      // Slice store will then begin the session

      PasswordAuthenticationResponse authResponse = new PasswordAuthenticationResponse();
      authResponse.setAuthenticated(true);
      connector.enqueueResponse(authResponse);      

      if(authException)
      {
         connector.enqueueProtocolException(protocolException);
         connector.enqueueCommunicationException(communicationException);
      } else
      {
         connector.enqueueProtocolException(NO_EXCEPTION);
         connector.enqueueCommunicationException(NO_EXCEPTION);                  
      }
      
      connector.enqueueResponse(new VaultBindResponse());            

      if(bindException)
      {
         connector.enqueueProtocolException(protocolException);
         connector.enqueueCommunicationException(communicationException);
      } else
      {
         connector.enqueueProtocolException(NO_EXCEPTION);
         connector.enqueueCommunicationException(NO_EXCEPTION);                  
      }
      
      connector.enqueueResponse(new BeginSessionResponse(0));

      if(startSessionException)
      {
         connector.enqueueProtocolException(protocolException);
         connector.enqueueCommunicationException(communicationException);         
      } else
      {
         connector.enqueueProtocolException(NO_EXCEPTION);
         connector.enqueueCommunicationException(NO_EXCEPTION);                  
      }
      
      sliceStore.startSession();

      PasswordAuthenticationRequest authRequest = (PasswordAuthenticationRequest) connector.dequeueRequest();
      assertEquals(authRequest.getUsername(), "test-user");
      assertEquals(authRequest.getPassword(), "test-pass");
      
      VaultBindRequest bindRequest = (VaultBindRequest) connector.dequeueRequest(); 
      assertEquals(vaultId, bindRequest.getVaultIdentifier());
      
      assertTrue(connector.dequeueRequest() instanceof BeginSessionRequest);
      
      assertEquals(connector.dequeueRequest(), null);
   }   
   
   private void startSession() throws SliceStoreIOException, SliceStoreNotFoundException
   {
      testStartSession(NO_EXCEPTION, NO_EXCEPTION, false, false, false);
   }
   
   @Test
   public void testEndSession() throws SliceStoreIOException, SliceStoreNotFoundException
   {
      testMethod(new RemoteSliceStoreTestMethod() 
      {
         public void execute(Exception protocolException, Exception communicationException) throws Exception
         {
            testEndSession(protocolException, communicationException);
         }         
      });
   }
   
   private void testEndSession(Exception protocolException, Exception communicationException) throws SliceStoreIOException, SliceStoreNotFoundException
   {
      testStartSession(NO_EXCEPTION, NO_EXCEPTION, false, false, false);
            
      connector.enqueueResponse(new EndSessionResponse());
      connector.enqueueProtocolException(protocolException);
      connector.enqueueCommunicationException(communicationException);
      
      sliceStore.endSession();
      
      assertTrue(connector.dequeueRequest() instanceof EndSessionRequest);      
   }
   
   @Test
   public void testBeginTransaction() throws SliceStoreTransactionException, SliceStoreIOException, SliceStoreNotFoundException
   {
      testMethod(new RemoteSliceStoreTestMethod() 
      {
         public void execute(Exception protocolException, Exception communicationException) throws Exception
         {
            startSession();
            testBeginTransaction(protocolException, communicationException);
         }         
      });
   }
   
   private SliceStoreTransaction testBeginTransaction(Exception protocolException, Exception communicationException) throws SliceStoreTransactionException, SliceStoreIOException, SliceStoreNotFoundException
   {
      // Slice store will send begin transaction request      
      SliceStoreTransaction tx = sliceStore.createTransaction(123);
      
      connector.enqueueResponse(new BeginTransactionResponse());
      connector.enqueueProtocolException(protocolException);
      connector.enqueueCommunicationException(communicationException);
      
      sliceStore.beginTransaction(tx);
      
      BeginTransactionRequest beginTxRequest = (BeginTransactionRequest) connector.dequeueRequest();
      assertEquals(123L, beginTxRequest.getTransactionID());
      assertTrue(sliceStore.isActiveTransaction(tx));
      
      return tx;      
   }
   
   private SliceStoreTransaction beginTransaction() throws SliceStoreTransactionException, SliceStoreIOException, SliceStoreNotFoundException
   {
      return testBeginTransaction(NO_EXCEPTION, NO_EXCEPTION);      
   }
   
   @Test
   public void testGetTransaction() throws SliceStoreTransactionException, SliceStoreIOException
   {
      SliceStoreTransaction tx = sliceStore.createTransaction(123);
      assertEquals(123L, sliceStore.getTransaction(123).getID());
      assertFalse(sliceStore.isActiveTransaction(tx));
   }
   
   @Test
   public void testCommitTransaction() throws SliceStoreTransactionException, SliceStoreIOException, SliceStoreNotFoundException
   {
      testMethod(new RemoteSliceStoreTestMethod() 
      {
         public void execute(Exception protocolException, Exception communicationException) throws Exception
         {
            startSession();
            testCommitTransaction(protocolException, communicationException);
         }         
      });
   }
   
   public void testCommitTransaction(Exception protocolException, Exception communicationException) throws SliceStoreTransactionException, SliceStoreIOException, SliceStoreNotFoundException
   {
      SliceStoreTransaction tx = beginTransaction();
      
      connector.enqueueResponse(new CommitTransactionResponse());
      connector.enqueueProtocolException(protocolException);
      connector.enqueueCommunicationException(communicationException);
      
      sliceStore.commitTransaction(tx);
      
      CommitTransactionRequest commitRequest = (CommitTransactionRequest) connector.dequeueRequest();
      assertEquals(tx.getID(), commitRequest.getTransactionID());      
   }
   
   @Test 
   public void testRollbackTransaction() throws SliceStoreTransactionException, SliceStoreIOException, SliceStoreNotFoundException
   {
      testMethod(new RemoteSliceStoreTestMethod() 
      {
         public void execute(Exception protocolException, Exception communicationException) throws Exception
         {
            startSession();
            testRollbackTransaction(protocolException, communicationException);
         }         
      });
   }
   
   private void testRollbackTransaction(Exception protocolException, Exception communicationException) throws SliceStoreIOException, SliceStoreTransactionException, SliceStoreNotFoundException
   {
      SliceStoreTransaction tx = beginTransaction();
      
      connector.enqueueResponse(new RollbackTransactionResponse());
      connector.enqueueProtocolException(protocolException);
      connector.enqueueCommunicationException(communicationException);
      
      sliceStore.rollbackTransaction(tx);
      
      RollbackTransactionRequest rollbackRequest = (RollbackTransactionRequest) connector.dequeueRequest();
      
      assertEquals(tx.getID(), rollbackRequest.getTransactionID());      
   }

   @Test
   public void testRead()
   {
      testMethod(new RemoteSliceStoreTestMethod() 
      {
         public void execute(Exception protocolException, Exception communicationException) throws Exception
         {
            startSession();
            testRead(protocolException, communicationException);
         }         
      });      
   }
   
   private void testRead(Exception protocolException, Exception communicationException) throws SliceStoreIOException, IllegalSourceNameException, SliceStoreNotFoundException
   {
      List<SliceName> sliceNames = new ArrayList<SliceName>();
      sliceNames.add(new SliceName("1", 0));      
      DataSlice dataSlice = new DataSlice(new SliceName("1", 0));
      dataSlice.setData("test".getBytes());
      DataSlice[] originalSlices = {dataSlice};
      MultipleReadResponse response = new MultipleReadResponse();
      response.setDataSlices(originalSlices);
      connector.enqueueResponse(response);
      connector.enqueueProtocolException(protocolException);
      connector.enqueueCommunicationException(communicationException);
      
      List<DataSlice> slices = sliceStore.read(sliceNames);
      
      assertEquals(originalSlices.length, slices.size());
      
      //////////////////////////////////////////////////////////////////////////
      sliceNames = new ArrayList<SliceName>();
      sliceNames.add(new SliceName("1", 0));      
      dataSlice = new DataSlice(new SliceName("5", 0));
      dataSlice.setData("test".getBytes());
      response = new MultipleReadResponse();
      response.setDataSlices(originalSlices);
      connector.enqueueResponse(response);
      connector.enqueueProtocolException(protocolException);
      connector.enqueueCommunicationException(communicationException);
      
      DataSlice slice = sliceStore.read(new SliceName("5", 0));
            
   }
   
   @Test
   public void testWrite()
   {
      testMethod(new RemoteSliceStoreTestMethod() 
      {
         public void execute(Exception protocolException, Exception communicationException) throws Exception
         {
            startSession();
            testWrite(protocolException, communicationException, false);
         }         
      });            
      
      testMethod(new RemoteSliceStoreTestMethod() 
      {
         public void execute(Exception protocolException, Exception communicationException) throws Exception
         {
            testWrite(protocolException, communicationException, true);
         }         
      });            
   }
   
   private void testWrite(Exception protocolException, Exception communicationException, boolean allowOverwriteNewer) throws SliceStoreIOException, IllegalSourceNameException, SliceStoreTransactionException, SliceStoreQuotaException, SliceStoreNotFoundException
   {
      connector.enqueueResponse(new MultipleWriteResponse());
      connector.enqueueProtocolException(protocolException);
      connector.enqueueCommunicationException(communicationException);
      
      List<DataSlice> originalSlices = new ArrayList<DataSlice>();
      DataSlice originalSlice = new DataSlice(new SliceName("1",0));
      originalSlice.setData("test".getBytes());
      originalSlices.add(originalSlice);          
      
      sliceStore.write(originalSlices, allowOverwriteNewer);
      
      MultipleWriteRequest request = (MultipleWriteRequest) connector.dequeueRequest();
      DataSlice[] dataSlices = request.getDataSlices();
      assertEquals(originalSlices.size(), dataSlices.length);
      assertEquals("test", new String(dataSlices[0].getData()));
      assertEquals(allowOverwriteNewer, request.getAllowOverwriteNewer());

      //////////////////////////////////////////////////////////////////////////
      
      connector.enqueueResponse(new MultipleWriteResponse());
      connector.enqueueProtocolException(protocolException);
      connector.enqueueCommunicationException(communicationException);

      sliceStore.write(originalSlice, allowOverwriteNewer);

      request = (MultipleWriteRequest) connector.dequeueRequest();
      dataSlices = request.getDataSlices();
      assertEquals(originalSlices.size(), dataSlices.length);
      assertEquals("test", new String(dataSlices[0].getData()));
      assertEquals(allowOverwriteNewer, request.getAllowOverwriteNewer());      
   }
   
   @Test
   public void testExists()
   {
      testMethod(new RemoteSliceStoreTestMethod() 
      {
         public void execute(Exception protocolException, Exception communicationException) throws Exception
         {
            startSession();
            testExists(protocolException, communicationException, false);
         }         
      });            

      testMethod(new RemoteSliceStoreTestMethod() 
      {
         public void execute(Exception protocolException, Exception communicationException) throws Exception
         {
            startSession();
            testExists(protocolException, communicationException, false);
         }         
      });            
   }
   
   private void testExists(Exception protocolException, Exception communicationException, boolean exists) throws SliceStoreIOException, SliceStoreNotFoundException
   {
      connector.enqueueResponse(new ExistsResponse(exists));
      connector.enqueueProtocolException(protocolException);
      connector.enqueueCommunicationException(communicationException);
      
      assertEquals(exists, sliceStore.exists(new SliceName("1", 0)));
      
      assertTrue(connector.dequeueRequest() instanceof ExistsRequest);
   }
   
   @Test
   public void testRemove()
   {
      testMethod(new RemoteSliceStoreTestMethod() 
      {
         public void execute(Exception protocolException, Exception communicationException) throws Exception
         {
            startSession();
            testRemove(protocolException, communicationException);
         }         
      });                  
   }
   
   private void testRemove(Exception protocolException, Exception communicationException) throws SliceStoreIOException, IllegalSourceNameException, SliceStoreNotFoundException
   {
      connector.enqueueResponse(new MultipleRemoveResponse());
      connector.enqueueProtocolException(protocolException);
      connector.enqueueCommunicationException(communicationException);
      
      List<SliceName> originalSliceNames = new ArrayList<SliceName>();
      originalSliceNames.add(new SliceName("1", 0));
      
      sliceStore.remove(originalSliceNames);
      
      MultipleRemoveRequest request = (MultipleRemoveRequest) connector.dequeueRequest();
      SliceName[] sliceNames = request.getSliceNames();
      
      assertEquals(originalSliceNames.size(), sliceNames.length);
      assertEquals("1", sliceNames[0].getSourceName().getName());
   }
   
   @Test
   public void testListBegin()
   {
      testMethod(new RemoteSliceStoreTestMethod() 
      {
         public void execute(Exception protocolException, Exception communicationException) throws Exception
         {
            startSession();
            testListBegin(protocolException, communicationException);
         }         
      });      
   }
   
   private void testListBegin(Exception protocolException, Exception communicationException) throws SliceStoreIOException, SliceStoreNotFoundException
   {
      connector.responseQueue.add(new ListBeginResponse());
      connector.enqueueProtocolException(protocolException);
      connector.enqueueCommunicationException(communicationException);
      
      sliceStore.listBegin();
      
      assertTrue((ListBeginRequest) connector.dequeueRequest() instanceof ListBeginRequest);
   }
   
   @Test
   public void testListInProgress() throws SliceStoreIOException, SliceStoreNotFoundException
   {
      testMethod(new RemoteSliceStoreTestMethod() 
      {
         public void execute(Exception protocolException, Exception communicationException) throws Exception
         {
            startSession();
            testListInProgress(protocolException, communicationException, false);
         }         
      });      
      
      testMethod(new RemoteSliceStoreTestMethod() 
      {
         public void execute(Exception protocolException, Exception communicationException) throws Exception
         {
            startSession();
            testListInProgress(protocolException, communicationException, true);
         }         
      });      
   }
   
   private void testListInProgress(Exception protocolException, Exception communicationException, boolean inProgress) throws SliceStoreIOException, SliceStoreNotFoundException
   {
      connector.responseQueue.add(new ListInProgressResponse(inProgress));
      connector.enqueueProtocolException(protocolException);
      connector.enqueueCommunicationException(communicationException);
      
      assertEquals(inProgress, sliceStore.listInProgress());      
   }
   
   @Test
   public void testListContinue()
   {
      testMethod(new RemoteSliceStoreTestMethod() 
      {
         public void execute(Exception protocolException, Exception communicationException) throws Exception
         {
            startSession();
            testListContinue(protocolException, communicationException);
         }         
      });            
   }
   
   private void testListContinue(Exception protocolException, Exception communicationException) throws SliceStoreIOException, SliceStoreNotFoundException
   {  
      List<SliceInfo> originalSlices = new ArrayList<SliceInfo>();
      originalSlices.add(new SliceInfo(new SourceName("1"), 9));
      originalSlices.add(new SliceInfo(new SourceName("2"), 9));
      originalSlices.add(new SliceInfo(new SourceName("3"), 10));
      originalSlices.add(new SliceInfo(new SourceName("4"), 11));
      
      ListContinueResponse response = new ListContinueResponse();
      response.setSlices(originalSlices);
      connector.enqueueResponse(response);
      connector.enqueueProtocolException(protocolException);
      connector.enqueueCommunicationException(communicationException);
      
      List<SliceInfo> slices = sliceStore.listContinue();
      
      assertTrue(connector.dequeueRequest() instanceof ListContinueRequest);
      
      assertEquals(4, slices.size());
   }
   
   @Test
   public void testListStop()
   {
      testMethod(new RemoteSliceStoreTestMethod() 
      {
         public void execute(Exception protocolException, Exception communicationException) throws Exception
         {
            startSession();
            testListStop(protocolException, communicationException);
         }         
      });                  
   }
   
   private void testListStop(Exception protocolException, Exception communicationException) throws SliceStoreIOException, SliceStoreNotFoundException
   {
      connector.enqueueResponse(new ListStopResponse());
      connector.enqueueProtocolException(protocolException);
      connector.enqueueCommunicationException(communicationException);
      
      sliceStore.listStop();
      
      assertTrue(connector.dequeueRequest() instanceof ListStopRequest);
   }

   @Test
   public void testVerifyIntegrity()
   {
      testMethod(new RemoteSliceStoreTestMethod() 
      {
         public void execute(Exception protocolException, Exception communicationException) throws Exception
         {
            startSession();
            testVerifyIntegrity(protocolException, communicationException);
         }         
      });                        
   }
   
   private void testVerifyIntegrity(Exception protocolException, Exception communicationException) throws VaultDescriptorException, SliceStoreIOException, IllegalSourceNameException, SliceStoreNotFoundException
   {
      List<SliceInfo> originalCorruptedSlices = new ArrayList<SliceInfo>();
      originalCorruptedSlices.add(new SliceInfo(new SourceName("1"), 1));
      
      connector.enqueueResponse(new IntegrityVerificationResponse(originalCorruptedSlices));
      connector.enqueueProtocolException(protocolException);
      connector.enqueueCommunicationException(communicationException);
      
      List<SliceInfo> slices = new ArrayList<SliceInfo>();
      slices.add(new SliceInfo(new SourceName("2"), 0));
      List<SliceInfo> corruptedSlices = sliceStore.verifyIntegrity(slices);
      
      assertEquals(originalCorruptedSlices.size(), corruptedSlices.size());

      IntegrityVerificationRequest request = (IntegrityVerificationRequest) connector.dequeueRequest();
      
      slices = request.getSlices();
      assertEquals(1, slices.size());
      
      assertEquals("2", slices.get(0).getSourceName().getName());     
   }
   
   private void testMethod(RemoteSliceStoreTestMethod method)
   {
      try
      {
        connector.reset();
         method.execute(NO_EXCEPTION, NO_EXCEPTION);
      }
      catch (Exception e)
      {
         e.printStackTrace();
         fail("Unexpected exception caught: " + e.getMessage());
      }
      
      for(Exception createStoreException : protocolExceptions)
      {
         try
         {
            connector.reset();
            method.execute(createStoreException, NO_EXCEPTION);

            fail("Expected an exception");
         } catch(Exception ex)
         {
            // Expected exception
         }
      }

      for(Exception communicationException : communicationExceptions)
      {
         try
         {
            connector.reset();
            method.execute(NO_EXCEPTION, communicationException);

            fail("Expected an exception");
         } catch(SliceStoreIOException ex)
         {
            // Expected exception
         }
         catch (Throwable e)
         {
            e.printStackTrace();
            fail("Unexpected exception caught: " + e.getMessage());
         }
      }            
   }
}
