//
// Cleversafe open-source code header - Version 1.1 - December 1, 2006
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2007 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 10 W. 35th Street, 16th Floor #84,
// Chicago IL 60616
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// @author: ivolvovski
//
// Date: Oct 29, 2007
//---------------------

package org.cleversafe.layer.slicestore.block;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Properties;

import org.cleversafe.layer.grid.DataSlice;
import org.cleversafe.layer.grid.SliceName;
import org.cleversafe.layer.grid.SourceName;
import org.cleversafe.layer.slicestore.SliceStore;
import org.cleversafe.layer.slicestore.SliceStoreTestBase;
import org.cleversafe.layer.slicestore.SliceStoreTransaction;
import org.cleversafe.layer.slicestore.exceptions.IllegalSourceNameException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreExistsException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreIOException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreNotFoundException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreQuotaException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreTransactionException;
import org.cleversafe.test.TestException;
import org.junit.Test;

public class BlockMultiFileSliceStoreTest extends SliceStoreTestBase
{
   static public final int BLOCK_SIZE = 4096;
   static public final long NUM_BLOCKS = 100L * 1024;

   private List<BlockMultiFileSliceStore.FileStorageDefinition> fsd = null;

   //   @BeforeClass
   //   public static void setUpBeforeClass() throws Exception
   //   {
   //   }

   @Override
   public SliceStore getSliceStore() throws SliceStoreIOException
   {
      // Create Slice Store and start a session

      File dbFile = null; // Must be a directory
      String baseDir = null;

      try
      {
         dbFile = getOutputDirectory();

         // Ensure directory is empty before running test
         deleteDirectory(dbFile);

         getOutputDirectory().mkdirs();
         baseDir = getOutputDirectory().getPath();
      }
      catch (TestException e1)
      {
         e1.printStackTrace();
      }

      long totalBlocks = NUM_BLOCKS;

      String blockSizeStringDef = System.getProperty("org.cleversafe.directory_count");
      int numOfDirs = 0;
      long blockSizes[] = null;
      if (blockSizeStringDef == null)
      {
         blockSizes = new long[1];
         numOfDirs = 0;
      }
      else
      {
         String[] blockSizeStr = blockSizeStringDef.split(":");
         blockSizes = new long[blockSizeStr.length + 1];
         for (int i = 0; i < blockSizeStr.length; i++)
         {
            if (totalBlocks >= 0)
            {
               int s = Integer.parseInt(blockSizeStr[i]);
               blockSizes[i] = s;
               totalBlocks -= s;
               numOfDirs++;
            }
            else
            {
               blockSizes[i] = -1;
            }
         }
      }
      if (totalBlocks >= 0)
      {
         blockSizes[numOfDirs++] = totalBlocks;
      }
      this.fsd = new ArrayList<BlockMultiFileSliceStore.FileStorageDefinition>(numOfDirs);
      for (int i = 0; i < numOfDirs; i++)
      {
         this.fsd.add(new BlockMultiFileSliceStore.FileStorageDefinition(baseDir, blockSizes[i]));
      }

      try
      {
         return new BlockMultiFileSliceStore(this.fsd);
      }
      catch (IOException e)
      {
         throw new SliceStoreIOException("Failed to instatiate slice store", e);
      }
   }

   @Override
   protected void createStore() throws SliceStoreExistsException, SliceStoreIOException
   {
      // TODO: Replace null with vault descriptor

      // Load vault descriptor
      byte[] descriptorBytes = null;

      try
      {
         File vaultDescriptorFile =
               new File(getInputDirectory() + File.separator + "vault-descriptor.xml");
         InputStream in = new FileInputStream(vaultDescriptorFile);
         descriptorBytes = new byte[(int) vaultDescriptorFile.length()];
         in.read(descriptorBytes);
         in.close();
      }
      catch (TestException ex)
      {
      }
      catch (IOException ex)
      {
      }

      //////////////////////////////////////////////////////////////////////////

      this.sliceStore.createStore("block", BLOCK_SIZE, NUM_BLOCKS * BLOCK_SIZE, // 100MB
            this.vaultACL, descriptorBytes);

      for (int i = 0; i < this.fsd.size(); i++)
      {
         File f =
               new File(this.fsd.get(i).getDataPath() + File.separator
                     + BlockMultiFileSliceStore.BLOCK_FILE_BASE + i
                     + BlockMultiFileSliceStore.BLOCK_FILE_EXT);
         logger.info("Observed size=" + f.length());
         logger.info("Expected size=" + this.fsd.get(i).getBlockCapacity()
               * (BLOCK_SIZE + BlockMultiFileSliceStore.getBlockSizeOverhead()));
         assert f.length() == this.fsd.get(i).getBlockCapacity()
               * (BLOCK_SIZE + BlockMultiFileSliceStore.getBlockSizeOverhead());
      }

      logger.info("Created slice store " + this.sliceStore.toString());
   }

   private static boolean deleteDirectory(File dir)
   {
      if (dir.isDirectory())
      {
         String[] children = dir.list();
         for (int i = 0; i < children.length; i++)
         {
            boolean success = deleteDirectory(new File(dir, children[i]));

            if (!success)
            {
               return false;
            }
         }
      }

      // The directory is now empty so delete it
      return dir.delete();
   }

   @Test
   public void testParameterModification()
   {
      BlockMultiFileSliceStore blockStore = (BlockMultiFileSliceStore) this.sliceStore;

      blockStore.setBlockSize(1024);
      assertEquals(1024L, blockStore.getBlockSize());

      blockStore.setListBlockCount(100);
      assertEquals(100, blockStore.getListBlockCount());

      blockStore.setMaxSliceNumber(10000);
      assertEquals(10000L, blockStore.getMaxSliceNumber());

      blockStore.setSynchronous(false);
      assertEquals(false, blockStore.getSynchronous());
   }

   @Test
   public void testReadWriteErrors() throws SliceStoreTransactionException, SliceStoreIOException,
         SliceStoreNotFoundException
   {
      // Test large source name
      SliceStoreTransaction tx = this.sliceStore.createTransaction(1);
      this.sliceStore.beginTransaction(tx);

      DataSlice dataSlice = new DataSlice(new SliceName(String.valueOf(Integer.MAX_VALUE), 0));
      dataSlice.setTransactionId(1);
      dataSlice.setData("This is a test 1".getBytes());

      try
      {
         this.sliceStore.write(dataSlice, false);
         this.sliceStore.commitTransaction(tx);

         fail();
      }
      catch (Exception ex)
      {
         ex.printStackTrace();
      }
      finally
      {
         this.sliceStore.rollbackTransaction(tx);
      }

      try
      {
         this.sliceStore.read(new SliceName(String.valueOf(Integer.MAX_VALUE), 0));
         fail();
      }
      catch (Exception ex)
      {
         ex.printStackTrace();
      }

      // Test large slice data
      tx = this.sliceStore.createTransaction(1);
      this.sliceStore.beginTransaction(tx);

      dataSlice = new DataSlice(new SliceName("0", 0));
      dataSlice.setTransactionId(1);

      byte largeData[] = new byte[4 * 1024 * 1024];
      dataSlice.setData(largeData);

      try
      {
         this.sliceStore.write(dataSlice, false);

         this.sliceStore.commitTransaction(tx);
         fail();
      }
      catch (Exception ex)
      {
         ex.printStackTrace();
         this.sliceStore.rollbackTransaction(tx);
      }

      // Test non-integer source name
      tx = this.sliceStore.createTransaction(1);
      this.sliceStore.beginTransaction(tx);

      dataSlice = new DataSlice(new SliceName("block", 0));
      dataSlice.setTransactionId(1);
      dataSlice.setData("This is a test 1".getBytes());

      try
      {
         this.sliceStore.write(dataSlice, false);

         this.sliceStore.commitTransaction(tx);
         fail();
      }
      catch (Exception ex)
      {
         ex.printStackTrace();
      }

      try
      {
         this.sliceStore.read(new SliceName("block", 0));

         fail();
      }
      catch (Exception ex)
      {
         ex.printStackTrace();
      }
   }

   @Test
   public void testBeyondEOF() throws SliceStoreTransactionException, SliceStoreIOException,
         SliceStoreNotFoundException, IllegalSourceNameException, SliceStoreQuotaException
   {
      if (this.fsd.size() == 1)
      {
         return; // need at least 2 files to check condition
      }
      long firstBlockInSecondFile = this.fsd.get(0).getBlockCapacity();
      SliceName name00 = new SliceName("0", 0);
      SliceName name10 = new SliceName(new Long(firstBlockInSecondFile).toString(), 0);
      byte[] data00 = "This is a block 0".getBytes();
      byte[] data10 = "This is a block 0 in second partition".getBytes();

      DataSlice slice00 = new DataSlice(name00, 99L, data00);
      DataSlice slice10 = new DataSlice(name10, 99L, data10);
      SliceStoreTransaction tx = this.sliceStore.createTransaction(99);

      this.sliceStore.beginTransaction(tx);
      this.sliceStore.write(slice00);
      this.sliceStore.write(slice10);

      this.sliceStore.commitTransaction(tx);

      assert (Arrays.equals(this.sliceStore.read(name00).getData(), data00));
      assert (Arrays.equals(this.sliceStore.read(name10).getData(), data10));
      // last block in the
      assert this.sliceStore.exists(new SliceName(new Long(firstBlockInSecondFile - 1).toString(),
            0)) == false;
      this.sliceStore.listBegin();
      try
      {
         this.sliceStore.listContinue();
      }
      catch (SliceStoreIOException ex)
      {
         fail("IOException");
      }

   }

   @Test
   public void testMultiReadWriteErrors() throws SliceStoreTransactionException,
         SliceStoreIOException, SliceStoreNotFoundException
   {
      // Test large source name
      SliceStoreTransaction tx = this.sliceStore.createTransaction(1);
      this.sliceStore.beginTransaction(tx);

      List<DataSlice> slices = new ArrayList<DataSlice>();

      DataSlice dataSlice = new DataSlice(new SliceName(String.valueOf(Integer.MAX_VALUE), 0));
      dataSlice.setTransactionId(1);
      dataSlice.setData("This is a test 1".getBytes());
      slices.add(dataSlice);

      dataSlice = new DataSlice(new SliceName("1", 0));
      dataSlice.setTransactionId(1);
      dataSlice.setData("This is a test 2".getBytes());
      slices.add(dataSlice);

      try
      {
         this.sliceStore.write(slices, false);

         this.sliceStore.commitTransaction(tx);
         fail();
      }
      catch (Exception ex)
      {
         // Exception is expected
      }
      finally
      {
         this.sliceStore.rollbackTransaction(tx);
      }

      List<SliceName> sliceNames = new ArrayList<SliceName>();
      sliceNames.add(new SliceName(String.valueOf(Integer.MAX_VALUE), 0));
      sliceNames.add(new SliceName("1", 0));
      try
      {
         this.sliceStore.read(sliceNames);
         fail();
      }
      catch (Exception ex)
      {
         // Exception is expected
      }

      // Test large slice data
      tx = this.sliceStore.createTransaction(1);
      this.sliceStore.beginTransaction(tx);

      slices = new ArrayList<DataSlice>();

      dataSlice = new DataSlice(new SliceName(String.valueOf(Integer.MAX_VALUE), 0));
      dataSlice.setTransactionId(1);

      byte largeData[] = new byte[4 * 1024 * 1024];
      dataSlice.setData(largeData);
      slices.add(dataSlice);

      dataSlice = new DataSlice(new SliceName("1", 0));
      dataSlice.setTransactionId(1);
      dataSlice.setData("This is a test 2".getBytes());
      slices.add(dataSlice);

      try
      {
         this.sliceStore.write(slices, false);

         this.sliceStore.commitTransaction(tx);
         fail();
      }
      catch (Exception ex)
      {
         // Exception is expected
         this.sliceStore.rollbackTransaction(tx);
      }

      // Test non-integer source name
      tx = this.sliceStore.createTransaction(1);
      this.sliceStore.beginTransaction(tx);

      slices = new ArrayList<DataSlice>();

      dataSlice = new DataSlice(new SliceName("block", 0));
      dataSlice.setTransactionId(1);
      dataSlice.setData("This is a test 1".getBytes());
      slices.add(dataSlice);

      dataSlice = new DataSlice(new SliceName("1", 0));
      dataSlice.setTransactionId(1);
      dataSlice.setData("This is a test 2".getBytes());
      slices.add(dataSlice);

      try
      {
         this.sliceStore.write(slices, false);

         this.sliceStore.commitTransaction(tx);
         fail();
      }
      catch (Exception ex)
      {
         // Exception is expected
      }

      sliceNames = new ArrayList<SliceName>();
      sliceNames.add(new SliceName("block", 0));
      sliceNames.add(new SliceName("1", 0));
      try
      {
         this.sliceStore.read(sliceNames);

         fail();
      }
      catch (Exception ex)
      {
         // Exception is expected
      }
   }

   @Test
   public void testSavePropertiesFile() throws SliceStoreIOException, TestException,
         FileNotFoundException, IOException
   {
      BlockMultiFileSliceStore blockStore = (BlockMultiFileSliceStore) this.sliceStore;

      Properties props = new Properties();
      props.setProperty("test", "yes");

      File propsFile = new File(getOutputDirectory(), "test.props");
      blockStore.saveProperiesToFile(propsFile, props);

      Properties readProps = new Properties();
      readProps.load(new FileInputStream(propsFile));
      assertEquals("yes", readProps.getProperty("test"));

      // Attempt to save to directory, should fail
      try
      {
         blockStore.saveProperiesToFile(getOutputDirectory(), props);
         fail();
      }
      catch (SliceStoreIOException ex)
      {
         // Exception expected
      }
   }

   @Test
   public void testAdd2Files()
   {
      if (this.fsd.size() == 1)
      { // No split
         return;
      }
      long lastBlock = this.fsd.get(0).getBlockCapacity();
      try
      {

         // Generate a key value pair
         byte val1[] = new byte[4096];
         SliceStoreTestBase.randomizeBuffer(val1);
         byte val2[] = new byte[4096];
         SliceStoreTestBase.randomizeBuffer(val2);
         byte val3[] = new byte[4096];
         SliceStoreTestBase.randomizeBuffer(val3);

         // Create Transaction
         SliceStoreTransaction trans = this.sliceStore.createTransaction(0);
         this.sliceStore.beginTransaction(trans);

         // Store slice
         SliceName name1 = new SliceName(new SourceName(Long.toString(lastBlock - 1)), 1);
         SliceName name2 = new SliceName(new SourceName(Long.toString(lastBlock)), 1);
         SliceName name3 = new SliceName(new SourceName(Long.toString(lastBlock + 1)), 1);
         List<DataSlice> slices =
               Arrays.asList(new DataSlice[]{
                     new DataSlice(name1, trans.getID(), val1),
                     new DataSlice(name2, trans.getID(), val2),
                     new DataSlice(name3, trans.getID(), val3),
               });

         this.sliceStore.write(slices);

         // Commit transaction
         this.sliceStore.commitTransaction(trans);

         // Attempt to get slice
         List<DataSlice> ds = this.sliceStore.read(Arrays.asList(new SliceName[]{
               name1, name2, name3
         }));
         assertTrue(Arrays.equals(val1, ds.get(0).getData()));
         assertTrue(Arrays.equals(val2, ds.get(1).getData()));
         assertTrue(Arrays.equals(val3, ds.get(2).getData()));
      }
      catch (Exception ex)
      {
         fail(ex.getMessage());
      }

   }

   @Test
   public void testSliceStoreSizeBoundary() throws Exception
   {
      BlockMultiFileSliceStore blockStore = (BlockMultiFileSliceStore) this.sliceStore;

      try
      {
         SliceName sliceName = new SliceName(Long.toString(NUM_BLOCKS - 1), 0);
         DataSlice dataSlice = blockStore.read(sliceName);
         assertEquals(dataSlice.getSliceName(), sliceName);
      }
      catch (Exception e)
      {
         fail("Unexpected exception: " + e.getMessage());
      }

      try
      {
         blockStore.read(new SliceName(Long.toString(NUM_BLOCKS), 0));
         fail("Expected SliceStoreIOException");
      }
      catch (SliceStoreIOException expected)
      {
      }
      catch (Exception e)
      {
         fail("Unexpected exception: " + e.getMessage());
      }
   }
}
