//
// Cleversafe open-source code header - Version 1.1 - December 1, 2006
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2007 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 10 W. 35th Street, 16th Floor #84,
// Chicago IL 60616
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// Author: Jason Resch
//
// Date: May 25, 2007
//---------------------

package org.cleversafe.layer.slicestore;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.InputStream;
import java.io.OutputStream;
import java.security.SecureRandom;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.UUID;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;

import org.apache.log4j.Logger;
import org.cleversafe.layer.grid.DataSlice;
import org.cleversafe.layer.grid.SliceName;
import org.cleversafe.layer.grid.SourceName;
import org.cleversafe.layer.slicestore.exceptions.IllegalSourceNameException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreExistsException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreIOException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreLayerException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreNotFoundException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreQuotaException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreTransactionException;
import org.cleversafe.test.BaseTest;
import org.cleversafe.test.IteratedWorkItem;
import org.cleversafe.test.PerformanceTestResult;
import org.cleversafe.test.PerformanceTester;
import org.cleversafe.test.WorkItem;
import org.cleversafe.test.WorkItemProvider;
import org.cleversafe.vault.FileVaultACL;
import org.cleversafe.vault.VaultACL;
import org.cleversafe.vault.VaultDescriptor;
import org.cleversafe.vault.exceptions.VaultDescriptorException;
import org.cleversafe.vault.exceptions.VaultIOException;
import org.cleversafe.vault.exceptions.VaultSecurityException;
import org.junit.After;
import org.junit.Before;
import org.junit.Test;

// TODO: Describe class or interface
public abstract class SliceStoreTestBase extends BaseTest
{
   // //////////////////////////////////////////////////////////////////////////////////////////////
   // Implements the following tests:
   //
   // * simpleAddGetTest - Adds an item to the database, retrieves and compares it to what was put
   // in
   // * doesExistTest - Checks for the existence of an item in the database
   // * addRemoveTest - Adds an item, checks that it is there, then removes it checking that it is
   // gone
   // * transactionIdTest - Ensures that the transacrtionId stored matches the returned value
   // * updateItemTest - Checks the ability to update an existing key, in separate transactions
   // * updateItemSameTransactionTest - Performs update of same key within a single transaction
   // * readNonexistantTest - Attempts to read a non existent item, expecting an exception
   // * rollbackTransactionTest - Begins a transaction, writes to it but then rolls back.
   // * manyItemStoreTest - Adds 1,000 items with different keys and values, checks for all of them
   // * testMultithreadedWriteRead - Uses 10 threads to add 100 items in total, then uses 10 threads
   // to read
   //
   // Not currently implemented, but planned tests:
   //
   // * oneKeyManyValueTest - Stores multiple values for the same key (Obsolete now that only one
   // value per key is maintained)
   // * listRevisionsTest - Does a list of all revisions for a certain slice name (Obsolete now that
   // only one value per key is maintained)
   // * listAllTest - Lists all slices added and all their revisions
   //
   private static final int DATA_SIZE = 4096;

   private static final double MIN_READ_RATE = (100 * 1024 * 1024) / 8;
   private static final double MIN_WRITE_RATE = (10 * 1024 * 1024) / 8;
   private static final long PERFORMANCE_TEST_DATA_SIZE = 100 * 1024 * 1024;

   protected static Logger logger = Logger.getLogger(SliceStoreTestBase.class);

   protected SliceStore sliceStore;
   protected VaultACL vaultACL;

   // Slice name definitions
   private static final String simpleAddGet = "1000";
   private static final String doesExist = "2";
   private static final String addRemoveTest = "3";
   private static final String transactionIdTest = "4";
   private static final String updateItemTest = "5";
   private static final String updateItemSameTransactionTest = "6";
   private static final String readNonexistantTest = "7";
   private static final String rollbackTransactionExist = "8";
   private static final int manyItemStoreTest = 101;
   private static final String multiThreadedReader = "9";

   /**
    * Get the Slice Store to test
    * 
    * @throws SliceStoreIOException
    */
   public abstract SliceStore getSliceStore() throws SliceStoreIOException;

   /*
    * @BeforeClass public static void runBeforeTests() throws Exception { for (int i=0; i<1000;
    * i++) { System.out.println("Running before class!!!!!!!!"); } }
    * 
    * @AfterClass public static void runAfterTests() throws Exception { for (int i=0; i<1000; i++) {
    * System.out.println("Running after class!!!!!!!!"); } }
    */

   @Before
   public void setUp() throws Exception
   {
      this.sliceStore = getSliceStore();
      this.vaultACL = createACL();

      //////////////////////////////////////////////////////////////////////////

      assertFalse(this.sliceStore.isOperational());

      createStore();

      assertFalse(this.sliceStore.isOperational());

      startSession();

      assertTrue(this.sliceStore.isOperational());
   }

   @After
   public void tearDown() throws Exception
   {
      assertTrue(this.sliceStore.isOperational());

      endSession();

      assertFalse(this.sliceStore.isOperational());

      deleteStore();

      assertFalse(this.sliceStore.isOperational());
   }

   abstract protected void createStore() throws SliceStoreExistsException, SliceStoreIOException;

   private void startSession()
   {
      try
      {
         this.sliceStore.startSession();
      }
      catch (Exception ex)
      {
         fail(ex.getMessage());
      }
   }

   private void endSession()
   {
      try
      {
         this.sliceStore.endSession();
      }
      catch (Exception ex)
      {
         fail(ex.getMessage());
      }
   }

   private void deleteStore()
   {
      try
      {
         this.sliceStore.deleteStore();
      }
      catch (Exception ex)
      {
         fail(ex.getMessage());
      }
   }

   private VaultACL createACL() throws Exception
   {
      final String aclFileName = getInputDirectory().getPath() + File.separator + "acl.der";

      assertTrue(new File(aclFileName).exists());

      /*
       * File aclFile = new File(aclFileName); aclFile.getParentFile().mkdirs();
       * 
       * FileOutputStream out = new FileOutputStream(aclFileName);
       * 
       * PasswordCredentials credentials = new PasswordCredentials();
       * credentials.setUsername("username"); credentials.setPassword("password");
       * 
       * System.out.println( "Creating Vault ACL for account: " +
       * credentials.getAccountIdentifier().toString() );
       * 
       * KeyPairGenerator keygen = KeyPairGenerator.getInstance("RSA"); keygen.initialize( 2048 );
       * KeyPair keypair = keygen.generateKeyPair();
       * 
       * VaultACLFactory fact = new VaultACLFactory(); fact.create( out, vaultIdentifier,
       * credentials, keypair.getPublic() ); out.close();
       */

      FileInputStream in = new FileInputStream(aclFileName);
      return new FileVaultACL(in);
   }

   @Test
   public void testSimpleAddGet()
   {

      try
      {

         // Generate a key value pair
         byte val[] = new byte[4096];
         randomizeBuffer(val);

         // Create Transaction
         SliceStoreTransaction trans = this.sliceStore.createTransaction(0);
         this.sliceStore.beginTransaction(trans);

         // Store slice
         SliceName name = new SliceName(new SourceName(simpleAddGet), 1);
         this.sliceStore.write(new DataSlice(name, trans.getID(), val));

         // Commit transaction
         this.sliceStore.commitTransaction(trans);

         // Attempt to get slice
         DataSlice ds = this.sliceStore.read(name);

         assertTrue(Arrays.equals(val, ds.getData()));
      }
      catch (Exception ex)
      {
         fail(ex.getMessage());
      }

   }

   @Test
   public void testDoesExist() throws Exception
   {

      // Generate a key value pair
      SliceName name = new SliceName(new SourceName(doesExist), 1);
      byte[] val = "My Data".getBytes();

      // Remove item to make sure it is not there
      wasRemoved(name);

      // Ensure item is not currently found
      boolean found = this.sliceStore.exists(name);
      assertEquals(false, found);

      // Create Transaction
      SliceStoreTransaction trans = this.sliceStore.createTransaction(0);
      this.sliceStore.beginTransaction(trans);

      // Store slice
      this.sliceStore.write(new DataSlice(name, trans.getID(), val));

      // Commit transaction
      this.sliceStore.commitTransaction(trans);

      // Ensure item is now found
      found = this.sliceStore.exists(name);
      assertEquals(true, found);

   }

   @Test
   public void testAddRemove() throws Exception
   {
      // Generate a key value pair
      SliceName name = new SliceName(new SourceName(addRemoveTest), 1);
      byte[] val = "My Data".getBytes();

      // Remove item to make sure it is not there
      wasRemoved(name);

      // Ensure item is not currently found
      boolean found = this.sliceStore.exists(name);
      assertEquals(false, found);

      // Create Transaction
      SliceStoreTransaction trans = this.sliceStore.createTransaction(0);
      this.sliceStore.beginTransaction(trans);

      // Store slice
      this.sliceStore.write(new DataSlice(name, trans.getID(), val));

      // Commit transaction
      this.sliceStore.commitTransaction(trans);

      // Ensure item is now found
      found = this.sliceStore.exists(name);
      assertEquals(true, found);

      // Now Delete the Item
      assertTrue(wasRemoved(name));

      // Check that it is gone
      found = this.sliceStore.exists(name);
      assertEquals(false, found);

      // Ensure false is returned
      assertFalse(wasRemoved(name));
   }

   @Test
   public void testTransactionId()
   {

      try
      {

         // Generate a key value pair
         byte val[] = new byte[4096];
         randomizeBuffer(val);

         // Create Transaction
         long transId = (long) (Math.random() * Long.MAX_VALUE);
         SliceStoreTransaction trans = this.sliceStore.createTransaction(transId);
         this.sliceStore.beginTransaction(trans);

         // Store slice
         SliceName name = new SliceName(new SourceName(transactionIdTest), 1);
         this.sliceStore.write(new DataSlice(name, trans.getID(), val));

         // Commit transaction
         this.sliceStore.commitTransaction(trans);

         // Attempt to get slice
         DataSlice ds = this.sliceStore.read(name);

         // Verify that the transaction ID matches
         assertEquals(transId, ds.getTransactionId());

      }
      catch (Exception ex)
      {
         fail(ex.getMessage());
      }

   }

   @Test
   public void testUpdateItem()
   {

      try
      {

         // Generate a key value pair
         byte val[] = new byte[4096];
         randomizeBuffer(val);

         // Create Transaction
         SliceStoreTransaction trans = this.sliceStore.createTransaction(0);
         this.sliceStore.beginTransaction(trans);

         // Store slice
         SliceName name = new SliceName(updateItemTest, 0);
         this.sliceStore.write(new DataSlice(name, trans.getID(), val));

         // Commit transaction
         this.sliceStore.commitTransaction(trans);

         // Attempt to get slice
         DataSlice readDs = this.sliceStore.read(name);

         assertTrue(Arrays.equals(val, readDs.getData()));

         // Generate a new key value pair
         byte val2[] = new byte[4096];
         randomizeBuffer(val2);

         // Create another transaction object
         SliceStoreTransaction trans2 = this.sliceStore.createTransaction(1);
         this.sliceStore.beginTransaction(trans2);

         // Perform new write
         this.sliceStore.write(new DataSlice(name, trans2.getID(), val2));

         // Commit transaction
         this.sliceStore.commitTransaction(trans2);

         // Attempt to get slice
         DataSlice ds2 = this.sliceStore.read(name);

         // Ensure the data was updated properly
         assertTrue(Arrays.equals(val2, ds2.getData()));
      }
      catch (Exception ex)
      {
         fail(ex.getMessage());
      }

   }

   @Test
   public void testUpdateItemSameTransaction()
   {

      try
      {

         // Generate a key value pair
         byte val[] = new byte[4096];
         randomizeBuffer(val);

         // Create Transaction
         SliceStoreTransaction trans = this.sliceStore.createTransaction(0);
         this.sliceStore.beginTransaction(trans);

         // Store slice
         SliceName name = new SliceName(updateItemSameTransactionTest, 0);

         // First Write
         this.sliceStore.write(new DataSlice(name, trans.getID(), val));

         // Generate a newkey value pair
         byte val2[] = new byte[4096];
         randomizeBuffer(val2);

         // Second Write
         this.sliceStore.write(new DataSlice(name, trans.getID(), val));

         // Commit transaction
         this.sliceStore.commitTransaction(trans);

         // Attempt to get slice
         DataSlice buff = this.sliceStore.read(name);
         byte[] returned = buff.getData();

         assertEquals(val2.length, returned.length);

         assertTrue(Arrays.equals(val, returned));

      }
      catch (Exception ex)
      {
         fail(ex.getMessage());
      }

   }

   @Test
   public void testReadNonexistant()
   {
      // Store slice
      SliceName name = new SliceName(readNonexistantTest, 0);

      // Assert item does not exist
      try
      {
         assertEquals(false, this.sliceStore.exists(name));
      }
      catch (SliceStoreIOException e)
      {
         fail("Slice store IO exception");
      }
      catch (IllegalSourceNameException e)
      {
         fail("Illegal source name exception");
      }
      catch (SliceStoreNotFoundException e)
      {
         fail("Slice store not found exception");
      }

      // Attempt to get slice
      try
      {
         assertTrue(this.sliceStore.read(name).getData() == null);
      }
      catch (SliceStoreIOException e)
      {
         fail("Slice store IO exception");
      }
      catch (IllegalSourceNameException e)
      {
         fail("Illegal source name exception");
      }
      catch (SliceStoreNotFoundException e)
      {
         fail("Slice store not found exception");
      }
      DataSlice dataSlice = new DataSlice(name);

      // Test large source name
      SliceStoreTransaction tx;
      try
      {
         tx = this.sliceStore.createTransaction(1);

         this.sliceStore.beginTransaction(tx);

         dataSlice.setTransactionId(1);
         byte[] data = "This is a test 1".getBytes();
         dataSlice.setData(data);
         this.sliceStore.write(dataSlice);
         this.sliceStore.commitTransaction(tx);
         byte[] readData = this.sliceStore.read(name).getData();
         assertTrue(Arrays.equals(readData,data));
      }
      catch (SliceStoreTransactionException e)
      {
         // TODO Auto-generated catch block
         e.printStackTrace();
      }
      catch (SliceStoreIOException e)
      {
         // TODO Auto-generated catch block
         e.printStackTrace();
      }
      catch (SliceStoreNotFoundException e)
      {
         // TODO Auto-generated catch block
         e.printStackTrace();
      }
      catch (IllegalSourceNameException e)
      {
         // TODO Auto-generated catch block
         e.printStackTrace();
      }
      catch (SliceStoreQuotaException e)
      {
         // TODO Auto-generated catch block
         e.printStackTrace();
      }
   }

   @Test
   public void testRollbackTransaction()
   {

      try
      {

         // Generate a key value pair
         SliceName name = new SliceName(rollbackTransactionExist, 0);
         byte[] val = "My Data".getBytes();

         // Ensure item is not currently found
         boolean found = this.sliceStore.exists(name);
         assertEquals(false, found);

         // Create Transaction
         SliceStoreTransaction trans = this.sliceStore.createTransaction(0);
         this.sliceStore.beginTransaction(trans);

         // Store slice
         this.sliceStore.write(new DataSlice(name, trans.getID(), val));

         // Rollback transaction
         this.sliceStore.rollbackTransaction(trans);

         // Ensure item is still not found
         found = this.sliceStore.exists(name);
         assertEquals(false, found);

      }
      catch (Exception ex)
      {
         fail(ex.getMessage());
      }

   }

   @Test
   public void testManyItemStore() throws Exception
   {

      // Perform Remove of items
      for (int itr = 0; itr < 1000; itr++)
      {
         SliceName name = new SliceName("" + (manyItemStoreTest + itr), 0);
         wasRemoved(name);
         assertEquals(false, this.sliceStore.exists(name));
      }

      // Create Transaction
      SliceStoreTransaction trans = this.sliceStore.createTransaction(0);
      this.sliceStore.beginTransaction(trans);

      // Perform Store of items
      for (int itr = 0; itr < 1000; itr++)
      {
         SliceName name = new SliceName("" + (manyItemStoreTest + itr), 0);
         byte val[] = Integer.toHexString(itr).getBytes();
         this.sliceStore.write(new DataSlice(name, trans.getID(), val));
      }

      // Commit transaction
      this.sliceStore.commitTransaction(trans);

      // Ensure items exist and have the proper values
      for (int itr = 0; itr < 1000; itr++)
      {
         SliceName name = new SliceName("" + (manyItemStoreTest + itr), 0);
         byte val[] = Integer.toHexString(itr).getBytes();

         assertEquals(true, this.sliceStore.exists(name));
         DataSlice buff = this.sliceStore.read(name);
         byte[] returned = buff.getData();

         assertEquals(val.length, returned.length);
         for (int idx = 0; idx < val.length; idx++)
         {
            assertEquals(val[idx], returned[idx]);
         }
      }

   }

   //@Test
   public void testMultithreadedWriteRead() throws Exception
   {
      int threadPoolSize = 10;

      // Attempt to write items to the database
      try
      {

         ExecutorService executor = Executors.newFixedThreadPool(threadPoolSize);
         List<Future<Boolean>> results = new ArrayList<Future<Boolean>>();

         // Submit the tasks to the executor
         for (int itr = 0; itr < threadPoolSize; itr++)
         {
            Callable<Boolean> writeTask = new MultithreadedWriter(itr);
            results.add(executor.submit(writeTask));
         }

         executor.shutdown();

         // Get the results of the write tasks
         for (Future<Boolean> result : results)
         {
            result.get();
         }

      }
      catch (Exception ex)
      {
         fail("An exception occured during the multithreaded write");
      }

      // Verify items were added by attempting to read them
      try
      {

         ExecutorService executor = Executors.newFixedThreadPool(threadPoolSize);
         List<Future<Boolean>> results = new ArrayList<Future<Boolean>>();

         // Submit the tasks to the executor
         for (int itr = 0; itr < threadPoolSize; itr++)
         {
            Callable<Boolean> readTask = new MultithreadedReader(itr);
            results.add(executor.submit(readTask));
         }

         executor.shutdown();

         // Get the results of the write tasks
         for (Future<Boolean> result : results)
         {
            result.get();
         }

      }
      catch (Exception ex)
      {
         fail("An exception occured during the multithreaded read");
      }

   }

   // Class for multithreaded writing
   public class MultithreadedWriter implements Callable<Boolean>
   {
      private final long transactionId;

      public MultithreadedWriter(long transactionId)
      {
         this.transactionId = transactionId;
      }

      public Boolean call() throws Exception
      {
         final int numberOfWrites = 10;

         try
         {
            SliceStoreTransaction trans =
                  SliceStoreTestBase.this.sliceStore.createTransaction(this.transactionId);
            SliceStoreTestBase.this.sliceStore.beginTransaction(trans);

            for (int itr = 0; itr < numberOfWrites; itr++)
            {
               SliceName name = new SliceName("" + (numberOfWrites * this.transactionId + itr), 0);
               byte[] data = ("MultithreadedWrite_" + this.transactionId).getBytes();

               SliceStoreTestBase.this.sliceStore.write(new DataSlice(name, trans.getID(), data));
            }

            SliceStoreTestBase.this.sliceStore.commitTransaction(trans);

            return new Boolean(true);

         }
         catch (Exception ex)
         {
            logger.error(ex.getMessage());
            logger.error("An exception occured during multithreaded write", ex);
            throw ex;
         }
      }

   };

   // Class for multithreaded reading
   public class MultithreadedReader implements Callable<Boolean>
   {
      private final long transactionId;

      public MultithreadedReader(long transactionId)
      {
         this.transactionId = transactionId;
      }

      public Boolean call() throws Exception
      {
         final int numberOfReads = 10;

         try
         {

            for (int itr = 0; itr < numberOfReads; itr++)
            {
               SliceName name = new SliceName("" + (numberOfReads * this.transactionId + itr), 0);
               byte[] data = ("MultithreadedWrite_" + this.transactionId).getBytes();

               DataSlice buff = SliceStoreTestBase.this.sliceStore.read(name);

               if (buff.getData().length != data.length)
               {
                  throw new Exception("The expected data is of the wrong length");
               }

               for (int idx = 0; idx < data.length; idx++)
               {
                  if (data[idx] != buff.getData()[idx])
                  {
                     throw new Exception("The returned data does not equal the expected.");
                  }
               }

            }

            return new Boolean(true);

         }
         catch (Exception ex)
         {
            logger.debug("An exception occured during multithreaded read", ex);
            throw ex;
         }
      }

   };

   //   @Test
   //   public void testReadPerformance() throws SliceStoreLayerException
   //   {
   //      testReadPerformance(new SimplePerformanceTester());
   //   }

   public void testReadPerformance(PerformanceTester tester) throws SliceStoreLayerException
   {
      byte data[] = new byte[DATA_SIZE];
      randomizeBuffer(data);
      SliceStoreTransaction transaction = this.sliceStore.createTransaction(0);
      this.sliceStore.beginTransaction(transaction);
      this.sliceStore.write(new DataSlice(new SliceName(multiThreadedReader, 0),
            transaction.getID(), data));
      this.sliceStore.commitTransaction(transaction);

      WorkItemProvider workItemProvider = new WorkItemProvider()
      {
         public WorkItem createWorkItem(long id) throws Exception
         {
            return new IteratedWorkItem(id, DATA_SIZE, 15)
            {
               @Override
               protected void performIteration() throws Exception
               {
                  SliceStoreTestBase.this.sliceStore.read(new SliceName(multiThreadedReader, 0));
               }

            };
         }
      };

      try
      {
         PerformanceTestResult result = tester.run(workItemProvider, PERFORMANCE_TEST_DATA_SIZE);

         if (result.getWorkUnitsPerSecond() < MIN_READ_RATE)
         {
            fail("The read rate of " + (result.getWorkUnitsPerSecond() / 1024)
                  + "KB/s was less than the minimum");
         }
      }
      catch (Exception e)
      {
         fail(e.getMessage());
      }
   }

   //   @Test
   //   public void testWritePerformance()
   //   {
   //      testWritePerformance(new SimplePerformanceTester());
   //   }

   public void testWritePerformance(PerformanceTester tester)
   {
      final byte data[] = new byte[DATA_SIZE];
      randomizeBuffer(data);

      WorkItemProvider workItemProvider = new WorkItemProvider()
      {
         public WorkItem createWorkItem(long id) throws Exception
         {
            return new IteratedWorkItem(id, DATA_SIZE, 10)
            {
               private SliceStoreTransaction transaction;

               @Override
               protected void beforeIterations() throws Exception
               {
                  this.transaction = SliceStoreTestBase.this.sliceStore.createTransaction(getId());
                  SliceStoreTestBase.this.sliceStore.beginTransaction(this.transaction);
               }

               @Override
               protected void performIteration() throws Exception
               {
                  SliceStoreTestBase.this.sliceStore.write(new DataSlice(new SliceName(
                        getIterationIdString(), 0), this.transaction.getID(), data));
               }

               @Override
               protected void afterIterations() throws Exception
               {
                  SliceStoreTestBase.this.sliceStore.commitTransaction(this.transaction);
               }
            };
         }
      };

      try
      {
         PerformanceTestResult result = tester.run(workItemProvider, PERFORMANCE_TEST_DATA_SIZE);

         if (result.getWorkUnitsPerSecond() < MIN_WRITE_RATE)
         {
            fail("The write rate of " + (result.getWorkUnitsPerSecond() / 1024)
                  + " KB/s was less than the " + (MIN_WRITE_RATE / 1024) + " KB/s minimum");
         }
      }
      catch (Exception e)
      {
         fail(e.getMessage());
      }
   }

   protected static void randomizeBuffer(byte data[])
   {
      SecureRandom generator = new SecureRandom();
      generator.nextBytes(data);
   }

   // //////////////////////////////////////////////////////////////////////////////////////////////
   // Contractural Tests
   //
   // The following tests validate that the contract (expected behavior and usage scenarious) are
   // consistent across all implementations. For example: isOperational() should only return true
   // when inbetween startSession and endSession. createStore and deleteStore should be called
   // outside of sessions. inactiveTransactions should not be returned by getTransaction.
   //
   // testNoCreateWithinSession - Verifies that createStore cannot be called within a session
   // testSliceStoreExistsDupeCreation - Verifies SliceStoreExists exception is thrown
   // testNoDeleteWithinSession - Verifies that deleteStore cannot be called within a session
   // testStartSessionDuplicateCall - Calls startSession when SliceStore is operational, expecting
   // exception
   // testEndSessionWhenNotOperational - Calls endSession when SliceStore is not operational,
   // expecting an exception
   // testEndSessionTransactionsCleared - Ensures that transactions are not accessible across
   // sessions
   // * testRegisterForNotificationNotOperational - Ensures that registerForNotification cannot be
   // used when store is not operational
   // testCreateTransactionNotOperational - Ensures that create transaction fails when store is not
   // operational
   // testBeginTransactionNotOperational - Ensures that begin transaction fails when store is not
   // operational
   // testBeginTransactionTwice - Ensures that transaction that is active cannot be begun twice
   // testBeginNonExistantTransaction - Ensures that an unknown transaction cannot be begun
   // testCommitTransactionNotOperational - Ensures that the slice store is operational when
   // committing transaction
   // testCommitTransactionTwice - Ensures that a transaction object cannot be committed twice in a
   // row / cannot commit non existent transaction
   // testRollbackTransactionNotOperational - Ensures that a transaction can't be rolled back when
   // not operational
   // testRollbackTransactionTwice - Ensures that a transaction object cannot be rolled back twice /
   // cannot rollback non existent transaction
   // testGetTransactionNotOperational - Ensures that store must be operational to use
   // getTransaction
   // testIsActiveTransactionNotOperational - Ensures that isActiveTransaction cannot be used when
   // store is not operational
   // * testListBeginNotOperational - Ensures that listBegin cannot be used when slice store is not
   // operational
   // * testListBeginWhileListing - Ensures that listBegin cannot be called while currently listing
   // * testListBeginWithSourceNameNotOperational - Ensures that listBegin cannot be used when slice
   // store is not operational
   // * testListBeginWithSourceNameWhileListing - Ensures that listBegin cannot be called while
   // currently listing
   // * testListContinueNotOperational - Ensures that listContinue cannot be used on a non
   // operational store
   // * testListContinueNotListing - Ensures that listContinue must be called while a listing is in
   // progress
   // * testListStopNotOperational - Ensures that listStop cannot be used on a non operational slice
   // store
   // * testListStopNotlisting - Ensures that listStop cannot be used when not listring
   // * testListInProgressNotOperational - Ensures that listInProgress must be called on an
   // operational slice store
   // testExistsNotOperational - Ensures that exists must be called when store is operational
   // testRemoveNotOperational - Ensures that remove must be called when store is operational
   // testWriteNotOperational - Ensures that write must be called when store is operational
   // testWriteInvalidTransaction - Ensures that the transaction supplied with the write is active
   // testWriteManyNotOperational - Ensures that write must be called when store is operational
   // testWriteManyInvalidTransaction - Ensures that the transaction supplied with the write is
   // active
   // testReadNotOperational - Ensures that read must be done when store is operational
   // testReadManyNotOperational - Ensures that reading many must be done when store is operational
   //
   // (*) Not implemented
   //

   @Test
   public void testNoCreateWithinSession()
   {
      try
      {
         createStore();
         fail("Expected an exception to be thrown");
      }
      catch (SliceStoreExistsException e)
      {
         //fail("Slice store exists exception");
         // Expected behavior
      }
      catch (SliceStoreIOException e)
      {
         fail("Unexpected Slice store IO exception");
      }
      catch (IllegalStateException e)
      {
         fail("Unexpected Slice store state exception");
      }
   }

   @Test
   public void testSliceStoreExistsDupeCreation()
   {
      try
      {
         createStore();
         fail("Expected an exception to be thrown");
      }
      catch (SliceStoreExistsException e)
      {
         // Expected behavior
      }
      catch (SliceStoreIOException e)
      {
         fail("Slice store IO exception");
      }
   }

   @Test
   public void testNoDeleteWithinSession()
   {
      try
      {
         this.sliceStore.deleteStore();
         fail("Expected an exception to be thrown");
      }
      catch (IllegalStateException e)
      {
         // Expected behavior
      }
      catch (SliceStoreIOException e)
      {
         fail("Slice Store IO exception");
      }
      catch (SliceStoreNotFoundException e)
      {
         fail("Slice store not found exception");
      }
   }

   @Test
   public void testStartSessionDuplicateCall()
   {
      assertTrue(this.sliceStore.isOperational());
      try
      {
         this.sliceStore.startSession();
         fail("Expected exception to be thrown");
      }
      catch (IllegalStateException e)
      {
         // Expected Behavior
      }
      catch (SliceStoreIOException e)
      {
         fail("SliceStore IO Exception was thrown");
      }
      catch (SliceStoreNotFoundException e)
      {
         fail("Slice store not found exception");
      }
   }

   @Test
   public void testEndSessionWhenNotOperational()
   {
      try
      {
         getSliceStore().endSession();
         fail("Expected exception to be thrown");
      }
      catch (IllegalStateException e)
      {
         // Expected Behavior
      }
      catch (SliceStoreIOException e)
      {
         fail("SliceStore IO Exception was thrown");
      }
   }

   @Test
   public void testEndSessionTransactionsCleared() throws SliceStoreTransactionException,
         SliceStoreIOException, SliceStoreNotFoundException
   {
      final long transID = 12345;

      this.sliceStore.createTransaction(transID);

      SliceStoreTransaction trans = this.sliceStore.getTransaction(transID);

      assertFalse(this.sliceStore.isActiveTransaction(trans));

      assertEquals(transID, trans.getID());

      this.sliceStore.endSession();

      this.sliceStore.startSession();

      try
      {
         this.sliceStore.getTransaction(transID);
         fail("Expected SliceStoreTransactionException to be thrown");
      }
      catch (SliceStoreTransactionException e)
      {
         // Expected behavior
      }
      catch (SliceStoreIOException e)
      {
         fail("Slice Store IO Exception was thrown");
      }
      catch (SliceStoreNotFoundException e)
      {
         fail("Slice store not found exception");
      }
   }

   @Test
   public void testCreateTransactionNotOperational()
   {
      try
      {
         getSliceStore().createTransaction(1);
         fail("Expected exception to be thrown");
      }
      catch (SliceStoreIOException e)
      {
         fail("SliceStore IO Exception was thrown");
      }
      catch (SliceStoreTransactionException e)
      {
         fail("Slice Store Transaction Exception was thrown");
      }
      catch (IllegalStateException e)
      {
         // Expected Behavior
      }
      catch (SliceStoreNotFoundException e)
      {
         fail("Slice store not found exception");
      }
   }

   @Test
   public void testBeginTransactionNotOperational() throws SliceStoreTransactionException,
         SliceStoreIOException, SliceStoreNotFoundException
   {
      SliceStoreTransaction trans = this.sliceStore.createTransaction(1);

      try
      {
         getSliceStore().beginTransaction(trans);
         fail("Expected exception to be thrown");
      }
      catch (IllegalStateException e)
      {
         // Expected Behavior
      }
      catch (SliceStoreIOException e)
      {
         fail("SliceStore IO Exception was thrown");
      }
      catch (SliceStoreTransactionException e)
      {
         fail("Slice Store Transaction Exception was thrown");
      }
      catch (SliceStoreNotFoundException e)
      {
         fail("Slice store not found exception");
      }
   }

   @Test
   public void testBeginTransactionTwice() throws SliceStoreTransactionException,
         SliceStoreIOException, SliceStoreNotFoundException
   {
      final long transID = 769;

      SliceStoreTransaction trans = this.sliceStore.createTransaction(transID);

      assertFalse(this.sliceStore.isActiveTransaction(trans));

      this.sliceStore.beginTransaction(trans);

      assertTrue(this.sliceStore.isActiveTransaction(trans));

      try
      {
         this.sliceStore.beginTransaction(trans);
         fail("Expected exception to be thrown");
      }
      catch (SliceStoreTransactionException e)
      {
         // Expected behavior
      }
      catch (SliceStoreIOException e)
      {
         fail("Slice store IO exception was thrown");
      }
      catch (SliceStoreNotFoundException e)
      {
         fail("Slice store not found exception");
      }
   }

   @Test
   public void testBeginNonExistantTransaction() throws SliceStoreTransactionException,
         SliceStoreIOException, SliceStoreNotFoundException
   {
      final long transactionId = 232;

      SliceStoreTransaction trans = this.sliceStore.createTransaction(transactionId);
      this.sliceStore.beginTransaction(trans);
      this.sliceStore.rollbackTransaction(trans);

      try
      {
         this.sliceStore.beginTransaction(trans);
         fail("Expected exception to be thrown");
      }
      catch (SliceStoreTransactionException e)
      {
         // Expected Behavior
      }
      catch (SliceStoreIOException e)
      {
         fail("Slice store IO exception was thrown");
      }
      catch (SliceStoreNotFoundException e)
      {
         fail("Slice store not found exception");
      }
   }

   @Test
   public void testCommitTransactionNotOperational() throws SliceStoreTransactionException,
         SliceStoreIOException, SliceStoreNotFoundException
   {

      SliceStoreTransaction trans = this.sliceStore.createTransaction(1);
      this.sliceStore.beginTransaction(trans);

      try
      {
         getSliceStore().commitTransaction(trans);
         fail("Expected exception to be thrown");
      }
      catch (IllegalStateException e)
      {
         // Expected Behavior
      }
      catch (SliceStoreIOException e)
      {
         fail("SliceStore IO Exception was thrown");
      }
      catch (SliceStoreTransactionException e)
      {
         fail("Slice Store Transaction Exception was thrown");
      }
      catch (SliceStoreNotFoundException e)
      {
         fail("Slice store not found exception");
      }
   }

   @Test
   public void testCommitTransactionTwice() throws SliceStoreTransactionException,
         SliceStoreIOException, SliceStoreNotFoundException
   {
      final long transID = 652;

      SliceStoreTransaction trans = this.sliceStore.createTransaction(transID);
      this.sliceStore.beginTransaction(trans);

      assertTrue(this.sliceStore.isActiveTransaction(trans));

      assertEquals(transID, this.sliceStore.getTransaction(transID).getID());

      this.sliceStore.commitTransaction(trans);

      assertFalse(this.sliceStore.isActiveTransaction(trans));

      try
      {
         this.sliceStore.getTransaction(transID);
         fail("Expected an exception");
      }
      catch (SliceStoreTransactionException e)
      {
         // Expected behavior
      }
      catch (SliceStoreIOException e)
      {
         fail("Slice store IO exception was thrown");
      }
      catch (SliceStoreNotFoundException e)
      {
         fail("Slice store not found exception");
      }

      try
      {
         this.sliceStore.commitTransaction(trans);
         fail("Expected an exception to be thrown");
      }
      catch (SliceStoreTransactionException e)
      {
         // Expected behavior
      }
      catch (SliceStoreIOException e)
      {
         fail("Slice store IO exception was thrown");
      }
      catch (SliceStoreNotFoundException e)
      {
         fail("Slice store not found exception");
      }
   }

   @Test
   public void testRollbackTransactionNotOperational() throws SliceStoreTransactionException,
         SliceStoreIOException, SliceStoreNotFoundException
   {

      SliceStoreTransaction trans = this.sliceStore.createTransaction(1);
      this.sliceStore.beginTransaction(trans);

      try
      {
         getSliceStore().rollbackTransaction(trans);
         fail("Expected exception to be thrown");
      }
      catch (IllegalStateException e)
      {
         // Expected Behavior
      }
      catch (SliceStoreIOException e)
      {
         fail("SliceStore IO Exception was thrown");
      }
      catch (SliceStoreTransactionException e)
      {
         fail("Slice Store Transaction Exception was thrown");
      }
      catch (SliceStoreNotFoundException e)
      {
         fail("Slice store not found exception");
      }
   }

   @Test
   public void testRollbackTransactionTwice() throws SliceStoreTransactionException,
         SliceStoreIOException, SliceStoreNotFoundException
   {
      final long transID = 955;

      SliceStoreTransaction trans = this.sliceStore.createTransaction(transID);
      this.sliceStore.beginTransaction(trans);

      assertTrue(this.sliceStore.isActiveTransaction(trans));

      assertEquals(transID, this.sliceStore.getTransaction(transID).getID());

      this.sliceStore.rollbackTransaction(trans);

      assertFalse(this.sliceStore.isActiveTransaction(trans));

      try
      {
         this.sliceStore.getTransaction(transID);
         fail("Expected an exception");
      }
      catch (SliceStoreTransactionException e)
      {
         // Expected behavior
      }
      catch (SliceStoreIOException e)
      {
         fail("Slice store IO exception was thrown");
      }
      catch (SliceStoreNotFoundException e)
      {
         fail("Slice store not found exception");
      }

      try
      {
         this.sliceStore.rollbackTransaction(trans);
         fail("Expected an exception to be thrown");
      }
      catch (SliceStoreTransactionException e)
      {
         // Expected behavior
      }
      catch (SliceStoreIOException e)
      {
         fail("Slice store IO exception was thrown");
      }
      catch (SliceStoreNotFoundException e)
      {
         fail("Slice store not found exception");
      }
   }

   @Test
   public void testGetTransactionNotOperational()
   {
      try
      {
         getSliceStore().getTransaction(1);
         fail("Expected exception to be thrown");
      }
      catch (IllegalStateException e)
      {
         // Expected Behavior
      }
      catch (SliceStoreIOException e)
      {
         fail("SliceStore IO Exception was thrown");
      }
      catch (SliceStoreTransactionException e)
      {
         fail("Slice Store Transaction Exception was thrown");
      }
      catch (SliceStoreNotFoundException e)
      {
         fail("Slice store not found exception");
      }
   }

   @Test
   public void testIsActiveTransactionNotOperational() throws SliceStoreTransactionException,
         SliceStoreIOException, SliceStoreNotFoundException
   {
      SliceStoreTransaction trans = this.sliceStore.createTransaction(1);

      try
      {
         getSliceStore().isActiveTransaction(trans);
         fail("Expected exception to be thrown");
      }
      catch (IllegalStateException e)
      {
         // Expected Behavior
      }
   }

   @Test
   public void testExistsNotOperational()
   {
      try
      {
         SliceName name = new SliceName("123", 0);
         getSliceStore().exists(name);
         fail("Expected exception to be thrown");
      }
      catch (IllegalStateException e)
      {
         // Expected Behavior
      }
      catch (SliceStoreIOException e)
      {
         fail("Slice store IO exception was thrown");
      }
      catch (IllegalSourceNameException e)
      {
         fail("Illegal source name exception was thrown");
      }
      catch (SliceStoreNotFoundException e)
      {
         fail("Slice store not found exception");
      }
   }

   @Test
   public void testRemoveNotOperational()
   {
      try
      {
         SliceName name = new SliceName("123", 0);
         getSliceStore().remove(name);
         fail("Expected exception to be thrown");
      }
      catch (IllegalStateException e)
      {
         // Expected Behavior
      }
      catch (SliceStoreIOException e)
      {
         fail("Slice store IO exception was thrown");
      }
      catch (IllegalSourceNameException e)
      {
         fail("Illegal source name exception was thrown");
      }
      catch (SliceStoreNotFoundException e)
      {
         fail("Slice store not found exception");
      }
   }

   @Test
   public void testMultiReadWrite() throws SliceStoreTransactionException, SliceStoreIOException,
         SliceStoreNotFoundException, IllegalSourceNameException, SliceStoreQuotaException
   {
      SliceStoreTransaction tx = this.sliceStore.createTransaction(1);
      this.sliceStore.beginTransaction(tx);

      List<DataSlice> slices = new ArrayList<DataSlice>();

      DataSlice dataSlice = new DataSlice(new SliceName("0", 0));
      dataSlice.setTransactionId(1);
      dataSlice.setData("This is a test 1".getBytes());
      slices.add(dataSlice);

      dataSlice = new DataSlice(new SliceName("1", 0));
      dataSlice.setTransactionId(1);
      dataSlice.setData("This is a test 2".getBytes());
      slices.add(dataSlice);

      this.sliceStore.write(slices, false);

      this.sliceStore.commitTransaction(tx);

      List<SliceName> sliceNames = new ArrayList<SliceName>();
      sliceNames.add(new SliceName("0", 0));
      sliceNames.add(new SliceName("1", 0));
      List<DataSlice> readDataSlices = this.sliceStore.read(sliceNames);

      assertEquals(slices.size(), readDataSlices.size());

   }

   @Test
   public void testWriteNotOperational()
   {
      try
      {
         SliceName name = new SliceName("123", 0);
         SliceStoreTransaction trans = this.sliceStore.createTransaction(457);
         byte[] data = new byte[4096];
         getSliceStore().write(new DataSlice(name, trans.getID(), data));
         fail("Expected exception to be thrown");
      }
      catch (IllegalStateException e)
      {
         // Expected Behavior
      }
      catch (SliceStoreIOException e)
      {
         fail("Slice store IO exception was thrown");
      }
      catch (IllegalSourceNameException e)
      {
         fail("Illegal source name exception was thrown");
      }
      catch (SliceStoreTransactionException e)
      {
         fail("Slice store transaction exception was thrown");
      }
      catch (SliceStoreQuotaException e)
      {
         fail("Slice store quota exception was thrown");
      }
      catch (SliceStoreNotFoundException e)
      {
         fail("Slice store not found exception");
      }
   }

   @Test
   public void testWriteInvalidTransaction() throws SliceStoreTransactionException,
         SliceStoreIOException, SliceStoreNotFoundException
   {
      SliceStoreTransaction trans = this.sliceStore.createTransaction(457);
      this.sliceStore.beginTransaction(trans);
      this.sliceStore.rollbackTransaction(trans);

      try
      {
         SliceName name = new SliceName("123", 0);

         byte[] data = new byte[4096];
         this.sliceStore.write(new DataSlice(name, trans.getID(), data));
         fail("Expected exception to be thrown");
      }
      catch (SliceStoreIOException e)
      {
         fail("Slice store IO exception was thrown");
      }
      catch (IllegalSourceNameException e)
      {
         fail("Illegal source name exception was thrown");
      }
      catch (SliceStoreTransactionException e)
      {
         // Expected behavior
      }
      catch (SliceStoreQuotaException e)
      {
         fail("Slice store quota exception was thrown");
      }
      catch (SliceStoreNotFoundException e)
      {
         fail("Slice store not found exception");
      }
   }

   @Test
   public void testWriteManyNotOperational()
   {
      try
      {
         SliceName name = new SliceName("123", 0);
         SliceStoreTransaction trans = this.sliceStore.createTransaction(457);
         byte[] data = new byte[4096];

         List<DataSlice> dataSources = new ArrayList<DataSlice>();
         dataSources.add(new DataSlice(name, trans.getID(), data));

         getSliceStore().write(dataSources);
         fail("Expected exception to be thrown");
      }
      catch (IllegalStateException e)
      {
         // Expected Behavior
      }
      catch (SliceStoreIOException e)
      {
         fail("Slice store IO exception was thrown");
      }
      catch (IllegalSourceNameException e)
      {
         fail("Illegal source name exception was thrown");
      }
      catch (SliceStoreTransactionException e)
      {
         fail("Slice store transaction exception was thrown");
      }
      catch (SliceStoreQuotaException e)
      {
         fail("Slice store quota exception was thrown");
      }
      catch (SliceStoreNotFoundException e)
      {
         fail("Slice store not found exception");
      }
   }

   @Test
   public void testWriteManyInvalidTransaction() throws SliceStoreTransactionException,
         SliceStoreIOException, SliceStoreNotFoundException
   {
      SliceStoreTransaction trans = this.sliceStore.createTransaction(457);
      this.sliceStore.beginTransaction(trans);
      this.sliceStore.rollbackTransaction(trans);

      try
      {
         SliceName name = new SliceName("123", 0);

         byte[] data = new byte[4096];
         List<DataSlice> dataSources = new ArrayList<DataSlice>();
         dataSources.add(new DataSlice(name, trans.getID(), data));

         this.sliceStore.write(dataSources);

         fail("Expected exception to be thrown");
      }
      catch (SliceStoreIOException e)
      {
         fail("Slice store IO exception was thrown");
      }
      catch (IllegalSourceNameException e)
      {
         fail("Illegal source name exception was thrown");
      }
      catch (SliceStoreTransactionException e)
      {
         // Expected behavior
      }
      catch (SliceStoreQuotaException e)
      {
         fail("Slice store quota exception was thrown");
      }
      catch (SliceStoreNotFoundException e)
      {
         fail("Slice store not found exception");
      }
   }

   @Test
   public void testReadNotOperational()
   {
      try
      {
         SliceName name = new SliceName("123", 0);
         getSliceStore().read(name);
         fail("Expected exception to be thrown");
      }
      catch (IllegalStateException e)
      {
         // Expected Behavior
      }
      catch (SliceStoreIOException e)
      {
         fail("Slice store IO exception was thrown");
      }
      catch (IllegalSourceNameException e)
      {
         fail("Illegal source name exception was thrown");
      }
      catch (SliceStoreNotFoundException e)
      {
         fail("Slice store not found exception");
      }
   }

   @Test
   public void testReadManyNotOperational()
   {
      try
      {
         List<SliceName> names = Arrays.asList(new SliceName("123", 0));
         getSliceStore().read(names);
         fail("Expected exception to be thrown");
      }
      catch (IllegalStateException e)
      {
         // Expected Behavior
      }
      catch (SliceStoreIOException e)
      {
         fail("Slice store IO exception was thrown");
      }
      catch (IllegalSourceNameException e)
      {
         fail("Illegal source name exception was thrown");
      }
      catch (SliceStoreNotFoundException e)
      {
         fail("Slice store not found exception");
      }
   }

   @Test
   public void testGetIdentification()
   {
      assertNotNull(this.sliceStore.getIdentification());
   }

   @Test
   public void testGetVaultDescriptor() throws SliceStoreIOException, VaultIOException,
         VaultSecurityException, VaultDescriptorException, SliceStoreNotFoundException
   {
      VaultDescriptor vaultDescriptor = this.sliceStore.getVaultDescriptor();
      assert vaultDescriptor.getVaultType() != null;
   }

   @Test
   public void testListing() throws SliceStoreTransactionException, SliceStoreIOException,
         SliceStoreNotFoundException, IllegalSourceNameException, SliceStoreQuotaException
   {
      SliceStoreTransaction tx = this.sliceStore.createTransaction(1);
      this.sliceStore.beginTransaction(tx);

      DataSlice dataSlice = new DataSlice(new SliceName(new SourceName("0"), 0));
      dataSlice.setTransactionId(1);
      dataSlice.setData("This is a test".getBytes());
      this.sliceStore.write(dataSlice, false);
      this.sliceStore.commitTransaction(tx);

      this.sliceStore.listBegin();

      int sliceCount = 0;
      while (this.sliceStore.listInProgress())
      {
         List<SliceInfo> slices = this.sliceStore.listContinue();
         sliceCount += slices.size();
      }

      this.sliceStore.listStop();

      assert sliceCount == 1 : "Unexpected slice list";
   }

   @Test
   public void testIntegrityVerification() throws VaultDescriptorException, SliceStoreIOException,
         IllegalSourceNameException, VaultIOException, VaultSecurityException,
         SliceStoreNotFoundException, SliceStoreTransactionException, SliceStoreQuotaException
   {
      SliceStoreTransaction tx = this.sliceStore.createTransaction(1);
      this.sliceStore.beginTransaction(tx);

      DataSlice dataSlice = new DataSlice(new SliceName(new SourceName("0"), 0));
      dataSlice.setTransactionId(1);
      dataSlice.setData("This is a test".getBytes());
      this.sliceStore.write(dataSlice);
      this.sliceStore.commitTransaction(tx);

      List<SliceInfo> corruptedSlices = this.sliceStore.verifyIntegrity();
      assertEquals(1, corruptedSlices.size());
   }

   private boolean wasRemoved(SliceName name) throws SliceStoreIOException,
         IllegalSourceNameException, SliceStoreNotFoundException
   {
      if (this.sliceStore.remove(name))
      {
         return true;
      }
      else
         return false;
   }

}
