//
// Cleversafe open-source code header - Version 1.1 - December 1, 2006
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2007 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 10 W. 35th Street, 16th Floor #84,
// Chicago IL 60616
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// Author: Jason
//
// Date: Aug 8, 2007
//---------------------

package org.cleversafe.layer.slicestore;

import java.io.File;
import java.util.HashMap;
import java.util.Map;
import java.util.UUID;

import org.apache.log4j.Logger;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreNotFoundException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreExistsException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreIOException;
import org.cleversafe.test.BaseTest;
import org.junit.After;
import org.junit.Before;
import org.junit.Test;

// Tests the SliceStoreDescriptorManager
public class SliceStoreDescriptorManagerTest
{
   private static Logger _logger = Logger.getLogger(SliceStoreDescriptorManagerTest.class);

   private SliceStoreDescriptorManager manager = null;
   private UUID accountIdentifier = null;
   private SliceStoreDescriptor descriptor = null;

   @Before
   public void setUp() throws Exception
   {
      _logger.trace("Setting up environment to run unit test");

      // Setting up manager
      this.manager = new SliceStoreDescriptorManager();
      this.manager.setBasePath(System.getProperty(BaseTest.TEST_OUTPUT_PROPERTY, ".")
            + "/org/cleversafe/layer/slicestore/descriptors/");
      this.manager.setFileName("SliceStoreDescriptor.der");

      // Generating Account UUID
      this.accountIdentifier = UUID.nameUUIDFromBytes("DescriptorManagerTest".getBytes());

      // Creating slice store descriptor
      Map<String, String> options = new HashMap<String, String>();
      options.put("key", "value");
      this.descriptor = new SliceStoreDescriptor("vault-type");
   }

   @After
   public void tearDown() throws Exception
   {
      _logger.trace("Tearing down environment after unit test");
      this.manager = null;
   }

   @Test
   public void testSaveDescriptor()
   {
      // Ensure the file does not exist prior to the test
      File descriptorFile = this.manager.getDescriptorFile(this.accountIdentifier);
      descriptorFile.delete();
      junit.framework.Assert.assertFalse(descriptorFile.exists());

      try
      {
         this.manager.saveDescriptor(this.accountIdentifier, this.descriptor);
      }
      catch (SliceStoreExistsException e)
      {
         junit.framework.Assert.fail("Slice Store Exists Exception was thrown");
      }
      catch (SliceStoreIOException e)
      {
         junit.framework.Assert.fail("Slice store IO exception was thrown");
      }

      // Ensure the file was created
      boolean fileExists = this.manager.getDescriptorFile(this.accountIdentifier).exists();
      junit.framework.Assert.assertTrue(fileExists);
   }

   @Test
   public void testLoadDescriptor()
   {
      try
      {
         SliceStoreDescriptor loadedDescriptor =
               this.manager.loadDescriptor(this.accountIdentifier);

         junit.framework.Assert.assertEquals(this.descriptor.getVaultType(),
               loadedDescriptor.getVaultType());
      }
      catch (SliceStoreNotFoundException e)
      {
         junit.framework.Assert.fail("Slice store descriptor not found");
      }
      catch (SliceStoreIOException e)
      {
         junit.framework.Assert.fail("Slice store IO Exception was thrown");
      }
   }

   @Test
   public void testDeleteDescriptor()
   {
      // Ensure the file is there prior to deletion
      boolean fileExists = this.manager.getDescriptorFile(this.accountIdentifier).exists();
      junit.framework.Assert.assertTrue(fileExists);

      try
      {
         this.manager.deleteDescriptor(this.accountIdentifier);
      }
      catch (SliceStoreNotFoundException e)
      {
         junit.framework.Assert.fail("Slice store not found exception was thrown");
      }

      // Ensure the file is no longer there
      fileExists = this.manager.getDescriptorFile(this.accountIdentifier).exists();
      junit.framework.Assert.assertFalse(fileExists);
   }

}
