//
// Cleversafe open-source code header - Version 1.1 - December 1, 2006
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2007 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 10 W. 35th Street, 16th Floor #84,
// Chicago IL 60616
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// Author: Manish Motwani <mmotwani@cleversafe.com>
//         John Quigley <jquigley@cleversafe.com>
//
// Date: May 1, 2007
//---------------------

package org.cleversafe.layer.protocol;

import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;

import java.io.ByteArrayOutputStream;
import java.io.DataOutputStream;

import org.cleversafe.layer.protocol.exceptions.ProtocolSerializationException;
import org.junit.Test;

/**
 * This class unit tests the Protocol's Header class.
 * 
 * @author Manish Motwani, John Quigley
 */
public class GridProtocolHeaderCodecTest
{
   @Test
   public void testHeaderIntConstructor()
   {
      int start = 100;

      int input = start;

      try
      {
         GridProtocolHeaderCodec header =
               new GridProtocolHeaderCodec((short) input++, input++, input++, input++);

         input -= 4;

         assertTrue(header.getProtocolVersion() == input++);
         assertTrue(header.getMessageID() == input++);
         assertTrue(header.getOperationCode() == input++);
         assertTrue(header.getPayloadLength() == input++);
      }
      catch (Exception e)
      {
         fail(e.getMessage());
      }

      System.out.println("Header int constructor test passed!");
   }

   byte[] serialize(GridProtocolHeaderCodec header) throws ProtocolSerializationException
   {
      ByteArrayOutputStream bytes = new ByteArrayOutputStream();
      DataOutputStream out = new DataOutputStream(bytes);
      header.serialize(out);
      return bytes.toByteArray();
   }

   @Test
   public void testSerialize()
   {
      short protocolVersion = 131;
      int messageID = 149218;
      int operationCode = 33;
      int payloadLength = 101014941;

      // Smallest valid magic number
      try
      {
         GridProtocolHeaderCodec header =
               new GridProtocolHeaderCodec(protocolVersion, messageID, operationCode, payloadLength);
         GridProtocolHeaderCodec header2 = new GridProtocolHeaderCodec(serialize(header));
         assertTrue(header2.getMagic() == GridProtocolHeaderCodec.PROTOCOL_MAGIC);
         assertTrue(header2.getProtocolVersion() == protocolVersion);
         assertTrue(header2.getMessageID() == messageID);
         assertTrue(header2.getOperationCode() == operationCode);
         assertTrue(header2.getPayloadLength() == payloadLength);
      }
      catch (Exception e)
      {
         System.out.println(e.getMessage());
         fail(e.getMessage());
      }
   }

   @Test
   public void testHeaderBoundaryProtocolVersionNumbers()
   {
      short smallestValidProtocolVersionNumber = 0;
      short largestValidProtocolVersionNumber = 255; // 2^8 - 1

      short lowerBoundaryFailingProtocolVersionNumber = -1;
      short upperBoundaryFailingProtocolVersionNumber = 256; // 2^8

      int messageID = 149218;
      int payloadLength = 101014941;
      int operationCode = 13;

      // Smallest valid protocol version number
      try
      {
         GridProtocolHeaderCodec header =
               new GridProtocolHeaderCodec(smallestValidProtocolVersionNumber, messageID,
                     operationCode, payloadLength);
         GridProtocolHeaderCodec header2 = new GridProtocolHeaderCodec(serialize(header));
         assertTrue(header2.getMagic() == GridProtocolHeaderCodec.PROTOCOL_MAGIC);
         assertTrue(header2.getProtocolVersion() == smallestValidProtocolVersionNumber);
         assertTrue(header2.getMessageID() == messageID);
         assertTrue(header2.getOperationCode() == operationCode);
         assertTrue(header2.getPayloadLength() == payloadLength);
         System.out.println("Passed smallestValidProtocolVersionNumber!");
      }
      catch (Exception e)
      {
         System.out.println(e.getMessage());
         fail(e.getMessage());
      }

      // Largest valid protocol version number
      try
      {
         GridProtocolHeaderCodec header =
               new GridProtocolHeaderCodec(largestValidProtocolVersionNumber, messageID,
                     operationCode, payloadLength);
         GridProtocolHeaderCodec header2 = new GridProtocolHeaderCodec(serialize(header));
         assertTrue(header2.getMagic() == GridProtocolHeaderCodec.PROTOCOL_MAGIC);
         assertTrue(header2.getProtocolVersion() == largestValidProtocolVersionNumber);
         assertTrue(header2.getMessageID() == messageID);
         assertTrue(header2.getOperationCode() == operationCode);
         assertTrue(header2.getPayloadLength() == payloadLength);
         System.out.println("Passed largestValidProtocolVersionNumber!");
      }
      catch (Exception e)
      {
         System.out.println(e.getMessage());
         fail(e.getMessage());
      }

      // Lower boundary failing protocol version number
      try
      {
         new GridProtocolHeaderCodec(lowerBoundaryFailingProtocolVersionNumber, messageID,
               operationCode, payloadLength);
         fail("Lower boundary failing protocol version number did not fail!");
      }
      catch (Throwable e)
      {
         System.out.println("Passed lowerBoundaryFailingProtocolVersionNumber!");
         System.out.println("   - Exception caught: " + e.getMessage());
      }

      // Upper boundary failing protocol version number
      try
      {
         new GridProtocolHeaderCodec(upperBoundaryFailingProtocolVersionNumber, messageID,
               operationCode, payloadLength);
         fail("Upper boundary failing protocol version number did not fail!");
      }
      catch (Throwable e)
      {
         System.out.println("Passed upperBoundaryFailingProtocolVersionNumber!");
         System.out.println("   - Exception caught: " + e.getMessage());
      }
   }
}
