//
// Cleversafe open-source code header - Version 1.1 - December 1, 2006
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2007 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 10 W. 35th Street, 16th Floor #84,
// Chicago IL 60616
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// Author: Joshua Mullin
//
// Date: Jul 20, 2007
//---------------------

package org.cleversafe.layer.communication.network.policy;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;

import org.cleversafe.layer.communication.exceptions.CommunicationConnectionException;
import org.cleversafe.layer.communication.exceptions.CommunicationException;
import org.cleversafe.layer.communication.exceptions.CommunicationIOException;
import org.cleversafe.layer.communication.exceptions.CommunicationResponseException;
import org.cleversafe.layer.communication.policy.SmartConnectorManager;
import org.cleversafe.layer.communication.policy.SmartConnectorManagerQueue;
import org.cleversafe.layer.protocol.NoopRequest;
import org.cleversafe.layer.protocol.NoopResponse;
import org.cleversafe.layer.protocol.ProtocolOperation;
import org.cleversafe.layer.protocol.Request;
import org.cleversafe.layer.protocol.Response;
import org.cleversafe.util.BoundedThreadPoolExecutor;
import org.junit.Test;

/**
 * Unit test for TestApplicationConnection
 * 
 * @see TestApplicationConnection
 */
public class TestApplicationConnectionTest
{
   private TestApplicationConnection createTestApp()
   {
      List<Request> requestList = new ArrayList<Request>();
      List<Response> responseList = new ArrayList<Response>();
      requestList.add(new TestRequest());
      responseList.add(new TestResponse());
      TestApplicationConnection conn = new TestApplicationConnection(requestList, responseList);
      return conn;
   }

   @Test
   public void testMalformedRequests() throws CommunicationResponseException
   {
      TestApplicationConnection conn = createTestApp();
      Response resp = conn.exchange(new MalformedTestRequest());
      assertEquals("Request wasn't recognized as malformed", conn.lastRequestMalformed(), true);
      assertEquals("Response wasn't malformed", resp instanceof TestResponse, false);
   }

   @Test
   public void testGoodRequests() throws CommunicationResponseException
   {
      TestApplicationConnection conn = createTestApp();

      Response resp = conn.exchange(new TestRequest());
      assertEquals("Request was recognized as malformed", conn.lastRequestMalformed(), false);
      assertEquals("Response was malformed", resp instanceof TestResponse, true);
   }

   @Test
   public void waitForPing() throws InterruptedException, CommunicationException
   {
      // set this down so the test doesn't have to run so long
      SmartConnectorManagerQueue.PING_TIMEOUT = 1000;
      SmartConnectorManagerQueue.initialize();

      List<Request> requestList = new ArrayList<Request>();
      List<Response> responseList = new ArrayList<Response>();
      requestList.add(new NoopRequest());
      requestList.add(new NoopRequest());
      responseList.add(new NoopResponse());
      responseList.add(new NoopResponse());
      TestApplicationConnection conn = new TestApplicationConnection(requestList, responseList);
      SmartConnectorManager scm = new SmartConnectorManager();
      scm.setConnection(conn);
      conn.setManager(scm);
      conn.ensureConnected();

      assertEquals(0, conn.getNumIterations());
      Thread.sleep(500); // not long enough for a ping
      assertEquals(0, conn.getNumIterations());
      Thread.sleep(800);
      assertEquals(1, conn.getNumIterations());
      Thread.sleep(1200);
      assertEquals(2, conn.getNumIterations());
      assertEquals("Ping request as expected", conn.lastRequestMalformed(), false);
      scm.shutdown();
   }

   // after 2 errors occur, the connection should switch to error mode
   @Test
   public void testErrorRequest() throws InterruptedException, CommunicationException
   {
      SmartConnectorManagerQueue.ERROR_TIMEOUT = 1000;
      SmartConnectorManagerQueue.initialize();

      TestApplicationConnection conn =
            new TestApplicationConnection(new ArrayList<Request>(), new ArrayList<Response>());
      SmartConnectorManager scm = new SmartConnectorManager();
      scm.setConnection(conn);
      conn.setManager(scm);

      // initial connection
      assertTrue(conn.ensureConnected());
      
      assertFalse(scm.isErrorState());

      // second error request/response - causes disconnect and error state
      assertFalse(conn.ensureConnected()); // Should be false because connection was established
      conn.exchange(new ErrorRequest());
      assertFalse(conn.isConnected());
      assertTrue(scm.isErrorState());

      try
      {
         conn.ensureConnected();
         fail("Should have thrown error");
      }
      catch (CommunicationException e)
      {
         // this is correct behavior, ensure connected should throw an immediate error
      }

      // wait > timeout (1000 ms) - error manager should reconnect
      Thread.sleep(1200);
      assertTrue(conn.ensureConnected());    // Should be true due to asynchronous reconnection and in ERROR_CLEARED state
      assertFalse(conn.ensureConnected());    // Should be false now that we are in NOT_ERROR state
      assertFalse(scm.isErrorState());
      assertTrue(conn.isConnected());
   }

   // after 2 errors occur, the connection should switch to error mode
   @Test
   public void testIdleTimeout() throws InterruptedException, CommunicationException
   {
      SmartConnectorManagerQueue.IDLE_TIMEOUT = 2000;
      SmartConnectorManagerQueue.PING_TIMEOUT = 1000;
      SmartConnectorManagerQueue.initialize();

      List<Request> requestList = new ArrayList<Request>();
      List<Response> responseList = new ArrayList<Response>();
      requestList.add(new NoopRequest());
      requestList.add(new NoopRequest());
      requestList.add(new NoopRequest());

      responseList.add(new NoopResponse());
      responseList.add(new NoopResponse());
      responseList.add(new NoopResponse());

      TestApplicationConnection conn = new TestApplicationConnection(requestList, responseList);
      SmartConnectorManager scm = new SmartConnectorManager();
      scm.setConnection(conn);
      conn.setManager(scm);
      conn.ensureConnected();

      assertTrue(conn.isConnected());

      Thread.sleep(1200);
      assertEquals(1, conn.getNumIterations());
      assertTrue(conn.isConnected());

      Thread.sleep(1000);
      assertFalse(conn.isConnected());
      assertFalse(scm.isErrorState());
   }
   
   // Call ensure connected for a socket that is disconnected.  See that it returns true (meaning we just reconnected)
   @Test
   public void testSynchronousReconnect() throws CommunicationException
   {
      SmartConnectorManagerQueue.ERROR_TIMEOUT = 1000;
      SmartConnectorManagerQueue.initialize();

      TestApplicationConnection conn = createTestApp();
      SmartConnectorManager scm = new SmartConnectorManager();
      scm.setConnection(conn);
      conn.setManager(scm);

      // initial connection
      conn.ensureConnected();
      
      assertFalse(scm.isErrorState());

      // second error request/response - causes disconnect and error state
      conn.ensureConnected();
      conn.disconnect();
      assertFalse(conn.isConnected());
      assertTrue(conn.ensureConnected());  // Return value of true means we just restored the connection, in ERROR_CLEARED state
      assertFalse(conn.ensureConnected()); // Return value should be false as we are in the NOT_ERROR state
      assertTrue(conn.isConnected());
      assertFalse(scm.isErrorState());
   }

   // Cause a connection to enter an error state.  Wait for reconnection and subsequent ERROR_CLEARED status, then call
   // onError() to cause a state transition from ERROR_CLEARED to ERROR.  Ensure that the connection can still be restored and
   // used normally.
   @Test
   public void testErrorAfterReconnect() throws InterruptedException, CommunicationException
   {
      SmartConnectorManagerQueue.ERROR_TIMEOUT = 1000;
      SmartConnectorManagerQueue.initialize();

      TestApplicationConnection conn =
            new TestApplicationConnection(new ArrayList<Request>(), new ArrayList<Response>());
      SmartConnectorManager scm = new SmartConnectorManager();
      scm.setConnection(conn);
      conn.setManager(scm);

      // initial connection
      assertTrue(conn.ensureConnected());
      
      assertFalse(scm.isErrorState());

      // second error request/response - causes disconnect and error state
      assertFalse(conn.ensureConnected()); // Should be false because connection was established
      conn.exchange(new ErrorRequest());
      assertFalse(conn.isConnected());
      assertTrue(scm.isErrorState());

      try
      {
         conn.ensureConnected();
         fail("Should have thrown error");
      }
      catch (CommunicationException e)
      {
         // this is correct behavior, ensure connected should throw an immediate error
      }

      // wait > timeout (1000 ms) - error manager should reconnect
      Thread.sleep(1200);
      
      // We assume that the connection has been restored by this point and that we are now in the ERROR_CLEARED state
      // We now force a transition back to the onError state.
      conn.getManager().onError();
      
      try
      {
         conn.ensureConnected();
         fail("Should have thrown error");
      }
      catch (CommunicationException e)
      {
         // this is correct behavior, ensure connected should throw an immediate error
      }
      
      // wait > timeout (1000 ms) - error manager should reconnect
      Thread.sleep(1200);
      
      assertTrue(conn.ensureConnected());    // Should be true due to asynchronous reconnection and in ERROR_CLEARED state
      assertFalse(conn.ensureConnected());    // Should be false now that we are in NOT_ERROR state
      assertFalse(scm.isErrorState());
      assertTrue(conn.isConnected());
   }
   
   // Class used for testing race conditions
   class EnsureConnectedTask implements Callable<Boolean>
   {
      private TestApplicationConnection _conn = null;

      public EnsureConnectedTask(TestApplicationConnection conn)
      {
         this._conn = conn;
      }

      public Boolean call() throws CommunicationIOException, CommunicationConnectionException
      {
         return this._conn.ensureConnected();
      }
   }

   // Creates 10 threads which attempt to simultaneously call ensureConnected on a disconnected connection.  Ensure that
   // only one thread returns true, signifying that only one thread attempted to and successfully restored the connection.
   @Test
   public void testReconnectRaceConditionSucceedingReconnect()
   {
      final int numThreads = 10;
      final int numRequests = 50;
      ExecutorService executor = new BoundedThreadPoolExecutor("ensure-connected-test", numThreads);

      // Create connection
      TestApplicationConnection conn = createTestApp();
      SmartConnectorManager scm = new SmartConnectorManager();
      scm.setConnection(conn);
      conn.setManager(scm);
      
      // Ensure we are not connected yet, nor in an error state
      assertFalse(conn.isConnected());
      assertFalse(scm.isErrorState());
      
      // Add tasks to execute ensure connected
      List<Callable<Boolean>> tasks = new ArrayList<Callable<Boolean>>(numRequests);
      for (int i = 0; i < numRequests; i++)
      {
         tasks.add(i, new EnsureConnectedTask(conn));
      }

      // Get results
      List<Future<Boolean>> results = null;
      try
      {
         results = executor.invokeAll(tasks);
      }
      catch (final InterruptedException e)
      {
         fail(e.getMessage());
      }
      
      executor.shutdown();
      
      // We should now be connected
      assertTrue(conn.isConnected());

      // Count the return values from the various tasks
      int trueCount = 0;
      int falseCount = 0;
      for (Future<Boolean> result : results)
      {
         try
         {
            Boolean ensureConnectedReturnValue = result.get();
            if (ensureConnectedReturnValue.booleanValue() == true)
            {
               trueCount++;
            }
            else
            {
               falseCount++;
            }
         }
         catch (InterruptedException e)
         {
            fail(e.getMessage());
         }
         catch (ExecutionException e)
         {
            fail(e.getMessage());
         }
      }
      
      // Validate we got the expected return values
      assertEquals(1, trueCount);
      assertEquals(numRequests-1, falseCount);
   }
   
   // Like the previous test only the re-connection attempt fails.  Check for an exception to be thrown by all threads within
   // a short period of time.
   @Test
   public void testReconnectRaceConditionFailingReconnect()
   {
      final int numThreads = 10;
      final int numRequests = 50;
      ExecutorService executor = new BoundedThreadPoolExecutor("ensure-connected-test", numThreads);

      // Create connection
      TestApplicationConnection conn = createTestApp();
      SmartConnectorManager scm = new SmartConnectorManager();
      scm.setConnection(conn);
      conn.setManager(scm);
      
      // Ensure we are not connected yet, nor in an error state
      assertFalse(conn.isConnected());
      assertFalse(scm.isErrorState());

      // Set connection such that connection attempts always fail
      conn.setFailReconnects(true);
      
      // Add tasks to execute ensure connected
      List<Callable<Boolean>> tasks = new ArrayList<Callable<Boolean>>(numRequests);
      for (int i = 0; i < numRequests; i++)
      {
         tasks.add(i, new EnsureConnectedTask(conn));
      }

      // Get results
      long startTime = System.currentTimeMillis();
      List<Future<Boolean>> results = null;
      try
      {
         results = executor.invokeAll(tasks);
      }
      catch (final InterruptedException e)
      {
         fail(e.getMessage());
      }
      long endTime = System.currentTimeMillis();
      
      executor.shutdown();
      
      // We should not be connected
      assertFalse(conn.isConnected());

      // Count the return values from the various tasks
      int trueCount = 0;
      int falseCount = 0;
      int exceptionCount = 0;
      for (Future<Boolean> result : results)
      {
         try
         {
            Boolean ensureConnectedReturnValue = result.get();
            if (ensureConnectedReturnValue.booleanValue() == true)
            {
               trueCount++;
            }
            else
            {
               falseCount++;
            }
         }
         catch (InterruptedException e)
         {
            fail(e.getMessage());
         }
         catch (ExecutionException e)
         {
            exceptionCount++;
         }
      }
      
      // Validate we got the expected return values, should be zero due to exceptions thrown
      assertEquals(0, trueCount);
      assertEquals(0, falseCount);
      assertEquals(numRequests, exceptionCount);
      
      // Validate total time was less than 6 seconds
      // The test application connection takes 3 seconds to connect before failing by
      // returning an exception.  This is an attempt to ensure it was only called once.
      assertTrue(endTime - startTime < (3000 * 2));
      assertTrue(endTime - startTime >= 3000);
      
   }
   
   // Like the previous test only the threads begin on a connection in the state of ERROR_CLEARED, verifies single return value of
   // true while the rest are false.
   @Test
   public void testReconnectRaceConditionAfterReconnect() throws InterruptedException
   {
      SmartConnectorManagerQueue.ERROR_TIMEOUT = 1000;
      SmartConnectorManagerQueue.initialize();

      final int numThreads = 10;
      final int numRequests = 50;
      ExecutorService executor = new BoundedThreadPoolExecutor("ensure-connected-test", numThreads);

      // Create connection
      TestApplicationConnection conn = createTestApp();
      SmartConnectorManager scm = new SmartConnectorManager();
      scm.setConnection(conn);
      conn.setManager(scm);
      
      // Ensure we are not connected yet, nor in an error state
      assertFalse(conn.isConnected());
      assertFalse(scm.isErrorState());
      
      // second error request/response - causes disconnect and error state
      conn.exchange(new ErrorRequest());
      assertFalse(conn.isConnected());
      assertTrue(scm.isErrorState());

      // wait > timeout (1000 ms) - error manager should reconnect putting us in the ERROR_CLEARED state
      Thread.sleep(1200);
      
      // Add tasks to execute ensure connected
      List<Callable<Boolean>> tasks = new ArrayList<Callable<Boolean>>(numRequests);
      for (int i = 0; i < numRequests; i++)
      {
         tasks.add(i, new EnsureConnectedTask(conn));
      }

      // Get results
      List<Future<Boolean>> results = null;
      try
      {
         results = executor.invokeAll(tasks);
      }
      catch (final InterruptedException e)
      {
         fail(e.getMessage());
      }
      
      executor.shutdown();
      
      // We should now be connected
      assertTrue(conn.isConnected());

      // Count the return values from the various tasks
      int trueCount = 0;
      int falseCount = 0;
      for (Future<Boolean> result : results)
      {
         try
         {
            Boolean ensureConnectedReturnValue = result.get();
            if (ensureConnectedReturnValue.booleanValue() == true)
            {
               trueCount++;
            }
            else
            {
               falseCount++;
            }
         }
         catch (InterruptedException e)
         {
            fail(e.getMessage());
         }
         catch (ExecutionException e)
         {
            fail(e.getMessage());
         }
      }
      
      // Validate we got the expected return values
      assertEquals(1, trueCount);
      assertEquals(numRequests-1, falseCount);
   }
   

   public static class TestResponse implements Response
   {
      private String seedData;

      public boolean equals(TestResponse resp)
      {
         return (this.seedData.equals(resp.getSeedData())) ? true : false;
      }

      public String getSeedData()
      {
         return seedData;
      }

      public Exception getException()
      {
         return null;
      }

      public boolean getExceptionFlag()
      {
         return false;
      }

      public void setException(Exception exception)
      {
      }

      public ProtocolOperation getOperation()
      {
         return null;
      }

      public boolean isRequest()
      {
         return false;
      }

      public boolean isResponse()
      {
         return false;
      }

      public boolean isUnsolicited()
      {
         return false;
      }

   }

   public static class MalformedTestRequest extends TestRequest
   {
      private boolean newBoolean;

      public MalformedTestRequest()
      {
         this.myString = "This is not a string";
         this.myDouble = 27.8;
         this.myInt = 28;
         this.myBoolean = false;
         this.newBoolean = true;
      }

      public boolean getNewBoolean()
      {
         return newBoolean;
      }
   }

   /** Just a marker for the outside class */
   public static class ErrorRequest extends TestRequest
   {

   }

   public static class TestRequest implements Request
   {
      protected String myString;
      protected double myDouble;
      protected int myInt;
      protected boolean myBoolean;

      public TestRequest()
      {
         this.myString = "This is a String";
         this.myDouble = 67.5;
         this.myInt = 22;
         this.myBoolean = true;
      }

      public ProtocolOperation getOperation()
      {
         // TODO Auto-generated method stub
         return null;
      }

      public boolean isRequest()
      {
         // TODO Auto-generated method stub
         return false;
      }

      public boolean isResponse()
      {
         // TODO Auto-generated method stub
         return false;
      }

      public boolean isUnsolicited()
      {
         // TODO Auto-generated method stub
         return false;
      }

   }
}
