//
// Cleversafe open-source code header - Version 1.1 - December 1, 2006
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2007 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 10 W. 35th Street, 16th Floor #84,
// Chicago IL 60616
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// Author: Joshua Mullin
//
// Date: Jul 20, 2007
//---------------------

package org.cleversafe.layer.communication.network.policy;

import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.atomic.AtomicInteger;

import org.cleversafe.exceptions.NotImplementedException;
import org.cleversafe.layer.communication.Connector;
import org.cleversafe.layer.communication.exceptions.CommunicationConnectionException;
import org.cleversafe.layer.communication.exceptions.CommunicationIOException;
import org.cleversafe.layer.communication.exceptions.CommunicationInterruptedException;
import org.cleversafe.layer.communication.exceptions.CommunicationResponseException;
import org.cleversafe.layer.communication.exceptions.CommunicationTransmissionException;
import org.cleversafe.layer.communication.exceptions.NotConnectedException;
import org.cleversafe.layer.communication.network.IPNetworkConnector;
import org.cleversafe.layer.protocol.ProtocolMessage;
import org.cleversafe.layer.protocol.ProtocolOperation;
import org.cleversafe.layer.protocol.Request;
import org.cleversafe.layer.protocol.Response;
import org.cleversafe.serialization.ProtocolMessageFactory;

/**
 * A fake network connection where request and response objects can be passed and requests can be
 * checked against expected requests to make sure requests aren't malformed
 * 
 * @see Connector
 * 
 */
public class TestApplicationConnection extends IPNetworkConnector
{
   private List<Request> requests;
   private List<Response> responses;
   private List<Boolean> malformedRequests;
   private int numIterations;
   private Request lastRequest;
   private boolean failReconnects = false;
   private boolean _connected;
   private AtomicInteger numOutstandingExchanges = new AtomicInteger();

   /**
    * Constructor for matching requests with responses. Lists should be sorted in the order the
    * requests should happen.
    * 
    * @param requestList
    *           sorted list of requests
    * @param responseList
    *           sorted list of corresponding responses
    */
   public TestApplicationConnection(List<Request> requestList, List<Response> responseList)
   {
      this.malformedRequests = new ArrayList<Boolean>();
      this.requests = requestList;
      this.responses = responseList;
      int i;
      for (i = 0; i < this.requests.size(); i++)
      {
         this.malformedRequests.add(new Boolean(false));
      }
      this.numIterations = 0;
   }

   public int getNumIterations()
   {
      return numIterations;
   }

   public Request lastRequest()
   {
      return lastRequest;
   }

   public Response exchange(Request req)
   {
      lastRequest = req;
      this.numOutstandingExchanges.incrementAndGet();
      try
      {
         if (req instanceof TestApplicationConnectionTest.ErrorRequest)
         {
            getManager().onError();
            return null;
         }
         else if (!requestsEqual(req, this.requests.get(this.numIterations)))
         {
            this.malformedRequests.set(this.numIterations, new Boolean(true));
            numIterations++;
         }
         else
         {
            return this.responses.get(this.numIterations++);
         }
      }
      catch (ClassNotFoundException e)
      {
         // Do nothing.
      }
      finally
      {
         this.numOutstandingExchanges.decrementAndGet();
      }

      return new BadTestResponse();
   }

   public boolean lastRequestMalformed()
   {
      return this.malformedRequests.get(this.numIterations - 1);
   }

   private static boolean requestsEqual(Request given, Request specified)
         throws ClassNotFoundException
   {
      boolean result = true;
      // Use Reflection to get the private data for comparison
      // TODO: Implement a Request.equals method to avoid this
      Class<? extends Request> givenClass = given.getClass();
      Class<? extends Request> specifiedClass = specified.getClass();

      // Verify that the classes are of the same type
      if (!(givenClass.equals(specifiedClass)))
      {
         result = false;
      }
      Field givenFields[] = givenClass.getDeclaredFields();
      Field specifiedFields[] = specifiedClass.getDeclaredFields();

      // Now verify that their members have the same values
      if (givenFields.length == specifiedFields.length)
      {
         int i;
         for (i = 0; i < givenFields.length; i++)
         {
            if (!(givenFields[i].getGenericType().equals(specifiedFields[i].getGenericType())))
            {
               result = false;
            }
         }
      }
      else
      {
         result = false;
      }

      return result;
   }

   public void shutdown()
   {
      throw new NotImplementedException("auto-generated stub must be implemented");
   }

   public void initialize()
   {
   }

   public boolean isConnected()
   {
      return _connected;
   }
   

   public boolean isFailReconnects()
   {
      return failReconnects;
   }

   public void setFailReconnects(boolean failReconnects)
   {
      this.failReconnects = failReconnects;
   }

   public String getLocalAddress()
   {
      throw new NotImplementedException("auto-generated stub must be implemented");
   }

   public String getRemoteAddress()
   {
      return "remote test address";
   }

   public boolean ensureConnected() throws CommunicationIOException,
         CommunicationConnectionException
   {
      return getManager().ensureConnected();
   }

   public void connect() throws CommunicationIOException, CommunicationConnectionException
   {
      if (isFailReconnects() == false)
      {
         _connected = true;
      }
      else
      {
         try
         {
            Thread.sleep(3000);
         }
         catch (InterruptedException e)
         {
            throw new RuntimeException("Unexpected thread interrupted exception");
         }
         throw new CommunicationIOException("Connection failed - isFailReconnects() is true");
      }
   }

   public void disconnect() throws CommunicationIOException
   {
      _connected = false;
   }

   /**
    * Generic class representing a bad response. If a program receives one of these, It shouldn't
    * know what to do with it and should throw an error.
    */
   public static class BadTestResponse implements Response
   {
      // Stubs generated to match the interface.
      public BadTestResponse()
      {
         // Default Constructor
      }

      public Exception getException()
      {
         return null;
      }

      public boolean getExceptionFlag()
      {
         return false;
      }

      public void setException(Exception exception)
      {
      }

      public ProtocolOperation getOperation()
      {
         return null;
      }

      public boolean isRequest()
      {
         return false;
      }

      public boolean isResponse()
      {
         return false;
      }

      public boolean isUnsolicited()
      {
         return false;
      }

   }

   public Response exchange(Request request, int timeout) throws CommunicationIOException,
         NotConnectedException, CommunicationInterruptedException, CommunicationResponseException,
         CommunicationTransmissionException
   {
      throw new NotImplementedException("not yet implemented");
   }

   public String getIdentification()
   {
      return "Test Connector";
   }

   @Override
   public ProtocolMessage getNotification() throws CommunicationInterruptedException
   {
      return null;
   }

   @Override
   public ProtocolMessage getNotification(int timeout) throws CommunicationInterruptedException
   {
      return null;
   }

   public ProtocolMessageFactory getProtocolMessageFactory()
   {
      throw new NotImplementedException("not implemented");
   }

   public int getNumOutstandingExchanges()
   {
      return this.numOutstandingExchanges.get();
   }
}
