
package org.cleversafe.ida;

//
//Cleversafe open-source code header - Version 1.1 - December 1, 2006
//
//Cleversafe Dispersed Storage(TM) is software for secure, private and
//reliable storage of the world's data using information dispersal.
//
//Copyright (C) 2005-2007 Cleversafe, Inc.
//
//This program is free software; you can redistribute it and/or
//modify it under the terms of the GNU General Public License
//as published by the Free Software Foundation; either version 2
//of the License, or (at your option) any later version.
//
//This program is distributed in the hope that it will be useful,
//but WITHOUT ANY WARRANTY; without even the implied warranty of
//MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//GNU General Public License for more details.
//
//You should have received a copy of the GNU General Public License
//along with this program; if not, write to the Free Software
//Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
//USA.
//
//Contact Information: Cleversafe, 10 W. 35th Street, 16th Floor #84,
//Chicago IL 60616
//email licensing@cleversafe.org
//
//END-OF-HEADER
//-----------------------
//Author: Vance Thornton
//
//---------------------

import static org.junit.Assert.assertEquals;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.util.ArrayList;
import java.util.List;

import org.cleversafe.ida.exceptions.IDAInvalidParametersException;
import org.cleversafe.util.math.CombinationGenerator;
import org.junit.Test;

public abstract class InformationDispersalTestBase
{
   private static int CHUNK_SIZE = 4096;
   private static int NUM_SLICES = 16;
   private static int THRESHOLD = 12;
   
   private static int MIN_CHUNK_SIZE = 1;
   private static int MAX_CHUNK_SIZE = 8192;
   
   private static int MIN_NUM_SLICES = 1;
   private static int MAX_NUM_SLICES = 32;
   
   private static int RANDOM_CONFIGURATION_RUNS = 10000;

   public abstract InformationDispersalCodec getIDA() throws
         IDAInvalidParametersException;

   /**
    * Tests all permutations of having a threshold number of slices
    * 
    * @throws Exception
    */
   @Test
   public void testEncodeDecodePermutations() throws Exception
   {
      // Generate a random 4KB data buffer
      byte data[] = new byte[CHUNK_SIZE];
      randomizeBuffer(data);

      // Execute encode / decode operation
      InformationDispersalCodec ida = this.getIDA();
      System.out.print("Testing IDA for referral " + ida.getName() + "... ");
      
      ida.setNumSlices(NUM_SLICES);
      ida.setThreshold(THRESHOLD);
      ida.setChunkSize(CHUNK_SIZE);
      ida.initialize();
      
      executeEncodeDecode(data, ida.getEncoder(), ida.getDecoder());
   }


   /**
    * Uses the default numSlices and threshold but varies the chunk size
    * from min to max, verifying correct encode and decode as well as the 
    * encoded size estimation
    * 
    * @throws Exception 
    */
   @Test
   public void testRangedChunkSize() throws Exception
   {
      for (int size = MIN_CHUNK_SIZE; size <= MAX_CHUNK_SIZE; size++)
      {
      
         // Generate a random data buffer
         byte data[] = new byte[size];
         randomizeBuffer(data);
   
         // Execute encode / decode operation
         InformationDispersalCodec ida = this.getIDA();
         
         ida.setNumSlices(NUM_SLICES);
         ida.setThreshold(THRESHOLD);
         ida.setChunkSize(size);
         
         ida.initialize();
         
         int dispersedSize = (int) ida.getDispersedSize(size);
         
         executeEncodeDecodeOnce(data, ida.getEncoder(), ida.getDecoder(), dispersedSize);
      }
   }
   

   /**
    * Uses the default chunk size but varies the numSlices and threshold
    * from min to max, verifying correct encode and decode as well as the 
    * encoded size estimation
    * 
    * @throws Exception 
    */
   @Test
   public void testRangedGridDimensions() throws Exception
   {
      // Generate a random 4KB data buffer
      byte data[] = new byte[CHUNK_SIZE];
      randomizeBuffer(data);
      
      for (int numSlices = MIN_NUM_SLICES; numSlices <= MAX_NUM_SLICES; numSlices++)
      {
         for (int threshold = 1; threshold <= numSlices; threshold++)
         {
            // Execute encode / decode operation
            InformationDispersalCodec ida = this.getIDA();
            
            ida.setNumSlices(numSlices);
            ida.setThreshold(threshold);
            ida.setChunkSize(CHUNK_SIZE);
            
            ida.initialize();
            
            int dispersedSize = (int) ida.getDispersedSize(CHUNK_SIZE);
            
            executeEncodeDecodeOnce(data, ida.getEncoder(), ida.getDecoder(), dispersedSize);
         }
      }
   }
   
   /**
    * Uses a variety of different data lengths keeping everything else constant
    * to ensure that the IDA can handle messages of various lengths
    * 
    * @throws Exception 
    */
   @Test
   public void testPaddingMechanism() throws Exception
   {
      InformationDispersalCodec ida = this.getIDA();
      
      // Execute encode / decode operation
      ida.setNumSlices(NUM_SLICES);
      ida.setThreshold(THRESHOLD);
      ida.setChunkSize(CHUNK_SIZE);
      
      ida.initialize();
      
      InformationDispersalEncoder encoder = ida.getEncoder();
      InformationDispersalDecoder decoder = ida.getDecoder();
      
      for (int size = 0; size <= CHUNK_SIZE*3; size++)
      {
         // Generate a random data buffer
         byte data[] = new byte[size];
         randomizeBuffer(data);

         int dispersedSize = (int) ida.getDispersedSize(size);
         
         executeEncodeDecodeOnce(data, encoder, decoder, dispersedSize);
      }
   }
   
   /**
    * Randomly selects chunk size, numSlices, threshold, and length of message
    * and attempts an encode and decode checking size against expected
    * 
    * @throws Exception 
    */
   @Test
   public void testRandomConfigurations() throws Exception
   {      
      for (int cnt = 0; cnt < RANDOM_CONFIGURATION_RUNS; cnt++)
      {
         int chunkSize = randRange(MIN_CHUNK_SIZE, MAX_CHUNK_SIZE);
         int numSlices = randRange(MIN_NUM_SLICES, MAX_NUM_SLICES);
         int threshold = randRange(1, numSlices);
         int dataSize = randRange(1, chunkSize * 3);
         
         // Generate Data
         byte data[] = new byte[dataSize];
         randomizeBuffer(data);
         
         // Setup IDA
         InformationDispersalCodec ida = this.getIDA();
         
         ida.setNumSlices(numSlices);
         ida.setThreshold(threshold);
         ida.setChunkSize(chunkSize);
         
         ida.initialize();
         
         int dispersedSize = (int) ida.getDispersedSize(dataSize);
         
         executeEncodeDecodeOnce(data, ida.getEncoder(), ida.getDecoder(), dispersedSize);
      }
   }
   
   
   /**
    * Tests the update method of the IDA's encoder, makes many calls to
    * update to encode a message much larger than the chunk size.
    * 
    * @throws Exception
    */
   /*
   @Test
   public void testEncodeUpdate() throws Exception
   {
      InformationDispersalCodec ida = this.getIDA();
      
      // Setup IDA
      ida.setNumSlices(NUM_SLICES);
      ida.setThreshold(THRESHOLD);
      ida.setChunkSize(CHUNK_SIZE);
      
      ida.initialize();
      
      InformationDispersalEncoder encoder = ida.getEncoder();
      InformationDispersalDecoder decoder = ida.getDecoder();
      
      // Generate a random CHUNK_SIZE*100 sized data buffer
      byte data[] = new byte[CHUNK_SIZE*100];
      randomizeBuffer(data);
      
      ByteArrayInputStream inputData = new ByteArrayInputStream(data);
      List<ByteArrayOutputStream> outputData = new ArrayList<ByteArrayOutputStream>();
      for (int cnt=0; cnt<ida.getNumSlices(); cnt++)
      {
         outputData.add(new ByteArrayOutputStream());
      }
      
      // Make repeated calls to update, using various lengths of data at a time,
      // some smaller than a single chunk, some larger, some equal, some zero
      byte[] randomSizedArray;
      List<byte[]> encodedData = new ArrayList<byte[]>();
      while (inputData.available() > 0)
      {
         int randomReadSize = (int)(Math.random()*CHUNK_SIZE * 2);
         
         randomReadSize = (randomReadSize > inputData.available()) ?
               inputData.available() : randomReadSize;
         
         randomSizedArray = new byte[randomReadSize];
         inputData.read(randomSizedArray);
         
         encodedData = encoder.update(randomSizedArray);
         
         assertEquals(encodedData.size(), outputData.size());
         
         // Store any output from the update operation in outputData
         for (int idx = 0; idx < encodedData.size(); idx++)
         {
            outputData.get(idx).write(encodedData.get(idx));
         }
      }
      encodedData = encoder.finish(new byte[0]);
      
      // Store any output from the update operation in outputData
      for (int idx = 0; idx < encodedData.size(); idx++)
      {
         outputData.get(idx).write(encodedData.get(idx));
      }
      
      // Attempt a decode
      List<byte[]> dataToDecode = new ArrayList<byte[]>();      
      int faultTolerance = encoder.getNumSlices() - encoder.getThreshold();
      for (int idx = faultTolerance; idx < encoder.getNumSlices(); idx++)
      {
         dataToDecode.add(outputData.get(idx).toByteArray());
      }
      
      // Decode the data
      byte decodedData[] = decoder.finish(dataToDecode);

      assertEquals(data.length, decodedData.length);

      // Compare the decoded data to the original
      for (int idx = 0; idx < data.length; idx++)
      {
         assertEquals(data[idx], decodedData[idx]);
      }

   }
   */
   
   
   private void executeEncodeDecode(
         byte data[],
         InformationDispersalEncoder encoder,
         InformationDispersalDecoder decoder) throws Exception
   {
      // Encode the message
      List<byte[]> encodedData = encoder.finish(data);

      // For each supported outage case
      CombinationGenerator combinationGenerator =
            new CombinationGenerator(encoder.getNumSlices(), encoder.getThreshold());

      while (combinationGenerator.hasMore())
      {
         // Build a list of available shares
         List<byte[]> input = new ArrayList<byte[]>();

         int availableShares[] = combinationGenerator.getNext();

         for (int idx = 0; idx < availableShares.length; idx++)
         {
            input.add(encodedData.get(availableShares[idx]));
         }

         // Decode the data
         byte decodedData[] = decoder.finish(input);

         assertEquals(data.length, decodedData.length);

         // Compare the decoded data to the original
         for (int idx = 0; idx < data.length; idx++)
         {
            assertEquals(data[idx], decodedData[idx]);
         }
      }
   }
   
   private void executeEncodeDecodeOnce(
         byte data[],
         InformationDispersalEncoder encoder,
         InformationDispersalDecoder decoder,
         int expectedSize) throws Exception
   {
      // Encode the message
      List<byte[]> encodedData = encoder.finish(data);
      
      int actualSize = 0;
      for (int itr = 0; itr < encodedData.size(); itr++)
      {
         actualSize += encodedData.get(itr).length;
      }
      
      assertEquals(expectedSize, actualSize);

      // Build a list of available shares
      List<byte[]> input = new ArrayList<byte[]>();


      int faultTolerance = encoder.getNumSlices() - encoder.getThreshold();
      for (int idx = faultTolerance; idx < encoder.getNumSlices(); idx++)
      {
         input.add(encodedData.get(idx));
      }

      // Decode the data
      byte decodedData[] = decoder.finish(input);

      assertEquals(data.length, decodedData.length);

      // Compare the decoded data to the original
      for (int idx = 0; idx < data.length; idx++)
      {
         assertEquals(data[idx], decodedData[idx]);
      }
      
   }

   private void randomizeBuffer(byte buffer[])
   {
      for (int idx = 0; idx < buffer.length; idx++)
      {
         byte value = (byte) (Math.random() * 256);
         buffer[idx] = value;
      }
   }
   
   private int randRange(int min, int max)
   {
      return min + (int)(Math.random() * (max - min));
   }

}
