//
// Cleversafe open-source code header - Version 1.1 - December 1, 2006
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2007 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 10 W. 35th Street, 16th Floor #84,
// Chicago IL 60616
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// Author: ivolvovski
//
// Date: Jun 18, 2007
//---------------------

package org.cleversafe.config.evaluator;

import static org.junit.Assert.*;

import java.util.Arrays;
import java.util.UUID;

import org.cleversafe.config.evaluator.BooleanLiteral;
import org.cleversafe.config.evaluator.ByteLiteral;
import org.cleversafe.config.evaluator.DoubleLiteral;
import org.cleversafe.config.evaluator.Evaluatable;
import org.cleversafe.config.evaluator.FloatLiteral;
import org.cleversafe.config.evaluator.InstanceMethodEvaluator;
import org.cleversafe.config.evaluator.IntegerLiteral;
import org.cleversafe.config.evaluator.LiteralEvaluator;
import org.cleversafe.config.evaluator.LiteralEvaluatorFactory;
import org.cleversafe.config.evaluator.LongLiteral;
import org.cleversafe.config.evaluator.NamedEvaluator;
import org.cleversafe.config.evaluator.ReferenceEvaluator;
import org.cleversafe.config.evaluator.ShortLiteral;
import org.cleversafe.config.evaluator.StaticMethodEvaluator;
import org.cleversafe.config.evaluator.StringLiteral;
import org.cleversafe.config.evaluator.ReferenceEvaluator.GridLoadInitMethod;
import org.cleversafe.config.exceptions.ConfigurationException;
import org.junit.Test;


// TODO: Describe class or interface
public class EvaluatorTest
{
   @Test
   public void factoryTestPositive()
   {

      try
      {
         assertTrue(LiteralEvaluatorFactory.hasEvaluator("int"));
         assertTrue(LiteralEvaluatorFactory.hasEvaluator("double"));
         assertTrue(LiteralEvaluatorFactory.hasEvaluator("short"));
         assertTrue(LiteralEvaluatorFactory.hasEvaluator("byte"));
         assertTrue(LiteralEvaluatorFactory.hasEvaluator("long"));

         LiteralEvaluator ev = new StringLiteral("10");
         assertEquals(ev.evaluate().getClass(), String.class);
         assertEquals(ev.evaluate(), "10");
         ev = LiteralEvaluatorFactory.getEvaluator("string", "11");
         assertEquals(ev.evaluate().getClass(), String.class);
         assertEquals(ev.evaluate(), "11");

         ev = new IntegerLiteral(5);
         assertEquals(ev.evaluate().getClass(), Integer.class);
         assertEquals(ev.evaluate(), 5);
         ev = new IntegerLiteral("10");
         assertEquals(ev.evaluate().getClass(), Integer.class);
         assertEquals(ev.evaluate(), 10);
         ev = LiteralEvaluatorFactory.getEvaluator("int", "11");
         assertEquals(ev.evaluate().getClass(), Integer.class);
         assertEquals(ev.evaluate(), 11);

         ev = new LongLiteral(9999999999L);
         assertEquals(ev.evaluate().getClass(), Long.class);
         assertEquals(ev.evaluate(), 9999999999L);
         ev = new LongLiteral("1000000000000000000");
         assertEquals(ev.evaluate().getClass(), Long.class);
         assertEquals(ev.evaluate(), 1000000000000000000L);
         ev = LiteralEvaluatorFactory.getEvaluator("long", "11");
         assertEquals(ev.evaluate().getClass(), Long.class);
         assertEquals(ev.evaluate(), 11L);

         ev = new ShortLiteral((short) 777);
         assertEquals(ev.evaluate().getClass(), Short.class);
         assertEquals(ev.evaluate(), (short) 777);

         ev = new ShortLiteral("999");
         assertEquals(ev.evaluate().getClass(), Short.class);
         assertEquals(ev.evaluate(), (short) 999);
         ev = LiteralEvaluatorFactory.getEvaluator("short", "1000");
         assertEquals(ev.evaluate().getClass(), Short.class);
         assertEquals(ev.evaluate(), (short) 1000);

         ev = new ByteLiteral((byte) 125);
         assertEquals(ev.evaluate().getClass(), Byte.class);
         assertEquals(ev.evaluate(), (byte) 125);
         ev = new ByteLiteral("127");
         assertEquals(ev.evaluate().getClass(), Byte.class);
         assertEquals(ev.evaluate(), (byte) 127);
         ev = LiteralEvaluatorFactory.getEvaluator("byte", "10");
         assertEquals(ev.evaluate().getClass(), Byte.class);
         assertEquals(ev.evaluate(), (byte) 10);

         ev = new FloatLiteral(1.1f);
         assertEquals(ev.evaluate().getClass(), Float.class);
         float v0 = ((Float) ev.evaluate()).floatValue();
         assertTrue(Math.abs(v0 - (float) 1.1) < 0.00000001);
         ev = new FloatLiteral("127.11");
         assertEquals(ev.evaluate().getClass(), Float.class);
         v0 = ((Float) ev.evaluate()).floatValue();
         assertTrue(Math.abs(v0 - (float) 127.11) < 0.00000001);
         ev = LiteralEvaluatorFactory.getEvaluator("float", "2.11111E5");
         v0 = ((Float) ev.evaluate()).floatValue();
         assertEquals(ev.evaluate().getClass(), Float.class);
         assertTrue((Math.abs(v0 - (float) 211111)) < 0.00000001);

         ev = new DoubleLiteral(127.1111111111111111111);
         assertEquals(ev.evaluate().getClass(), Double.class);
         double v1 = ((Double) ev.evaluate()).doubleValue();
         assertTrue(Math.abs(v1 - (double) 127.1111111111111111111) < 0.00000001);
         ev = new DoubleLiteral("127.11");
         assertEquals(ev.evaluate().getClass(), Double.class);
         v1 = ((Double) ev.evaluate()).doubleValue();
         assertTrue(Math.abs(v1 - (double) 127.11) < 0.00000001);
         ev = LiteralEvaluatorFactory.getEvaluator("double", "2.11111E5");
         v1 = ((Double) ev.evaluate()).doubleValue();
         assertEquals(ev.evaluate().getClass(), Double.class);
         assertTrue((Math.abs(v1 - (double) 211111)) < 0.00000001);

         ev = new BooleanLiteral(false);
         assertEquals(ev.evaluate().getClass(), Boolean.class);
         assertEquals(ev.evaluate(), false);
         ev = new BooleanLiteral("true");
         assertEquals(ev.evaluate().getClass(), Boolean.class);
         assertEquals(ev.evaluate(), true);
         ev = LiteralEvaluatorFactory.getEvaluator("boolean", "false");
         assertEquals(ev.evaluate().getClass(), Boolean.class);
         assertEquals(ev.evaluate(), false);
      }
      catch (Exception e)
      {
         e.printStackTrace();
         fail();
      }
   }

   @Test(expected = RuntimeException.class)
   public void factoryTestNegative2() throws Exception
   {
      LiteralEvaluatorFactory.getEvaluator("Integer", "12");
   }

   @Test
   public void factoryTestNegative()
   {
      try
      {
         new IntegerLiteral("10.a");
         fail("Read int: 10");
      }
      catch (Exception ex)
      {
      }
      try
      {
         new ByteLiteral("260");
         fail("Read byte: 260");
      }
      catch (Exception ex)
      {
      }
   }

   @Test(expected = RuntimeException.class)
   public void factoryTestNegative1()
   {
      new IntegerLiteral("10.a");
   }

   @Test
   public void instanceEvaluatorTest()
   {
      try
      {
         UUID uuid = UUID.fromString("773259c0-dbb4-4f87-8367-f5c7d6c4394b");
         UUID uuid1 = UUID.fromString("773259c0-dbb4-4f87-ffff-f5c7d6c4394b");

         InstanceMethodEvaluator inst = new InstanceMethodEvaluator(uuid,
               "compareTo", Arrays.asList(new Evaluatable[]{
                  new ReferenceEvaluator(uuid1)
               }));
         Object res = inst.evaluate();
         assertEquals(res.getClass(), Integer.class);
         assertNotSame(res, 0);

         int cmp = ((Integer) (res)).intValue();
         assertTrue(Math.abs(cmp) == 1);

         inst = new InstanceMethodEvaluator(uuid, "getLeastSignificantBits",
               Arrays.asList(new Evaluatable[]{}));
         res = inst.evaluate();
         assertEquals(res.getClass(), Long.class);
         assertEquals(res, uuid.getLeastSignificantBits());
      }
      catch (Exception ex)
      {
         ex.printStackTrace();
         fail();
      }
   }

   @Test
   public void staticEvaluatorTest()
   {
      try
      {
         UUID uuid = UUID.fromString("773259c0-dbb4-4f87-8367-f5c7d6c4394b");

         StaticMethodEvaluator stat = new StaticMethodEvaluator(UUID.class,
               "fromString", Arrays.asList(new Evaluatable[]{
                  new StringLiteral("773259c0-dbb4-4f87-8367-f5c7d6c4394b")
               }));

         Object res = stat.evaluate();
         assertEquals(res.getClass(), UUID.class);
         assertTrue(uuid.compareTo((UUID) res) == 0);
      }
      catch (Exception ex)
      {
         ex.printStackTrace();
         fail();
      }
   }

   @Test(expected = ConfigurationException.class)
   public void negativeStaticEvaluatorTest1() throws ConfigurationException
   {
      StaticMethodEvaluator stat = new StaticMethodEvaluator(UUID.class,
            "fromString", Arrays.asList(new Evaluatable[]{
               new IntegerLiteral(99)
            }));
      stat.evaluate();
   }

   @Test
   public void complexEvaluatorTest()
   {
      try
      {
         UUID uuid = UUID.fromString("773259c0-dbb4-4f87-8367-f5c7d6c4394b");
         long mostSigBits = uuid.getMostSignificantBits();

         StaticMethodEvaluator stat = new StaticMethodEvaluator(UUID.class,
               "fromString", Arrays.asList(new Evaluatable[]{
                  new StringLiteral("773259c0-dbb4-4f87-8367-f5c7d6c4394b")
               }));
         InstanceMethodEvaluator inst = new InstanceMethodEvaluator(stat
               .evaluate(), "getMostSignificantBits", Arrays
               .asList(new Evaluatable[]{}));

         Object res = inst.evaluate();
         assertEquals(res.getClass(), Long.class);
         long msb = ((Long) (res)).longValue();
         assertTrue(mostSigBits == msb);
      }
      catch (Exception ex)
      {
         ex.printStackTrace();
         fail();
      }
   }

   @Test
   public void conversionPrimitiveEvaluatorTest()
   {
      try
      {
         String sv = new String("123456789");

         InstanceMethodEvaluator inst = new InstanceMethodEvaluator(sv,
               "charAt", Arrays.asList(new Evaluatable[]{
                  new IntegerLiteral(1)
               }));

         Object res = inst.evaluate();
         assertEquals(res.getClass(), Character.class);
         assertEquals(res, '2');

         inst = new InstanceMethodEvaluator(sv, "charAt", Arrays
               .asList(new Evaluatable[]{
                  new ShortLiteral((short) 2)
               }));

         res = inst.evaluate();
         assertEquals(res.getClass(), Character.class);
         assertEquals(res, '3');
      }
      catch (Exception ex)
      {
         ex.printStackTrace();
         fail();
      }
   }

   @Test
   public void conversionEvaluatorTest()
   {
      try
      {
         String sv = new String("A");

         InstanceMethodEvaluator inst = new InstanceMethodEvaluator(sv,
               "equals", Arrays.asList(new Evaluatable[]{
                  new StringLiteral("A")
               }));

         Object res = inst.evaluate();
         assertEquals(res.getClass(), Boolean.class);
         assertEquals(res, true);

         inst = new InstanceMethodEvaluator(sv, "equals", Arrays
               .asList(new Evaluatable[]{
                  new ReferenceEvaluator(new Object())
               }));

         res = inst.evaluate();
         assertEquals(res.getClass(), Boolean.class);
         assertEquals(res, false);

         sv = new String("0123456");

         inst = new InstanceMethodEvaluator(sv, "substring", Arrays
               .asList(new Evaluatable[]{
                     new IntegerLiteral(2), new IntegerLiteral(5)
               }));
         res = inst.evaluate();
         assertEquals(res.getClass(), String.class);
         assertEquals(res, "234");

      }
      catch (Exception ex)
      {
         ex.printStackTrace();
         fail();
      }
   }

   @Test
   public void namedEvaluatorTest()
   {
      try
      {
         String name = "my parameter";
         int value = 5;
         LiteralEvaluator le = new IntegerLiteral(value);
         NamedEvaluator ne = new NamedEvaluator(name, le);

         assertEquals(ne.evaluate(), Integer.valueOf(value));
         assertEquals(ne.getName(), name);
      }
      catch (Exception e)
      {
         e.printStackTrace();
         fail();
      }
   }

   // Class with parameters, just for test...
   public static class ReferenceTest
   {
      private int counter = 0;
      private String helloString = "";

      public ReferenceTest()
      {

      }

      @GridLoadInitMethod()
      public void increment()
      {
         counter++;
      }

      public void setHelloString(String helloString)
      {
         this.helloString = helloString;
      }

      public void setCounter(int counter)
      {
         this.counter = counter;
      }

      public String getHelloString()
      {
         return this.helloString;
      }

      public int getCounter()
      {
         return this.counter;
      }
   }

   @Test
   public void referenceEvaluatorTest()
   {
      ReferenceTest testObject = new ReferenceTest();
      int counter = 45;
      String helloString = "hello references";
      NamedEvaluator counterParam = new NamedEvaluator("counter",
            new IntegerLiteral(counter));
      NamedEvaluator helloStringParam = new NamedEvaluator("hello-string",
            new StringLiteral(helloString));

      ReferenceEvaluator re = new ReferenceEvaluator(testObject, Arrays
            .asList(new NamedEvaluator[]{
                  counterParam, helloStringParam
            }));

      try
      {
         Object evaluatedObject = re.evaluate();
         assert evaluatedObject instanceof ReferenceTest;

         ReferenceTest evaluatedReferenceTest = (ReferenceTest) evaluatedObject;

         // counter + 1 (increment is called as initialize method)
         assertEquals(evaluatedReferenceTest.getCounter(), counter + 1);
         assertEquals(evaluatedReferenceTest.getHelloString(), helloString);
      }
      catch (Exception e)
      {
         e.printStackTrace();
         fail();
      }
   }
}
