//
// Cleversafe open-source code header - Version 1.1 - December 1, 2006
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2007 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 10 W. 35th Street, 16th Floor #84,
// Chicago IL 60616
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// Author: mmotwani
//
// Date: May 16, 2007
//---------------------

package org.cleversafe.config;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import org.cleversafe.config.exceptions.ConfigurationException;
import org.cleversafe.test.BaseTest;
import org.junit.Test;

/**
 * Tests XMLPropertiesProvider implementation of the PropertiesProvider interface. Reads properties
 * from an XML file and tests against hardcoded values.
 * 
 * @author Manish Motwani
 */
public class XMLPropertiesProviderTest
{
   private static PropertiesProvider properties = null;

   private static final String XMLFileName =
         System.getProperty(BaseTest.TEST_INPUT_PROPERTY, ".")
               + "/org/cleversafe/config/propertiesTest.xml";

   // ///////////////////////////////////////////////////////////////////////////
   // Constant Values from Test XML
   private final String[] stringListValues = {
         "this is great", "this is a list of greatness", "oh wow this is even greater!"
   };
   private final String stringValue = "this is just a single string";

   private final int[] intListValues = {
         1325, 4441, 21111
   };
   private final int intValue = 1883;

   private final boolean[] booleanListValues = {
         true, true, false, true, false, true
   };
   private final boolean booleanValue = true;

   private final double[] doubleListValues = {
         13.25, 44.41, 2111.1
   };
   private final double doubleValue = 1.883;

   // ///////////////////////////////////////////////////////////////////////////

   // Constructor
   public XMLPropertiesProviderTest()
   {
      if (properties == null)
      {
         try
         {
            System.setProperty(ConfigurationFactory.XML_PROPERTIES_CONFIG_PROPERTY, XMLFileName);
            properties =
                  ConfigurationFactory.getPropertiesProvider(ConfigurationFactory.XML_CONFIG_TYPE);

            assertTrue(properties instanceof XMLPropertiesProvider);

            System.clearProperty(ConfigurationFactory.XML_PROPERTIES_CONFIG_PROPERTY);
         }
         catch (Exception e)
         {
            e.printStackTrace();
         }
      }
   }

   @Test
   public void testGetStringValue()
   {
      // Should pass this test and show a value.
      try
      {
         String[] testArray = new String[]{
               "org", "cleversafe", "string", "single", "funtest"
         };
         String value = properties.getStringValue(testArray);
         assertEquals(value, this.stringValue);

         System.out.println("getStringValue test1 passed, value: " + value);
      }
      catch (ConfigurationException ce)
      {
         fail(ce.getMessage());
      }

      // Should pass this test and show a value.
      try
      {
         String[] testArray = new String[]{
               "org", "cleversafe", "string", "single", "funtest"
         };
         String value = properties.getValue(testArray);
         assertEquals(value, this.stringValue);

         System.out.println("getStringValue test1.5 passed, value: " + value);
      }
      catch (ConfigurationException ce)
      {
         fail(ce.getMessage());
      }

      // Should fail and catch an exception
      try
      {
         String[] testArray = new String[]{
               "some", "random", "array", "to", "fail"
         };
         properties.getStringValue(testArray);
         fail("Should not have reached here!");
      }
      catch (ConfigurationException ce)
      {
         System.out.println("getStringValue test2 passed, exception caught: " + ce.getMessage());
      }

      // Should fail and catch an exception
      try
      {
         String[] testArray = new String[]{
               "org", "cleversafe", "string", "list", "funtest"
         };
         properties.getStringValue(testArray);
         fail("Should not have reached here!");
      }
      catch (ConfigurationException ce)
      {
         System.out.println("getStringValue test3 passed, exception caught: " + ce.getMessage());
      }
   }

   @Test
   public void testGetStringListValues()
   {
      // Should pass this test and show the values.
      try
      {
         String[] testArray = new String[]{
               "org", "cleversafe", "string", "list", "funtest"
         };
         List<String> values = properties.getStringValuesList(testArray);

         assertEquals(values, Arrays.asList(this.stringListValues));

         System.out.println("getStringListValues test1 passed, values:" + values);
      }
      catch (ConfigurationException ce)
      {
         fail(ce.getMessage());
      }

      // Should pass this test and show the values.
      try
      {
         String[] testArray = new String[]{
               "org", "cleversafe", "string", "list", "funtest"
         };
         List<String> values = properties.getValuesList(testArray);

         assertEquals(values, Arrays.asList(this.stringListValues));

         System.out.println("getStringListValues test1.5 passed, values:" + values);
      }
      catch (ConfigurationException ce)
      {
         fail(ce.getMessage());
      }

      // Should fail and catch an exception
      try
      {
         String[] testArray = new String[]{
               "some", "random", "array", "to", "fail"
         };
         properties.getStringValuesList(testArray);
         fail("Should not have reached here!");
      }
      catch (ConfigurationException ce)
      {
         System.out.println("getStringListValues test2 passed, exception caught: "
               + ce.getMessage());
      }
   }

   @Test
   public void testSetDefaultStringValue()
   {
      try
      {
         String[] testArray = new String[]{
               "setting", "some", "default", "single", "string", "value"
         };
         String value = "my new default value";
         properties.setDefaultStringValue(testArray, value);

         String returnedValue = properties.getStringValue(testArray);

         assertEquals(returnedValue, value);
      }
      catch (ConfigurationException ce)
      {
         fail(ce.getMessage());
      }
   }

   @Test
   public void testSetDefaultStringListValues()
   {
      try
      {
         String[] testArray = new String[]{
               "setting", "some", "default", "list", "string", "values"
         };
         String[] values =
               {
                     "my new default values list item 1", "my new default values list item 2",
                     "my new default values list item 3"
               };
         properties.setDefaultStringValuesList(testArray, values);

         List<String> returnedValues = properties.getStringValuesList(testArray);

         assertEquals(Arrays.asList(values), returnedValues);
      }
      catch (ConfigurationException ce)
      {
         fail(ce.getMessage());
      }
   }

   @Test
   public void testNumStringValuesSet()
   {
      try
      {
         String[] testArray = new String[]{
               "setting", "some", "default", "list", "string", "values", "for", "numTest"
         };

         int emptyResult = properties.getNumStringValuesSet(testArray);

         assertTrue(emptyResult == 0);

         String[] values =
               {
                     "my new default values list item 1 for numTest",
                     "my new default values list item 2 for numTest",
                     "my new default values list item 3 for numTest"
               };
         properties.setDefaultStringValuesList(testArray, values);

         int newResult = properties.getNumStringValuesSet(testArray);

         assertTrue(newResult == values.length);
      }
      catch (Exception e)
      {
         fail(e.getMessage());
      }
   }

   @Test
   public void testGetIntValue()
   {
      // Should pass this test and show a value.
      try
      {
         String[] testArray = new String[]{
               "org", "cleversafe", "int", "single", "funtest"
         };
         int value = properties.getIntValue(testArray);
         assertEquals(value, this.intValue);

         System.out.println("getIntValue test1 passed, value: " + value);
      }
      catch (ConfigurationException ce)
      {
         fail(ce.getMessage());
      }

      // Should pass this test and show a value.
      try
      {
         String[] testArray = new String[]{
               "org", "cleversafe", "int", "single", "funtest"
         };
         String value = properties.getValue(testArray);
         assertEquals(value, String.valueOf(this.intValue));

         System.out.println("getIntValue test1.5 passed, value: " + value);
      }
      catch (ConfigurationException ce)
      {
         fail(ce.getMessage());
      }

      // Should fail and catch an exception
      try
      {
         String[] testArray = new String[]{
               "some", "random", "array", "to", "fail"
         };
         properties.getIntValue(testArray);
         fail("Should not have reached here!");
      }
      catch (ConfigurationException ce)
      {
         System.out.println("getIntValue test2 passed, exception caught: " + ce.getMessage());
      }

      // Should fail and catch an exception
      try
      {
         String[] testArray = new String[]{
               "org", "cleversafe", "int", "list", "funtest"
         };
         properties.getIntValue(testArray);
         fail("Should not have reached here!");
      }
      catch (ConfigurationException ce)
      {
         System.out.println("getIntValue test3 passed, exception caught: " + ce.getMessage());
      }
   }

   @Test
   public void testGetIntListValues()
   {
      // Should pass this test and show the values.
      try
      {
         String[] testArray = new String[]{
               "org", "cleversafe", "int", "list", "funtest"
         };
         List<Integer> values = properties.getIntValuesList(testArray);

         List<Integer> expectedValues = new ArrayList<Integer>();

         for (int intVal : this.intListValues)
         {
            expectedValues.add(new Integer(intVal));
         }

         assertEquals(values, expectedValues);

         System.out.println("getIntListValues test1 passed, values:" + values);
      }
      catch (ConfigurationException ce)
      {
         fail(ce.getMessage());
      }

      // Should pass this test and show the values.
      try
      {
         String[] testArray = new String[]{
               "org", "cleversafe", "int", "list", "funtest"
         };
         List<String> values = properties.getValuesList(testArray);

         List<String> expectedValues = new ArrayList<String>();

         for (int intVal : this.intListValues)
         {
            expectedValues.add(String.valueOf(intVal));
         }

         assertEquals(values, expectedValues);

         System.out.println("getIntListValues test1.5 passed, values:" + values);
      }
      catch (ConfigurationException ce)
      {
         fail(ce.getMessage());
      }

      // Should fail and catch an exception
      try
      {
         String[] testArray = new String[]{
               "some", "random", "array", "to", "fail"
         };
         properties.getIntValuesList(testArray);
         fail("Should not have reached here!");
      }
      catch (ConfigurationException ce)
      {
         System.out.println("getStringListValues test2 passed, exception caught: "
               + ce.getMessage());
      }
   }

   @Test
   public void testSetDefaultIntValue()
   {
      try
      {
         String[] testArray = new String[]{
               "setting", "some", "default", "single", "int", "value"
         };
         int value = 1134;
         properties.setDefaultIntValue(testArray, value);

         int returnedValue = properties.getIntValue(testArray);

         assertTrue(returnedValue == value);
      }
      catch (ConfigurationException ce)
      {
         fail(ce.getMessage());
      }
   }

   @Test
   public void testSetDefaultIntListValues()
   {
      try
      {
         String[] testArray = new String[]{
               "setting", "some", "default", "list", "int", "values"
         };
         int[] values = {
               441, 556, 655
         };
         properties.setDefaultIntValuesList(testArray, values);

         List<Integer> returnedValues = properties.getIntValuesList(testArray);

         List<Integer> expectedValues = new ArrayList<Integer>();

         for (int intVal : values)
         {
            expectedValues.add(new Integer(intVal));
         }

         assertEquals(returnedValues, expectedValues);
      }
      catch (ConfigurationException ce)
      {
         fail(ce.getMessage());
      }
   }

   @Test
   public void testNumIntValuesSet()
   {
      try
      {
         String[] testArray = new String[]{
               "setting", "some", "default", "list", "int", "values", "for", "numTest"
         };

         int emptyResult = properties.getNumIntValuesSet(testArray);

         assertTrue(emptyResult == 0);

         int[] values = {
               1234, 5678, 9101112
         };
         properties.setDefaultIntValuesList(testArray, values);

         int newResult = properties.getNumIntValuesSet(testArray);

         assertTrue(newResult == values.length);
      }
      catch (Exception e)
      {
         fail(e.getMessage());
      }
   }

   @Test
   public void testGetBooleanValue()
   {
      // Should pass this test and show a value.
      try
      {
         String[] testArray = new String[]{
               "org", "cleversafe", "boolean", "single", "funtest"
         };
         boolean value = properties.getBooleanValue(testArray);
         assertEquals(value, this.booleanValue);

         System.out.println("getBooleanValue test1 passed, value: " + Boolean.toString(value));
      }
      catch (ConfigurationException ce)
      {
         fail(ce.getMessage());
      }

      // Should pass this test and show a value.
      try
      {
         String[] testArray = new String[]{
               "org", "cleversafe", "boolean", "single", "funtest"
         };
         String value = properties.getValue(testArray);
         assertEquals(value, String.valueOf(this.booleanValue));

         System.out.println("getBooleanValue test1.5 passed, value: " + value);
      }
      catch (ConfigurationException ce)
      {
         fail(ce.getMessage());
      }

      // Should fail and catch an exception
      try
      {
         String[] testArray = new String[]{
               "some", "random", "array", "to", "fail"
         };
         properties.getBooleanValue(testArray);
         fail("Should not have reached here!");
      }
      catch (ConfigurationException ce)
      {
         System.out.println("getBooleanValue test2 passed, exception caught: " + ce.getMessage());
      }

      // Should fail and catch an exception
      try
      {
         String[] testArray = new String[]{
               "org", "cleversafe", "boolean", "list", "funtest"
         };
         properties.getBooleanValue(testArray);
         fail("Should not have reached here!");
      }
      catch (ConfigurationException ce)
      {
         System.out.println("getBooleanValue test3 passed, exception caught: " + ce.getMessage());
      }
   }

   @Test
   public void testGetBooleanListValues()
   {
      // Should pass this test and show the values.
      try
      {
         String[] testArray = new String[]{
               "org", "cleversafe", "boolean", "list", "funtest"
         };
         List<Boolean> values = properties.getBooleanValuesList(testArray);

         List<Boolean> expectedValues = new ArrayList<Boolean>();

         for (boolean intVal : this.booleanListValues)
         {
            expectedValues.add(new Boolean(intVal));
         }

         assertEquals(values, expectedValues);

         System.out.println("getBooleanListValues test1 passed, values:" + values);
      }
      catch (ConfigurationException ce)
      {
         fail(ce.getMessage());
      }

      // Should pass this test and show the values.
      try
      {
         String[] testArray = new String[]{
               "org", "cleversafe", "boolean", "list", "funtest"
         };
         List<String> values = properties.getValuesList(testArray);

         List<String> expectedValues = new ArrayList<String>();

         for (boolean intVal : this.booleanListValues)
         {
            expectedValues.add(String.valueOf(intVal));
         }

         assertEquals(values, expectedValues);

         System.out.println("getBooleanListValues test1.5 passed, values:" + values);
      }
      catch (ConfigurationException ce)
      {
         fail(ce.getMessage());
      }

      // Should fail and catch an exception
      try
      {
         String[] testArray = new String[]{
               "some", "random", "array", "to", "fail"
         };
         properties.getBooleanValuesList(testArray);
         fail("Should not have reached here!");
      }
      catch (ConfigurationException ce)
      {
         System.out.println("getBooleanListValues test2 passed, exception caught: "
               + ce.getMessage());
      }
   }

   @Test
   public void testSetDefaultBooleanValue()
   {
      try
      {
         String[] testArray = new String[]{
               "setting", "some", "default", "single", "boolean", "value"
         };
         boolean value = true;
         properties.setDefaultBooleanValue(testArray, value);

         boolean returnedValue = properties.getBooleanValue(testArray);

         assertTrue(returnedValue == value);
      }
      catch (ConfigurationException ce)
      {
         fail(ce.getMessage());
      }
   }

   @Test
   public void testSetDefaultBooleanListValues()
   {
      try
      {
         String[] testArray = new String[]{
               "setting", "some", "default", "list", "boolean", "values"
         };
         boolean[] values = {
               false, true, true, false, true
         };
         properties.setDefaultBooleanValuesList(testArray, values);

         List<Boolean> returnedValues = properties.getBooleanValuesList(testArray);

         List<Boolean> expectedValues = new ArrayList<Boolean>();

         for (boolean intVal : values)
         {
            expectedValues.add(new Boolean(intVal));
         }

         assertEquals(returnedValues, expectedValues);
      }
      catch (ConfigurationException ce)
      {
         fail(ce.getMessage());
      }
   }

   @Test
   public void testNumBooleanValuesSet()
   {
      try
      {
         String[] testArray = new String[]{
               "setting", "some", "default", "list", "boolean", "values", "for", "numTest"
         };

         int emptyResult = properties.getNumBooleanValuesSet(testArray);

         assertTrue(emptyResult == 0);

         boolean[] values = {
               true, true, false, true, false, true, true, true
         };
         properties.setDefaultBooleanValuesList(testArray, values);

         int newResult = properties.getNumBooleanValuesSet(testArray);

         assertTrue(newResult == values.length);
      }
      catch (Exception e)
      {
         fail(e.getMessage());
      }
   }

   @Test
   public void testGetDoubleValue()
   {
      // Should pass this test and show a value.
      try
      {
         String[] testArray = new String[]{
               "org", "cleversafe", "double", "single", "funtest"
         };
         double value = properties.getDoubleValue(testArray);
         assertEquals(value, this.doubleValue);

         System.out.println("getDoubleValue test1 passed, value: " + value);
      }
      catch (ConfigurationException ce)
      {
         fail(ce.getMessage());
      }

      // Should pass this test and show a value.
      try
      {
         String[] testArray = new String[]{
               "org", "cleversafe", "double", "single", "funtest"
         };
         String value = properties.getValue(testArray);
         assertEquals(value, String.valueOf(this.doubleValue));

         System.out.println("getDoubleValue test1.5 passed, value: " + value);
      }
      catch (ConfigurationException ce)
      {
         fail(ce.getMessage());
      }

      // Should fail and catch an exception
      try
      {
         String[] testArray = new String[]{
               "some", "random", "array", "to", "fail"
         };
         properties.getDoubleValue(testArray);
         fail("Should not have reached here!");
      }
      catch (ConfigurationException ce)
      {
         System.out.println("getDoubleValue test2 passed, exception caught: " + ce.getMessage());
      }

      // Should fail and catch an exception
      try
      {
         String[] testArray = new String[]{
               "org", "cleversafe", "double", "list", "funtest"
         };
         properties.getDoubleValue(testArray);
         fail("Should not have reached here!");
      }
      catch (ConfigurationException ce)
      {
         System.out.println("getDoubleValue test3 passed, exception caught: " + ce.getMessage());
      }
   }

   @Test
   public void testGetDoubleListValues()
   {
      // Should pass this test and show the values.
      try
      {
         String[] testArray = new String[]{
               "org", "cleversafe", "double", "list", "funtest"
         };
         List<Double> values = properties.getDoubleValuesList(testArray);

         List<Double> expectedValues = new ArrayList<Double>();

         for (double intVal : this.doubleListValues)
         {
            expectedValues.add(new Double(intVal));
         }

         assertEquals(values, expectedValues);

         System.out.println("getDoubleListValues test1 passed, values:" + values);
      }
      catch (ConfigurationException ce)
      {
         fail(ce.getMessage());
      }

      // Should pass this test and show the values.
      try
      {
         String[] testArray = new String[]{
               "org", "cleversafe", "double", "list", "funtest"
         };
         List<String> values = properties.getValuesList(testArray);

         List<String> expectedValues = new ArrayList<String>();

         for (double intVal : this.doubleListValues)
         {
            expectedValues.add(String.valueOf(intVal));
         }

         assertEquals(values, expectedValues);

         System.out.println("getDoubleListValues test1.5 passed, values:" + values);
      }
      catch (ConfigurationException ce)
      {
         fail(ce.getMessage());
      }

      // Should fail and catch an exception
      try
      {
         String[] testArray = new String[]{
               "some", "random", "array", "to", "fail"
         };
         properties.getDoubleValuesList(testArray);
         fail("Should not have reached here!");
      }
      catch (ConfigurationException ce)
      {
         System.out.println("getDoubleListValues test2 passed, exception caught: "
               + ce.getMessage());
      }
   }

   @Test
   public void testSetDefaultDoubleValue()
   {
      try
      {
         String[] testArray = new String[]{
               "setting", "some", "default", "single", "double", "value"
         };
         double value = 11.34;
         properties.setDefaultDoubleValue(testArray, value);

         double returnedValue = properties.getDoubleValue(testArray);

         assertTrue(returnedValue == value);
      }
      catch (ConfigurationException ce)
      {
         fail(ce.getMessage());
      }
   }

   @Test
   public void testSetDefaultDoubleListValues()
   {
      try
      {
         String[] testArray = new String[]{
               "setting", "some", "default", "list", "double", "values"
         };
         double[] values = {
               4.41, 556.0, 0.655
         };
         properties.setDefaultDoubleValuesList(testArray, values);

         List<Double> returnedValues = properties.getDoubleValuesList(testArray);

         List<Double> expectedValues = new ArrayList<Double>();

         for (double intVal : values)
         {
            expectedValues.add(new Double(intVal));
         }

         assertEquals(returnedValues, expectedValues);
      }
      catch (ConfigurationException ce)
      {
         fail(ce.getMessage());
      }
   }

   @Test
   public void testNumDoubleValuesSet()
   {
      try
      {
         String[] testArray = new String[]{
               "setting", "some", "default", "list", "double", "values", "for", "numTest"
         };

         int emptyResult = properties.getNumDoubleValuesSet(testArray);

         assertTrue(emptyResult == 0);

         double[] values = {
               12.34, 5678.0, 9101.112, 4444.991
         };
         properties.setDefaultDoubleValuesList(testArray, values);

         int newResult = properties.getNumDoubleValuesSet(testArray);

         assertTrue(newResult == values.length);
      }
      catch (Exception e)
      {
         fail(e.getMessage());
      }
   }

}
