//
// Cleversafe open-source code header - Version 1.1 - December 1, 2006
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2007 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 10 W. 35th Street, 16th Floor #84,
// Chicago IL 60616
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// Author: Zach
//
// Date: Jun 20, 2007
//---------------------

package org.cleversafe.authentication;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.security.KeyPair;
import java.security.KeyPairGenerator;
import java.security.SignatureException;
import java.security.cert.Certificate;
import java.security.cert.CertificateFactory;
import java.util.UUID;

import org.apache.log4j.Logger;
import org.apache.log4j.xml.DOMConfigurator;
import org.cleversafe.TestDataLiterals;
import org.cleversafe.authentication.credentials.PasswordCredentials;
import org.cleversafe.authentication.exceptions.CredentialsException;
import org.cleversafe.test.BaseTest;
import org.cleversafe.test.TestException;
import org.cleversafe.util.UUIDGen;
import org.junit.BeforeClass;
import org.junit.Test;

/**
 * Test of the properties file credentials manager implementation. Requires a file (defined in the
 * PropertiesFileCredentialsManager implementation) to be present in the current user's home
 * directory to operate or have the path overridden in the configuration framework according to the
 * resource org.cleversafe.authentication.credentialspath
 * 
 * @author Zachary Mark
 */
public class PropertiesFileCredentialsManagerTest extends BaseTest
{
   private static Logger _logger = Logger.getLogger(PropertiesFileCredentialsManagerTest.class);

   public static final String TEST_USERNAME = "testuser";
   public static final String TEST_PASSWORD = "testpass";
   public static final UUID TEST_ACCOUNT_UUID = UUID.nameUUIDFromBytes(TEST_USERNAME.getBytes());
   
   
   public static final String FIXED_CREDENTIALS_FILE_NAME =
         TestDataLiterals.FIXED_CREDENTIALS_FILE_NAME;
   public static final String FIXED_KEYSTORE_FILE_NAME = TestDataLiterals.FIXED_KEYSTORE_FILE_NAME;

   private static final String TEST_CREDENTIALS_FILE_NAME = "test.credentials";
   private static final String TEST_KEYSTORE_FILE_NAME = "test.jks";

   private static final String TEST_KEY_ALGORITHM = "RSA";
   private static final int TEST_KEY_SIZE = 1024;
   private static final String TEST_SIGNATURE_ALGORITHM = "SHA1withRSA";

   
   public File getInputDirectory(String testName) throws TestException
   {
      return new File(super.getInputDirectory() + File.separator + testName);
   }

   public File getOutputDirectory(String testName) throws TestException
   {
      return new File(super.getOutputDirectory() + File.separator + testName);
   }
   

   @BeforeClass
   public static void setUpBeforeClass() throws Exception
   {
      DOMConfigurator.configure(System.getProperty("log4j.configuration"));
   }

   @Test
   public void testGenerateCertificate() throws Exception
   {
      KeyPairGenerator keygen = KeyPairGenerator.getInstance("RSA");
      keygen.initialize(1024);
      KeyPair keypair = keygen.generateKeyPair();
      String account = UUID.randomUUID().toString();
      Certificate cert =
            PropertiesFileCredentialsManager.generateCertificate(account, keypair, "MD5withRSA");

      File file =
            new File(System.getProperty(BaseTest.TEST_INPUT_PROPERTY, ".")
                  + "/org/cleversafe/authentication/certificate.crt");

      // Write generated certificate to file.
      FileOutputStream out = new FileOutputStream(file);
      out.write(cert.getEncoded());
      out.close();

      // Read back certificate to make sure it is valid
      FileInputStream in = new FileInputStream(file);
      CertificateFactory cf = CertificateFactory.getInstance("X.509");
      Certificate recovered = cf.generateCertificate(in);
      in.close();

      assertEquals("written certficiate file to equal to generated cert", cert, recovered);
      try
      {
         recovered.verify(keypair.getPublic());
      }
      catch (SignatureException e)
      {
         fail("written certificate does not have proper signature");
      }

      file.delete();

   }

   @Test
   public void testUsernamePasswordLoadDefault() throws Exception
   {
      System.clearProperty(PropertiesFileCredentialsManager.SYSTEM_PROPERTY_CREDENTIALS_PATH);
      System.clearProperty(PropertiesFileCredentialsManager.SYSTEM_PROPERTY_KEYSTORE_PATH);
      String orig = PropertiesFileCredentialsManager.getDefaultCredentialsPath();
      PropertiesFileCredentialsManager.setDefaultCredentialsPath( this.getOutputDirectory().getAbsolutePath() );
      createFileFile(null, null);
      usernamePasswordLoad( new PropertiesFileCredentialsManager() );
      PropertiesFileCredentialsManager.setDefaultCredentialsPath( orig );
   }

   @Test
   public void testUsernamePasswordLoadProperties() throws Exception
   {
      System.setProperty(
            PropertiesFileCredentialsManager.SYSTEM_PROPERTY_CREDENTIALS_PATH,
            this.getOutputDirectory() + File.separator + TEST_ACCOUNT_UUID + ".credentials" );
      System.setProperty(
            PropertiesFileCredentialsManager.SYSTEM_PROPERTY_KEYSTORE_PATH,
            this.getOutputDirectory() + File.separator + TEST_ACCOUNT_UUID + ".jks" );
      String orig = PropertiesFileCredentialsManager.getDefaultCredentialsPath();
      PropertiesFileCredentialsManager.setDefaultCredentialsPath(
            this.getOutputDirectory() + File.separator + "nogo" );
      usernamePasswordLoad( new PropertiesFileCredentialsManager() );
      PropertiesFileCredentialsManager.setDefaultCredentialsPath( orig );
   }

   @Test
   public void testUsernamePasswordLoadOverride() throws Exception
   {
      System.setProperty(
            PropertiesFileCredentialsManager.SYSTEM_PROPERTY_CREDENTIALS_PATH,
            this.getOutputDirectory() + File.separator + FIXED_CREDENTIALS_FILE_NAME + ".bad" );
      System.setProperty(
            PropertiesFileCredentialsManager.SYSTEM_PROPERTY_KEYSTORE_PATH,
            this.getOutputDirectory() + File.separator + FIXED_KEYSTORE_FILE_NAME + ".bad" );
      String orig = PropertiesFileCredentialsManager.getDefaultCredentialsPath();
      PropertiesFileCredentialsManager.setDefaultCredentialsPath(
            this.getOutputDirectory() + File.separator + "nogo" );
      
      PropertiesFileCredentialsManager manager = new PropertiesFileCredentialsManager(
            new File(this.getOutputDirectory() + File.separator + TEST_ACCOUNT_UUID + ".credentials"),
            new File(this.getOutputDirectory() + File.separator + TEST_ACCOUNT_UUID + ".jks") );
      usernamePasswordLoad( manager );
      
      PropertiesFileCredentialsManager.setDefaultCredentialsPath( orig );
   }

   private void usernamePasswordLoad(CredentialsManager credentialsManager)
   {
      _logger.debug("Getting credentials from credentialsManager");

      Credentials credentials = null;
      try
      {
         credentials = credentialsManager.getCredentials(TEST_ACCOUNT_UUID);
      }
      catch (CredentialsException e)
      {
         fail(e.getMessage());
      }

      assertTrue("Retrieved credentials not of type PasswordCredentials",
            credentials instanceof PasswordCredentials);

      PasswordCredentials cred = (PasswordCredentials) credentials;

      try
      {
         // Since this is system dependent, just make sure no exceptions are thrown.
         credentialsManager.getAccountKeyPair(TEST_ACCOUNT_UUID);
      }
      catch (CredentialsException e)
      {
         fail(e.getMessage());
      }

      assertEquals("Usernames not equal", TEST_USERNAME, cred.getUsername());
      assertEquals("Passwords not equal", TEST_PASSWORD, cred.getPassword());
   }

   private void deleteTestFiles() throws TestException
   {
      File credentialsFile = new File( this.getOutputDirectory() + File.separator + TEST_CREDENTIALS_FILE_NAME );
      File keystoreFile = new File( this.getOutputDirectory() + File.separator + TEST_KEYSTORE_FILE_NAME );
      
      credentialsFile.delete();
      keystoreFile.delete();
   }

   //@Test
   public void testCreateFileFileOverride() throws Exception
   {
      deleteTestFiles();
      
      File credentialsFile = new File( this.getOutputDirectory() + File.separator + TEST_CREDENTIALS_FILE_NAME );
      File keystoreFile = new File( this.getOutputDirectory() + File.separator + TEST_KEYSTORE_FILE_NAME );
      
      // Testing create(file, file, ...) with a specific file overrides
      createFileFile(credentialsFile, keystoreFile);

   }

   @Test
   public void testCreateFileFileDefault() throws Exception
   {
      new File( this.getOutputDirectory() + File.separator + 
            UUIDGen.getUUIDFromUsername(TEST_USERNAME) +
            PropertiesFileCredentialsManager.CREDENTIALS_EXTENSION ).delete();
      new File( this.getOutputDirectory() + File.separator + 
            UUIDGen.getUUIDFromUsername(TEST_USERNAME) +
            PropertiesFileCredentialsManager.KEYSTORE_EXTENSION ).delete();
      
      
      // Setup system properties and default credentials path
      System.clearProperty(PropertiesFileCredentialsManager.SYSTEM_PROPERTY_CREDENTIALS_PATH);
      System.clearProperty(PropertiesFileCredentialsManager.SYSTEM_PROPERTY_KEYSTORE_PATH);
      String orig = PropertiesFileCredentialsManager.getDefaultCredentialsPath();
      PropertiesFileCredentialsManager.setDefaultCredentialsPath( this.getOutputDirectory().getAbsolutePath() );
      
      // Testing create(file, file, ...) with null entries
      createFileFile(null, null);

      // Clear default credentials path and remove generated files
      PropertiesFileCredentialsManager.setDefaultCredentialsPath( orig );

   }

   //@Test
   public void testCreateFileFileProperties() throws Exception
   {
      deleteTestFiles();
      
      _logger.debug("properties credentials path: " +
            this.getOutputDirectory() + File.separator + TEST_CREDENTIALS_FILE_NAME);
      _logger.debug("properties keystore path: " +
            this.getOutputDirectory() + File.separator + TEST_KEYSTORE_FILE_NAME);
      _logger.debug("default credentials path: " + 
            this.getOutputDirectory() + File.separator + "nogo");
      
      // Setup system properties and default credentials path
      System.setProperty(
            PropertiesFileCredentialsManager.SYSTEM_PROPERTY_CREDENTIALS_PATH,
            this.getOutputDirectory() + File.separator + TEST_CREDENTIALS_FILE_NAME );
      System.setProperty(
            PropertiesFileCredentialsManager.SYSTEM_PROPERTY_KEYSTORE_PATH,
            this.getOutputDirectory() + File.separator + TEST_KEYSTORE_FILE_NAME );
      String orig = PropertiesFileCredentialsManager.getDefaultCredentialsPath();
      PropertiesFileCredentialsManager.setDefaultCredentialsPath(
            this.getOutputDirectory() + File.separator + "nogo" );
      
      // Testing create(file, file, ...) with null entries
      createFileFile(null, null);

      // Clear default credentials path and remove generated files
      PropertiesFileCredentialsManager.setDefaultCredentialsPath(orig);
   }
   
   public static PropertiesFileCredentialsManager createFileFile(
         File credentialsFile,
         File keystoreFile,
         String username,
         String password) throws Exception
   {
      return PropertiesFileCredentialsManager.create(
            credentialsFile,
            keystoreFile,
            username,
            password,
            TEST_KEY_ALGORITHM,
            TEST_KEY_SIZE,
            TEST_SIGNATURE_ALGORITHM );
   }
   
   /**
    * Tests credentials creation and loading.
    * @param credentialsFile Uses system property if null.
    * @param keystoreFile Uses system property if null.
    * @throws Exception
    */
   private static void createFileFile(File credentialsFile, File keystoreFile) throws Exception
   {  
      PropertiesFileCredentialsManager manager =
         createFileFile(credentialsFile, keystoreFile, TEST_USERNAME, TEST_PASSWORD);
      
      Credentials credentials = manager.getCredentials( TEST_ACCOUNT_UUID );
      
      assertTrue("Incorrect credential class", credentials instanceof PasswordCredentials);
      assertEquals("Username does not match", TEST_USERNAME,
            ((PasswordCredentials) credentials).getUsername());
      assertEquals("Password does not match", TEST_PASSWORD,
            ((PasswordCredentials) credentials).getPassword());
      assertEquals("Grid account ID does not match", TEST_ACCOUNT_UUID,
            credentials.getAccountIdentifier());

      KeyPair keypair = manager.getAccountKeyPair(TEST_ACCOUNT_UUID);
      assertEquals("Private algorithm does not match", TEST_KEY_ALGORITHM,
            keypair.getPrivate().getAlgorithm());
      assertEquals("Public algorithm does not match", TEST_KEY_ALGORITHM,
            keypair.getPublic().getAlgorithm());

      // Reload the manager with the same files

      PropertiesFileCredentialsManager newManager =
            new PropertiesFileCredentialsManager(credentialsFile, keystoreFile);

      Credentials newCredentials = newManager.getCredentials(TEST_ACCOUNT_UUID);
      assertTrue("Incorrect credential class", newCredentials instanceof PasswordCredentials);
      assertEquals("Username does not match", TEST_USERNAME,
            ((PasswordCredentials) newCredentials).getUsername());
      assertEquals("Password does not match", TEST_PASSWORD,
            ((PasswordCredentials) newCredentials).getPassword());
      assertEquals("Grid account ID does not match", TEST_ACCOUNT_UUID,
            newCredentials.getAccountIdentifier());

      KeyPair newKeypair = manager.getAccountKeyPair(TEST_ACCOUNT_UUID);
      assertEquals("Private algorithm does not match", TEST_KEY_ALGORITHM,
            newKeypair.getPrivate().getAlgorithm());
      assertEquals("Public algorithm does not match", TEST_KEY_ALGORITHM,
            newKeypair.getPublic().getAlgorithm());

      // Check the two key pairs for equality
      assertEquals("Private keys do not match", keypair.getPrivate(), newKeypair.getPrivate());
      assertEquals("Public keys do not match", keypair.getPublic(), newKeypair.getPublic());
   }

}
