package org.cleversafe;

import java.io.BufferedReader;
import java.io.FileReader;
import java.lang.reflect.Method;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.List;
import java.util.Random;


public class CDBSpecParser {
	
	private static String inputFile = "/cdb.in";
	
	private class TagValue
	{
		private int bits;
		private String tag;
		private String value; // template uses string value
		private long intValue;  // instance uses long (intValue)
		
		public TagValue(TagValue tv)
		{
			bits = tv.bits;
			tag = tv.tag;
			value = tv.value;
			intValue = tv.intValue;
		}

		public TagValue(int b, String t, String v)
		{
			bits = b;
			tag = t;
			value = v;
		}
		public TagValue(int b, String t, long v)
		{
			bits = b;
			tag = t;
			intValue = v;
		}
		int getBits()
		{
			return bits;
		}
		String getTag()
		{
			return tag;
		}
		String getValue()
		{
			return value;
		}
		long getIntValue()
		{
			return intValue;
		}
		
	
		public String toString()
		{
			if (value != null)
				return (" | " + tag + " | " +  bits + " | " + value );
			else
				return (" | " + tag + " | " +  bits + " | " + Long.toHexString(intValue).toUpperCase() );
		}
	}
	
	private class CDB
	{
		private String name;
		private List<TagValue> tagList = new ArrayList<TagValue>();
		
		private List<CDB> instanceList = null;
		
		private int size = 0; // number of bits
		private byte[] dataBytes;

		public CDB(String n)
		{
			name = n;
			size = 0;
			dataBytes = null;
		}
		
		public CDB(CDB cdb)
		{
			this.name = cdb.name;
			this.size = cdb.size;
			for (TagValue tv : cdb.tagList) {
				this.tagList.add(new TagValue(tv));
			}
		}
		
		public int getSize()
		{
			return size;
		}
		
		public void setSize(int n)
		{
			size = n;
		}
		
		public byte [] getDataBytes()
		{
			return dataBytes;
		}
		
		public void setDataBytes( byte[] data)
		{
			dataBytes = data;
		}
	
		public void addInstance(CDB cdb)
		{
			if (instanceList == null) {
				instanceList = new ArrayList<CDB>();
			}
			instanceList.add(cdb);
		}
		public String getName()
		{
			return name;
		}

		public String toString()
		{
			String result = ("< " + name + "=" + size);
			for (TagValue tv : tagList)
			{
				result += tv.toString();
			}
			if (instanceList != null) {
				result += "( inst=" + instanceList.size() + " )"; 
			}
			result += ">";
			return result;
		}
		
		public void printInstances()
		{
			for (CDB one: instanceList) {
				System.out.println(one.toString());
			}
		}
		public void addTag(int bits, String tag, String val)
		{
			tagList.add(new TagValue(bits, tag, val));
			size += bits;
		}
		public void addTag(int bits, String tag, long val)
		{
			tagList.add(new TagValue(bits, tag, val));
			size += bits;
		}
		
		public void reduceBits(int b)
		{
			size -= b;
		}
		
		public List<byte[]> deepCopy(List<byte[]> srcList)
		{
			List<byte[]> dest = new ArrayList<byte[]>();
			for (byte[] ba : srcList)
			{
				byte[] newBa =  new byte[ba.length];
				for (int i=0; i< ba.length; i++) newBa[i] = ba[i];
				dest.add(newBa);
			}

			return dest;
		}
		
		public void generateInstances()
		{
			// add first instance.
			this.addInstance(new CDB(this.getName()));
			
			for (TagValue tv : this.tagList) {
				List<Long> valueList = convertValue(tv.getBits(), tv.getTag(), tv.getValue());
				List<CDB> newList = new ArrayList<CDB>();
				for (long val : valueList) {
					// add 'val' for each instance
					for(CDB inst : this.instanceList) {
						CDB instCopy = new CDB (inst);
						instCopy.addTag(tv.getBits(), tv.getTag(), val);
						newList.add(instCopy);
					}
				}
				// set the new List as instanceList.
				this.instanceList = newList;
			}
		}
		/**
		 * Generates List if byteArrays, each element is an array bytes that is filled with
		 * but pattern generated from the specification.
		 * 
		 * Assumes that parserCDBspec() is called first to parse the specifications.
		 * 
		 * @return
		 */
		
		public byte [] generateBytes()
		{
			int sizeBytes =  (int)Math.ceil(size * 1.0 / 8);
			long value = 0;
			// System.out.println("NumBits=" + size + ", numBytes=" + sizeBytes);
			byte [] result = new byte[ sizeBytes ];
			
			int currentByteIndx = 0;
			int currentBit = 7;
			byte currentByte = 0;
			for (int i=0; i< tagList.size(); i++)
			{
				TagValue tv = tagList.get(i);
				int bits = tv.getBits();
				value = tv.getIntValue();

				// copy bits from value into currentByte
				boolean empty = true;
				while (bits > 0) {
					empty = false;
					long mask = 1 << (bits-1);
					if ((mask & value)!=0) {
						currentByte = (byte)(currentByte | ( 1  << currentBit));
					}
					// value = value >>> 1;
					bits--;
					currentBit--;
					if (currentBit < 0) {
						// currentByte is filled, move to next byte
						result [currentByteIndx] = currentByte;
						empty = true;
						currentByteIndx++;
						currentBit = 7;
						currentByte = 0;
					}
				}
				if (! empty) {
					// write the last byte whatever it has, may be partially filled.
					result [currentByteIndx] = currentByte;
				}
			}
			this.setDataBytes(result);
			return result;
		}
		
		
		private List<Long> convertValue(int numBits, String tag, String valString)
		{
			List <Long> valList = new ArrayList<Long>();
			long localVal = 0;
			if (valString.startsWith("std")) {
				// add the beginning boundary condition.
				valList.add(0L);
				localVal = 1;
				for(int i=1; i < numBits; i++){
					localVal = localVal << 1;
					localVal |=  1;
					if(i % 8 == 0) {
						valList.add(localVal);
					}
				}
				valList.add((long) Math.pow(2,numBits)-1);
			} else if (valString.startsWith("random")) {
				int numRandom = 1; // default generate just 1.
				// generate random value of size bits;
				if (System.getProperty("test.randomSize." + getName() + "." + tag) != null) {
					numRandom = Integer.getInteger("test.randomSize." + getName()+ "." + tag);
				} else if (System.getProperty("test.randomSize." + getName()) != null) {
					numRandom = Integer.getInteger("test.randomSize." + getName());
				} else if (System.getProperty("test.randomSize") != null) {
					// Random property is set. Use it.
					numRandom = Integer.getInteger("test.randomSize");
				}

				if (valString.equals("random"))
					valList = generateRandomBits(numBits, numRandom, false);
				else
					valList = generateRandomBits(numBits, numRandom, true);
			} else if (valString.substring(0,1).equals("[") ){
				// we have a list specification. Parse all the values.
				String stripBraces = valString.substring(1, valString.length()-1);
				String valStrings[] = stripBraces.split(";");
				for (String one : valStrings) {
					String baseStr = one.substring(1, 2);
					String oneVal=null;
					int base = 10;
					if (baseStr.equals("x")) {
						base = 16;
						oneVal = one.substring(2);
					} else if (baseStr.equals("b")){
						base = 2;
						oneVal = one.substring(2);
					} else if (baseStr.equals("o")) {
						base = 8;
						oneVal = one.substring(2);
					}
					localVal = Long.parseLong(oneVal, base);
					valList.add(localVal);
				}
			} else {
				String baseStr = valString.substring(1, 2);
				int base = 10;
				if (baseStr.equals("x")) {
					base = 16;
					valString = valString.substring(2);
				} else if (baseStr.equals("b")){
					base = 2;
					valString = valString.substring(2);
				} else if (baseStr.equals("o")) {
					base = 8;
					valString = valString.substring(2);
				}
				localVal = Long.parseLong(valString, base);
				valList.add(localVal);
			}
			
			return valList;
		}
		
		public List<Long> generateRandomBits(int numBits, int numRandom, boolean assci)
		{
			List<Long> randomList = new ArrayList<Long>();
			
			long bitMask = 0;
			for (int i=0; i < numBits; i++) {
				bitMask = (bitMask << 1) | 1;
			}
			Random rnb = new Random(12345);
			
			int numGenerated = 1;
			while (numGenerated <= numRandom)
			{
				long ran = rnb.nextLong() & bitMask;
				if (assci) {
					// printable ASCII values fall [33-126] range (except whitespace).
					if ((numGenerated < 33) || (numGenerated > 126))
						continue;
				}
				if (uniqueNumber(randomList, ran)) {
					randomList.add( ran );
					numGenerated++;
				}
			}
			return randomList;
		}
		
		private boolean uniqueNumber(List<Long> list, long ran)
		{
			for(long num : list) 
			{
				if (num == ran) {
					return false;
				}
			}
			return true;
		}
	}
	
	private static List<CDB> parsedCDBs = new ArrayList<CDB>();
		
	public static List<CDB> getParsedCDBs()
	{
		return parsedCDBs;
	}
		
	private void parseOneSpec(String line)
	{
		if (line.equals("")){
			return;
		}
		// check for comment lines.
		String firstChar = line.substring(0,1);
		if (firstChar.equals("#")) {
			// ignore the line if it starts with a '#'.
			return;
		}
		String[] results = line.split(",");
		
		
		CDB cdb = new CDB(results[0]);
	
		int bits = 0;
		for (int i = 1; i < results.length; i++)
		{
			String[] elements = results[i].split("=");
			
			String tag = elements[0];
			String[] values =  elements[1].split(":");
			bits = Integer.parseInt(values[0]);
			String val = values[1];
			cdb.addTag(bits, tag, val);
		}
	
		if ((cdb.getSize() %8) != 0) {
			System.out.println("Not at byte boundry: " + cdb.getSize());
		} else {
			parsedCDBs.add(cdb);
		}
	}
	
	public void generateAllInstances()
	{
		for(CDB cdb : CDBSpecParser.parsedCDBs )
		{
			cdb.generateInstances();
		}
	}
	
	/**
	 * @param fileName with CDB specifications.
	 */
	private void parseCDBspecs(String path)
	{
	
		String line;
		int numLines=0;
		// parse all the lines one by one.
		try {
	        BufferedReader in = new BufferedReader(new FileReader(path));
	        while ((line = in.readLine()) != null) {
	        	numLines++;
	    		parseOneSpec(line);
	        }
	        in.close();
	    } catch (Exception e) {
	    	e.printStackTrace();
	    	return;
	    }
	    System.out.println("========================");
	    System.out.println("PARSED CDB SPEC FOLLOWS");
	    System.out.println("========================");
  
	    printParsedCDBs();
	}
	
	public void printParsedCDBs()
	{
		for (CDB cdb : parsedCDBs)
		{
			System.out.println(cdb.toString());
		}
	}
	
	public void printAllInstances()
	{
		for(CDB cdb:parsedCDBs) {
			cdb.printInstances();
		}
	}
	
	public void generateByteData()
	{
		for(CDB cdb : CDBSpecParser.parsedCDBs )
		{
			for (CDB inst : cdb.instanceList) {
				inst.generateBytes();
			}
		}
	}
	
	/**
	 * @param args
	 */
	public static void main(String[] args) {
		CDBSpecParser parser = new CDBSpecParser();

		String inputPath = System.getProperty("test.input", inputFile);
		parser.parseCDBspecs(inputPath);
		// parser.printParsedCDBs();
		parser.generateAllInstances();
		parser.printAllInstances();
		parser.generateByteData();

		// print the generated data
		for(CDB cdb : CDBSpecParser.parsedCDBs )
		{
			System.out.println(cdb.getName() + " instances:");
			for (CDB inst : cdb.instanceList) {
				byte [] data = inst.getDataBytes();
				System.out.println();
				parser.executeTest(inst,  data);
				
				System.out.println();
			}
		}		
	}
	
	public void executeTest(CDB cdb, byte[] data)
	{
		System.out.println(cdb.getName());

		for (byte b : data) {
			System.out.format("%02X ", b);
		}
		System.out.println();

		String className = cdb.getName();
		Class<?> classObj = null;
		final String packagePrefix = "org.jscsi.scsi.protocol.cdb";
		try {
			classObj = Class.forName(packagePrefix + "." + className);
		} catch (Exception ex) {
			ex.printStackTrace();
		}

		// Instantiate the object.
		Object obj = null;
		try {
			obj = classObj.newInstance();
		}catch (Exception ex) {
			ex.printStackTrace();
		}
		// Find 'encode()' method.
		//		Method[] allMethods = classObj.getDeclaredMethods();
		Method decode = null;
		try {
			Method[] allMethods = classObj.getDeclaredMethods();
		    for (Method m : allMethods) {
		    	if (m.getName().equals("decode")) {
		    		decode = m;
		    		break;
		    	}
			}
		} catch (Exception ex) {
			ex.printStackTrace();
	    	return;
		}

		// run the parser with data as input.
	    try {
	    	decode.invoke(obj, ByteBuffer.wrap(data));
	    } catch (Exception ex) {
	    	ex.printStackTrace();
	    	return;
	    }
		// verify the correctness by checking each tag and its value.
	    try {
	    	for (TagValue tv : cdb.tagList)
	    	{
	    		if (tv.getTag().equals("reserved"))
	    			continue;
	    		// find access method and compare the value.
	    		String prefix = null;
	    		if (tv.getBits() > 1) {
	    			// the field is more than 1 bit, hence the type of the field is a Number.
	    			prefix = "get";
		    		Method accessMethod = classObj.getMethod(prefix + tv.getTag());
		    		Object valObj = accessMethod.invoke(obj);
	    			Number val = (Number) valObj;
	    			Long valInput = (Long) tv.getIntValue(); 
		    		if (accessMethod.getReturnType().equals("long"))
		    		{
		    			// return value of the field is long
		    			if (valInput.longValue() == val.longValue())
		    			{
		    				System.out.format("Success: %s : %X == %X %n", tv.getTag(), val, tv.getIntValue());
		    			} 
		    			else 
		    			{
		    				System.out.format("Fail: %s : %X != %X %n", val, tv.getTag(), tv.getIntValue());
		    			}
		    		} else {
		    			// return value of the field is int
		    			if (val.intValue() == valInput.intValue())
		    			{
		    				System.out.format("Success: %s : %X == %X %n", tv.getTag(), val, tv.getIntValue());
		    			} 
		    			else 
		    			{
		    				System.out.format("Fail: %s : %X != %X %n", tv.getTag(), val, tv.getIntValue());
		    			}
		    		}
	    		} else {
	    			// the width of the field is one bit, hence the type is boolean.
	    			prefix = "is";
		    		Method accessMethod = classObj.getMethod(prefix + tv.getTag());
		    		Object valObj = accessMethod.invoke(obj);
	    			Boolean val = (Boolean) valObj;
	    			Long valInput = (Long) tv.getIntValue();
	    			if ( ((valInput == 1) && (val)) || ((valInput == 0) && !val))
	    			{
	    				System.out.format("Success: %s : %b == %X %n", tv.getTag(), val, tv.getIntValue());
	    			}
	    			else {
	    				System.out.format("Fail: %s : %b != %X %n", tv.getTag(), val, tv.getIntValue());
	    			}
	    		}
	    	}
	    } catch (Exception ex)
	    {
	    	ex.printStackTrace();
	    	return;
	    }
		
	}
}
