//
// Cleversafe open-source code header - Version 1.1 - December 1, 2006
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2007 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 10 W. 35th Street, 16th Floor #84,
// Chicago IL 60616
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// @author: mmotwani
//
// Date: Aug 16, 2007
//---------------------

package org.cleversafe.serialization;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.DataInputStream;
import java.io.DataOutputStream;

import org.cleversafe.layer.grid.DataSlice;
import org.cleversafe.layer.grid.SliceName;
import org.cleversafe.layer.protocol.BeginTransactionRequest;
import org.cleversafe.layer.protocol.BeginTransactionResponse;
import org.cleversafe.layer.protocol.CommitTransactionRequest;
import org.cleversafe.layer.protocol.CommitTransactionResponse;
import org.cleversafe.layer.protocol.MultipleReadRequest;
import org.cleversafe.layer.protocol.MultipleReadResponse;
import org.cleversafe.layer.protocol.MultipleWriteRequest;
import org.cleversafe.layer.protocol.MultipleWriteResponse;
import org.cleversafe.layer.protocol.ProtocolMessage;
import org.cleversafe.layer.protocol.exceptions.ProtocolSerializationException;
import org.cleversafe.serialization.asn1.ASN1GridProtocolMessageFactory;
import org.junit.AfterClass;
import org.junit.BeforeClass;
import org.junit.Test;

// TODO: Describe class or interface
public class ProtocolMessageFactoryPerformanceTest
{
   private static ProtocolMessageFactory pmFactory = null;

   private static SliceName[] sliceNames = null;
   private static DataSlice[] dataSlices = null;
   private static long txid = 1020221182;
   private static int numIterations = 10000;
   private static int numDataSlices = 32;
   private static int numSliceNames = 32;
   private static int SLICE_BLOCK_SIZE = 353;

   private static int GRID_WIDTH = 16;

   private static int BLOCK_SIZE = 4096;

   /*
   private static String SET_SLICE_NAMES_METHOD = "setSliceNames";
   private static String SET_DATA_SLICES_METHOD = "setDataSlices";
   private static String SET_TRANSACTION_ID_METHOD = "setTransactionID";
   */

   private Class<?>[] frequentProtocolMessages =
         new Class<?>[]{
               MultipleReadRequest.class, MultipleReadResponse.class, MultipleWriteRequest.class,
               MultipleWriteResponse.class, BeginTransactionRequest.class,
               BeginTransactionResponse.class, CommitTransactionRequest.class,
               CommitTransactionResponse.class
         };

   @BeforeClass
   public static void setup()
   {
      numIterations = Integer.getInteger("test.iteration", numIterations);
      numSliceNames = numDataSlices = Integer.getInteger("test.numSlices", numSliceNames);

      pmFactory = new ASN1GridProtocolMessageFactory();
      sliceNames = ProtocolMessageFactoryTestBase.getRandomSliceNames(numSliceNames);
      dataSlices =
            ProtocolMessageFactoryTestBase.getRandomDataSliceArray(numDataSlices, SLICE_BLOCK_SIZE);
   }

   @AfterClass
   public static void tearDown()
   {
      pmFactory = null;
   }

   private void testProtocolMessage(ProtocolMessage pm, int numIterations, boolean configure)
         throws Exception
   {
      String className = pm.getClass().getName();
      int indNameNoPack = className.lastIndexOf('.');
      String classNameNoPack = className.substring(indNameNoPack + 1);
      testProtocolMessageSGroup(new ProtocolMessage[]{
         pm
      }, classNameNoPack, numIterations, configure);
   }

   byte[] serialize(ProtocolMessage message) throws ProtocolSerializationException
   {
      ByteArrayOutputStream bytes = new ByteArrayOutputStream();
      DataOutputStream out = new DataOutputStream(bytes);
      pmFactory.serialize(out, message);
      return bytes.toByteArray();
   }

   private double testProtocolMessageSGroup(
         ProtocolMessage[] pms,
         String description,
         int numIterations,
         boolean configure) throws Exception
   {
      long totalSize = 0;
      long startTime = System.currentTimeMillis();
      for (int i = 0; i < numIterations; ++i)
      {
         if (configure)
         {
            for (int j = 0; j < pms.length; j++)
            {
               configureProtocolMessage(pms[j]);
            }
         }
         for (int j = 0; j < pms.length; j++)
         {
            byte[] serialMessage = serialize(pms[j]);
            int code = pms[j].getOperation().getOperationCode();
            pmFactory.deserialize(code,
                  new DataInputStream(new ByteArrayInputStream(serialMessage)));
            totalSize += serialMessage.length;
         }
      }
      double totalTime = System.currentTimeMillis() - startTime;
      float averageTime = ((float) totalTime) / ((float) numIterations);
      System.out.println(description + ": \t\t" + averageTime + "ms (" + (totalSize / pms.length)
            / numIterations + ")");
      return averageTime;
   }

   // don't use reflection here - it is much slower than direct lookup
   private void configureProtocolMessage(ProtocolMessage pm)
   {
      if (pm instanceof MultipleReadRequest)
      {
         ((MultipleReadRequest) pm).setSliceNames(sliceNames);
      }
      else if (pm instanceof MultipleReadResponse)
      {
         ((MultipleReadResponse) pm).setDataSlices(dataSlices);
      }
      else if (pm instanceof MultipleWriteRequest)
      {
         ((MultipleWriteRequest) pm).setDataSlices(dataSlices);
      }
      else if (pm instanceof BeginTransactionRequest)
      {
         ((BeginTransactionRequest) pm).setTransactionID(txid);
      }
      else if (pm instanceof CommitTransactionRequest)
      {
         ((CommitTransactionRequest) pm).setTransactionID(txid);
      }
      /*
       * Method setSliceBuffers = pm.getClass().getMethod(SET_DATA_SLICES_METHOD,
       * DataSlice[].class); _logger.info("Attempting to set slice buffers for " + pm.getClass());
       * setSliceBuffers.invoke(pm, (Object) dataSlices);
       * 
       * Method setSourceNames = pm.getClass().getMethod(SET_SLICE_NAMES_METHOD, SliceName[].class);
       * _logger.info("Attempting to set source names for " + pm.getClass());
       * setSourceNames.invoke(pm, (Object) sliceNames);
       * 
       * Method setTransactionID = pm.getClass().getMethod(SET_TRANSACTION_ID_METHOD, long.class);
       * _logger.info("Attempting to set transaction id for " + pm.getClass());
       * setTransactionID.invoke(pm, txid);
       */
   }

   private void testProtocolMessageAllSingle(boolean configure) throws Exception
   {
      for (Class<?> protocolMessage : frequentProtocolMessages)
      {
         testProtocolMessage((ProtocolMessage) protocolMessage.newInstance(), numIterations,
               configure);
      }
   }

   private void testProtocolMessageGroup(boolean configure, String operation, Class<?>... inGroup)
         throws Exception
   {
      ProtocolMessage[] objectGroup = new ProtocolMessage[inGroup.length];

      for (int i = 0; i < inGroup.length; i++)
      {
         objectGroup[i] = (ProtocolMessage) inGroup[i].newInstance();
      }
      double average = testProtocolMessageSGroup(objectGroup, operation, numIterations, configure);
      // Calculate throughput for a block
      // (1MB/8)/32*4K - # of chunks for 1Mbit
      int bytesInMbit = 1024 * 1024 / 8;
      double sendsPerMbit = (double) bytesInMbit / (BLOCK_SIZE * numDataSlices);
      double conversionsForMbit = sendsPerMbit * GRID_WIDTH;
      double timePerMbit = conversionsForMbit * (average / 1000);
      System.out.println(String.format("Throughput: %.0f Mbps", 1 / timePerMbit));
   }

   @Test
   public void testProtocolAllSelected() throws Exception
   {
      System.out.println("Protocol messages performance ");
      testProtocolMessageAllSingle(true);
      System.out.println();
   }

   @Test
   public void testProtocolReadGroup() throws Exception
   {
      System.out.println("Protocol read group: ");
      testProtocolMessageGroup(true, "Read roundtrip", MultipleReadRequest.class,
            MultipleReadResponse.class);
      System.out.println();
   }

   @Test
   public void testProtocolWriteGroup() throws Exception
   {
      System.out.println("Protocol write group: ");
      testProtocolMessageGroup(true, "Write roundtrip", BeginTransactionRequest.class,
            BeginTransactionResponse.class, MultipleWriteRequest.class,
            MultipleWriteResponse.class, CommitTransactionRequest.class,
            CommitTransactionResponse.class);
      System.out.println();
   }
}
